/***********************************************************************
*           See Accompanying licence file QSAS_licence                 *
***********************************************************************/
/* With adjustments for _WIN32 and __VMS compilers -- Patrick Daly */

#define CDF 0
#define QFT 1
#define QFD 2
#define TRY 3
#define CEF 4
#define CEF2 5
#define CEF2_GZ 6
#define NO_OF 7


#include "qie.h"
int cefChecking;
int n_errs;
int n_warn;
 
int main( int argc, char *argv[] )
{

QiCDFContents *QiSio;
QiOptions *QiSoptR;
QiOptions *QiSoptW;

long check = QMW_OK;
char *file_name;
char *file_name_copy;
char * header = NULL;
char * out_name = NULL;
int in_type = TRY;
int out_type;
QiOptions_priority choice = WARN;
QiOptions_rec_num rec_numbering = NUM_OFF;
char *flag;
char * ptr;
char * recStart = NULL;
char * recEnd = NULL;
char dbg_level = 'n';
int force_eor_write = 0;
int secDecimals = -1;
int cdfFileBack=3;

cefChecking = 0;
n_errs = 0;
n_warn = 0;
 
 if(argc<2)
{
  printf("Usage: %s  [-i{tdcea}] [-o{tdceax}] [-hName] [-nName] [-f] [-r] [-v] [-c] [-b] [-sxxx] [-exxx] [-L] [-tn] FileName\n",argv[0]);
  printf("options are...\n");
  printf(" -i{tdcea} for input type qft, qfd, cdf, cef or cef2(CAA) respectively\n");
  printf(" -o{tdceax} to specify output type as  qft, qfd, cdf, cef, cef2(CAA), no file respectively\n");
  printf(" -f forces overwrite, otherwise error returned if file exists\n");
  printf(" -r turns on record numbering if output file is flat\n");
  printf(" -v turns on  verbose messages and prints version number\n");
  printf(" -b causes cdf v2.7 files to be created for backward compatibility. Ignored if output not cdf.\n");
  printf(" -c turns on  cef 2 syntax compliance messages\n");
  printf(" -hName specifies Name as that of a detached header\n");
  printf(" -nName specifies Name as that of the output file stem\n");
  printf(" -sxxx Records starting with xxx are to be read. First record number is 0. Ignored when reading CDF\n");
  printf(" -exxx Records up to and including xxx are to be read. First record number is 0. Ignored when reading CDF\n");
  printf(" -L turns on writing of end-of-record specifier in header even for default of newline\n");
  printf(" -tn sets the output accuracy for times to have n places after the decimal point (in seconds)\n");
  printf("     for n > 3 Epoch16 is used in cdf files\n");
  printf("\n");
  printf(" Option flags MUST be space separated,\n");
  printf(" File type option must immediately follow -i and -o flags\n");
  printf(" Option -h is ignored for cdf type input file\n");
  printf(" If the input file has no extension use -i flag to set input type\n");
  exit(1);
}

argc--;
file_name = QiNewStr(argv[argc]);

if(file_name[0] == '-'){
  if( strcmp(file_name, "-v") == 0) {
    printf("Qtran version %s\n", QIE_VERSION);
    exit(0);
  }
  else{
    printf("Option flags must precede input data file\n");
    exit(1);
  }
}

/* remove extension if present and set file type accordingly */

if((ptr=strstr(file_name,".cdf")) != NULL) {
  in_type = CDF;
  out_type = CEF2;
  ptr[0] = '\0'; /* remove it */
}
else if((ptr=strstr(file_name,".CDF")) != NULL) {
  in_type = CDF;
  out_type = CEF2;
  ptr[0] = '\0'; /* remove it */
}
else if((ptr=strstr(file_name,".qft")) != NULL) {
  in_type = QFT;
  out_type = CDF;
  ptr[0] = '\0'; /* remove it */
}
else if((ptr=strstr(file_name,".QFT")) != NULL) {
  in_type = QFT;
  out_type = CDF;
  ptr[0] = '\0'; /* remove it */
}
else if((ptr=strstr(file_name,".qfd")) != NULL) {
  in_type = QFD;
  out_type = CDF;
  ptr[0] = '\0'; /* remove it */
}
else if((ptr=strstr(file_name,".QFD")) != NULL) {
  in_type = QFD;
  out_type = CDF;
  ptr[0] = '\0'; /* remove it */
}
else if((ptr=strstr(file_name,".cef.gz")) != NULL) {
  in_type = CEF2_GZ; // set to CAA as only CAA can be read compressed
  out_type = CDF;
  ptr[0] = '\0'; /* remove it */
}
else if((ptr=strstr(file_name,".CEF")) != NULL) {
  in_type = CEF; // set to CEF as CAA can be autodetected
  out_type = CDF;
  ptr[0] = '\0'; /* remove it */
}
else if((ptr=strstr(file_name,".cef")) != NULL) {
  in_type = CEF; // set to CEF as CAA can be autodetected
  out_type = CDF;
  ptr[0] = '\0'; /* remove it */
}

/* if use flag to override file type, or failing that fall through to Flat TRY */

while(--argc > 0){
  flag = QiNewStr(argv[argc]);
  if(flag[0] != '-') fprintf(stderr,"unrecognised flag %s\n",flag);
  else{
    switch(flag[1]){
      case 'o':
        if      (flag[2] == 'c') out_type = CDF;
        else if (flag[2] == 't') out_type = QFT;
        else if (flag[2] == 'd') out_type = QFD;
        else if (flag[2] == 'e') out_type = CEF;
        else if (flag[2] == 'a') out_type = CEF2;
        else if (flag[2] == 'x') out_type = NO_OF;
        break;
        
      case 'i':
        if      (flag[2] == 'c') in_type = CDF;
        else if (flag[2] == 't') in_type = QFT;
        else if (flag[2] == 'd') in_type = QFD;
        else if (flag[2] == 'e') in_type = CEF;
        else if (flag[2] == 'a') in_type = CEF2;
        else if (flag[2] == 'z') in_type = CEF2_GZ;
        break; 

      case 'h':
        header = QiNewStr(&(flag[2]));           
        break; 

      case 'n':
        out_name = QiNewStr(&(flag[2]));           
        break; 
            
      case 's':
        recStart = QiNewStr(&(flag[2]));           
        break; 
            
      case 'e':
        recEnd = QiNewStr(&(flag[2]));           
        break; 
            
      case 'f':
        choice = REPLACE;
        break;
        
      case 'v':
        printf("Qtran version %s\n", QIE_VERSION);
        dbg_level = 'f';
        break;
        
      case 'c':
        cefChecking = 1;
        break;
        
      case 'b':
        cdfFileBack = 2;
        break;
        
      case 'r':
        rec_numbering = NUM_ON;
        break;
        
      case 't':
        secDecimals = atoi(&(flag[2]));
        break;
        
      case 'L':
        force_eor_write = 1;
        break;
        
      default:
       fprintf(stderr,"unrecognised flag %s\n",flag);
    }  
  }
  free(flag);
}


 /*  READ in file  */
 /*---------------------*/
 
 /* set up read options */
 
  QiSoptR = (QiOptions *) QiMakeOptionsObj();
  
  QiSoptR->fp_display = stderr;
  QiSoptR->debug_choice = dbg_level;     
  if(header == NULL) QiSoptR->header = ATTACHED;
  else {
    QiSoptR->header = DETACHED;
    QiSoptR->header_name = QiNewStr(header);
  }

  /* Initialise object to hold whole file */
 
  QiSio = (QiCDFContents *) QiMakeCDFContentsObj();  
  /* specify file name and dir */
  
  /* pass in time resolution if set */
  if(secDecimals != -1) {
    QiSio->numSecDecimal = secDecimals;
	QiSio->UserSetSecDecimal = 1;
  }
  
  file_name_copy = QiNewStr(&(file_name[0]));
#if defined _WIN32
  ptr = strrchr(file_name_copy, '\\');
#elif defined __VMS
  ptr = strrchr(file_name_copy, ']');
#else
  ptr = strrchr(file_name_copy, '/');
#endif
  if(ptr != NULL){
    QiSio->io_f_name = QiNewStr(&(ptr[1]));
    ptr[1] = '\0';
    QiSio->io_f_path = QiNewStr(file_name_copy);
    QiSoptR->header_path = QiNewStr(file_name_copy);
  }
  else{
    QiSio->io_f_name = QiNewStr(file_name_copy);
  }
  
  if(recStart != NULL) QiSoptR->Nstart = atoi(recStart);
  if(recEnd != NULL) QiSoptR->Nend = atoi(recEnd);
 
  switch (in_type){
    case CDF:
      check = QiGetCSDSgenCDF( &(file_name[0]), QiSio ); /* file_name includes path */
      break;
    case QFT: 
      QiSoptR->f_type = TABULAR;           
      QiSio->io_f_extn = QiSoptR->EXTN_QFT;
      check = QiGetCSDSgenFlat( QiSio, QiSoptR);
      break;
    case QFD:
      QiSoptR->f_type = DELIMITED;           
      QiSio->io_f_extn = QiSoptR->EXTN_QFD;
      check = QiGetCSDSgenFlat( QiSio, QiSoptR);
      break;
    case CEF:
      QiSoptR->f_type = EXCHANGE;           
      QiSio->io_f_extn = QiSoptR->EXTN_CEF;
      check = QiGetCSDSgenFlat( QiSio, QiSoptR);
      break;
    case CEF2:
      QiSoptR->f_type = CAA;           
      QiSio->io_f_extn = QiSoptR->EXTN_CEF;
      check = QiGetCSDSgenFlat( QiSio, QiSoptR);
      break;
    case CEF2_GZ:
      printf("Compressed input files are not yet supported\n");
      exit(1);      
	  QiSoptR->f_type = CAA;           
      QiSio->io_f_extn = QiSoptR->EXTN_CEFGZ;
      check = QiGetCSDSgenFlat( QiSio, QiSoptR);
      break;
    case TRY:
      printf("Input file type unknown, trying to read from context\n");
      QiSoptR->f_type = UNSET;           
      check = QiGetCSDSgenFlat( QiSio, QiSoptR);
      break;
    default:
     printf("unrecognised input file\n");
     exit (1);
  } /* end switch on read */
  
  
  if(cefChecking == 1) {
  
     QiCheckCEFstructure(QiSio);
     QiCEFvalidate(QiSio);
     
  }
  
  if (check != QMW_OK) { 
    printf("%s\n", QiErrStr(check) );
    exit(1);
  }


 /***** write new file out, same stem name *****/
 
  
 /* set up write options */

  QiSoptW = (QiOptions *) QiMakeOptionsObj();
 
  /* defaults are set up already for these options, */
  /* this provides a list of what can be set. */
  
  QiSoptW->fp_display = stderr;
  QiSoptW->debug_choice = 'f';     /* 'w'  uses ARH fn to redirect to window */
  QiSoptW->header = ATTACHED;      /* DETACHED or NO_HEADER allowed */
  QiSoptW->priority = choice;
  QiSoptW->rec_numbering = rec_numbering;
  QiSoptW->attr_delim = ',';
  QiSoptW->data_delim = ',';
  QiSoptW->rec_end = '$';
  QiSoptW->row_end = '\n';
  QiSoptW->force_eor_write = force_eor_write;
  QiSoptW->writeVersion = cdfFileBack;

   
 if( (out_type == CEF || out_type == CEF2) && (in_type == CEF || in_type == CEF2) ){
     /* take care not to overwrite input file */
     long nameLen = strlen(QiSio->io_f_name);
     if (out_name != NULL  && strcmp(out_name, QiSio->io_f_name) != 0){
       if( QiSio->io_f_name != NULL) free(QiSio->io_f_name);
       QiSio->io_f_name = QiNewStr(out_name);
     }
     else{
       char * newName = (char *) malloc (nameLen+2);
       strcpy (newName, QiSio->io_f_name);
       strcat (newName, "~");
       printf(" Input and output file names are the same, output modified to %s\n", newName);
       free(QiSio->io_f_name);
       QiSio->io_f_name = QiNewStr(newName);
       free(newName);
     }
 }
 else if(out_name != NULL ) {
     /* set output stem if changed by Qtran -n argument */
     if( QiSio->io_f_name != NULL) free(QiSio->io_f_name);
     QiSio->io_f_name = QiNewStr(out_name);
 }

 switch(out_type){
   case CDF:
     QiSio->io_f_extn = QiSoptR->EXTN_CDF;
     check = QiWriteCSDSgenCDF(QiSio, QiSoptW) ;   
     break;
   case QFT:
     QiSio->io_f_extn = QiSoptR->EXTN_QFT;
     QiSoptW->f_type = TABULAR;           
     check = QiWriteCSDSgenFlat(QiSio, QiSoptW);   
     break;
   
   case QFD:
     QiSio->io_f_extn = QiSoptR->EXTN_QFD;
     QiSoptW->f_type = DELIMITED;           
     check = QiWriteCSDSgenFlat(QiSio, QiSoptW);   
     break;

   case CEF:
     QiSio->io_f_extn = QiSoptR->EXTN_CEF;
     QiSoptW->f_type = EXCHANGE;           
     check = QiWriteCSDSgenFlat(QiSio, QiSoptW);   
     break;

   case CEF2:
     QiSio->io_f_extn = QiSoptR->EXTN_CEF;
     QiSoptW->f_type = CAA;           
     check = QiWriteCSDSgenFlat(QiSio, QiSoptW);   
     break;

   case NO_OF: 
     check = QMW_OK;
     break;

   default:
     printf("cannot determine output file type\n");
     exit (1);     
 
 }  /* end switch on write */
 
 if (check != QMW_OK) { 
    if(n_errs > 0) printf("Qtran has found %d CEF2 syntax error(s) \n", n_errs);    
    if(n_warn > 0) printf("Qtran has found %d CEF2 syntax warning(s) \n", n_warn);
    printf("%s\n", QiErrStr(check) );
    exit(1);
 }

   if(cefChecking) {
     if(n_errs > 0) printf("Qtran has found %d CEF2 syntax error(s) \n", n_errs);    
     if(n_warn > 0) printf("Qtran has found %d CEF2 syntax warning(s) \n", n_warn);
     if(n_errs + n_warn == 0) printf("CEF2 syntax appears to be valid\n");
     printf("CEF2 Validation checking complete\n");
   }
   else printf( "Done\n" );
   exit(0);
                
}        /* end: main */

/* utilities that are specific to Qtran and must be different in QSAS */

/* redirects QSAS alert box call for stand alone */
 void QieAlertBox (const char * title, const char * text){
    fprintf(stderr, "%s: %s\n", title, text);
    return;

}

char * QieVersion(){
 static char str[80];
  strcpy(str, QIE_VERSION); 
  return &(str[0]);
}

int QiCEFmsg (int mode, const char *txt0,
              const char *txt1,
              const char *txt2){

  /* provides output for conformance testing */
  
  if(cefChecking == 1 ){
    if(mode == 0){
      n_warn++;
      printf("\nCEF 2 syntax warning: \n");
    }
    else if( mode != -1 ){
      n_errs++;
      printf("\nCEF 2 syntax failure: \n");
    }
    
    printf("%s ", txt0);
    printf("%s ", txt1);
    printf("%s \n", txt2);
  }
  else{
   /* do nothing */
  }
  return 0;
}
