/***********************************************************************
*           See Accompanying licence file QSAS_licence                 *
***********************************************************************/

#include "qie.h"
#include "qie_local.h"

long QiWriteCSDSgenCDF (QiCDFContents *QiSCDF, QiOptions *QiSOpt)
{

/* CDF   variables */

CDFid id;               /* ID of CDF for subsequent operations */
CDFstatus status;
char * full_name;
long check=QMW_OK;

  /* convert attribute string arrays to pointers to NRV vars */
  QiRemoveStrArrays(QiSCDF);

  /* remove ISO time format and replace with epoch */
  QiRemoveISO_TIME(QiSCDF);

  /* ensure file name and path exist */
  QiEnsureFileName(QiSCDF, QiSOpt);

  /* remove any variables that have no data or NULL ptrs */
   if (QiMakeSafe (QiSCDF) == NULL){
     QieAlertBox("Error", "A variable or data is missing in structure after file read");
     return BAD_STRUCT;
   }

  /* test input data structure to ensure it useable */

  check = QiTestStructure(QiSCDF);
  if (check != QMW_OK) {
     QieAlertBox("Error", "Sparse records. Some data is not being found");
     return BAD_STRUCT;
  }

   full_name =
     (char *) malloc(strlen(QiSCDF->io_f_path)+strlen(QiSCDF->io_f_name)+6);
      /* allow extra space for .cdf later */
   strcpy(full_name, QiSCDF->io_f_path);
   strcat(full_name, QiSCDF->io_f_name);

  /* create .cdf file */

  check = QiCreateCDF(full_name, QiSOpt);
  if(check != QMW_OK) return check;

  /* open cdf file */

   status = CDFlib(OPEN_, CDF_, full_name, &id, NULL_);
     if (status != CDF_OK) return CDF_OPEN_ERR;

  /* write structure to file */

  check = QiWriteContents(QiSCDF, QiSOpt);
    if (check != QMW_OK){
       /* delete badly formed CDF file and return warning */
       CDFlib(DELETE_, CDF_, NULL_);
       return check;
    }

   /* Close CDF and exit cleanly */

   CDFlib(CLOSE_, CDF_, NULL_);

   free(full_name);

   return QMW_OK;

} /* end  QiWriteCSDSgenCDF  */



/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiTestStructure
*
* PURPOSE: Test structure is adequate to write CSDS cdf file.
*
* DESCRIPTION: Tests that all variables have same number of records.
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM: For each variable in vardata array compare number of records
*              with that specified in top level of file object.
*
***********************************************************************/

long QiTestStructure(QiCDFContents *QiSCDF)
{
   long n;

      /* test number of records is same for each variable */
      /* This is essential to match data and time in records */

      for (n=0; n < QiSCDF->n_vars; n++){
        if( QiSCDF->vardata[n]->rec_vary == VARY ){
           if( QiSCDF->vardata[n]->max_rec_num != QiSCDF->n_recs-1){		     
             return SPARSE_RECS;
           }
        }
      }

      /* other tests may be added later */

   return QMW_OK;

 }/* end QiTestStructure */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiWriteContents
*
* PURPOSE: Write contents of structure to new cdf file.
*
* DESCRIPTION: Writes all Global attributes together with data for each variable
*              in vardata array and the variable attributes associated.
*              Variable attributes are not written for variables that do not appear in
*              vardata array, so metadata object need not be modified for subsetting.
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED: QiWriteGlobals
*                 QiWriteVariable
*
* ALGORITHM:  call QiWriteGlobals to write out all global attributes present
*               in metadata object.
*             for each variable in vardata array call QiWriteVariable to output
*             data and variable attributes.
*
***********************************************************************/

 long QiWriteContents (QiCDFContents *QiSCDF,
                       QiOptions *QiSOpt)
 {
	long check = QMW_OK;
	long n, nn, m, nextAttr;
	char *parameter_id = NULL;
	int duplicate=0;

	/* write Global Attributes */

	check = QiWriteGlobals (QiSCDF);
	if(check != QMW_OK) {
		/* issue warning and continue */
		QiDisplayMessage(QiErrStr(check), QiSOpt);
	}

	
	/* truncate variable names to cdf limit CDF_VAR_NAME_LEN */
   
	for(n=0; n< QiSCDF->n_vars; n++){
   	
		if(strlen(QiSCDF->vardata[n]->name) > CDF_VAR_NAME_LEN){
			parameter_id = (char *) malloc(strlen(QiSCDF->vardata[n]->name)+1);
			strcpy(parameter_id, QiSCDF->vardata[n]->name);
			free(QiSCDF->vardata[n]->name);
			QiSCDF->vardata[n]->name = (char*) malloc(CDF_VAR_NAME_LEN+1);
			strncpy(QiSCDF->vardata[n]->name, parameter_id, CDF_VAR_NAME_LEN - 10);
			QiSCDF->vardata[n]->name[CDF_VAR_NAME_LEN - 10] = '\0';
			strcat(QiSCDF->vardata[n]->name, "...");
			strncat(QiSCDF->vardata[n]->name, &(parameter_id[strlen(parameter_id)-7]), 7);
			QiSCDF->vardata[n]->name[CDF_VAR_NAME_LEN] = '\0';
			printf("Truncating name \"%s\"\nTo \"%s\"\n %d = %ld\n", parameter_id, QiSCDF->vardata[n]->name, CDF_VAR_NAME_LEN, strlen(QiSCDF->vardata[n]->name));
			
			/* correct attributes that point to it */
			for (nn=0; nn< QiSCDF->n_vars; nn++){
				for(m=0; m<QiSCDF->vardata[nn]->num_v_attrs; m++)
				{
					if((QiSCDF->vardata[nn]->attribute+m)->data_type == CDF_CHAR)
					{
						if( strcmp((char *)(QiSCDF->vardata[nn]->attribute+m)->data, parameter_id) == 0)
						{
							free ( (QiSCDF->vardata[nn]->attribute+m)->data );
							(QiSCDF->vardata[nn]->attribute+m)->data = (char*) malloc(CDF_VAR_NAME_LEN+1);
							strncpy((char*)(QiSCDF->vardata[nn]->attribute+m)->data, QiSCDF->vardata[n]->name, CDF_VAR_NAME_LEN);
							((char*)(QiSCDF->vardata[nn]->attribute+m)->data)[CDF_VAR_NAME_LEN] = '\0';
							(QiSCDF->vardata[nn]->attribute+m)->num_elems = 1;
						}
					}
				}
			}
			
			/* add extra attribute with full name */
			if(QiSCDF->vardata[n]->num_v_attrs < MAX_N_ATTRS)
			{
				nextAttr = QiSCDF->vardata[n]->num_v_attrs;
				(QiSCDF->vardata[n]->attribute+nextAttr)->data_type = CDF_CHAR;
				(QiSCDF->vardata[n]->attribute+nextAttr)->num_elems = 1;
				(QiSCDF->vardata[n]->attribute+nextAttr)->name = (char *) malloc(20);
				strncpy((QiSCDF->vardata[n]->attribute+nextAttr)->name, "PARAMETER_FULL_NAME", 19);
				(QiSCDF->vardata[n]->attribute+nextAttr)->name[19] = '\0';
				(QiSCDF->vardata[n]->attribute+nextAttr)->data = (char *) malloc(strlen(parameter_id)+1);
				strcpy((char*)(QiSCDF->vardata[n]->attribute+nextAttr)->data, parameter_id);
				((char*)(QiSCDF->vardata[n]->attribute+nextAttr)->data)[strlen(parameter_id)] = '\0';
				QiSCDF->vardata[n]->num_v_attrs++;
			}   /* end write of extra attr for var name too long */

		} /* end if on name too long */
	}


   /* Write Variables and their metadata */

   for(n=0; n< QiSCDF->n_vars; n++){
     check = QiWriteVariable (QiSCDF, n);
     if(check != QMW_OK) return check;
   }

   return check;

 } /* end QiWriteContents */

/***********************************************************************/

void QiRemoveStrArrays(QiCDFContents *QiSCDF){

  /* for each variable, test if attribute is string array */
  int i,j,n,nAttr,pad; 
  for (n=0; n < QiSCDF->n_vars; n++){

    struct QiSCDFVariable * v = QiSCDF->vardata[n];
      for(nAttr=0; nAttr < v->num_v_attrs; nAttr++){

         if( v->attribute[nAttr].data_type==CDF_CHAR || v->attribute[nAttr].data_type==CDF_UCHAR ){
		   if (v->attribute[nAttr].num_elems > 1){
			 long maxAttrStrLen, attrStrLen, ptr; 
			 char * charData, *newAttrName ; 
			 QiCDFVariable *attrV ;

			 if(QiSCDF->n_vars >= MAX_N_VARS){
			    printf(" Exceeded number of available variable pointers\n Text Array attribute will be concatenated\n");
				return;
			 }
			 
			 maxAttrStrLen = 1; 
			 attrStrLen = 0; 
			 charData = (char *)v->attribute[nAttr].data; 
			 
			 for(i=0; i < v->attribute[nAttr].num_elems; i++){
				/* num_elems for a Vattr is the number of strings in attr array : Changed for CEF2*/
				long entryLength =  strlen(&(charData[attrStrLen]));
				attrStrLen += entryLength+1;
				if( entryLength > maxAttrStrLen) maxAttrStrLen = entryLength;
			}
			 
		     /* create variable */
			 	
			 newAttrName = (char *) malloc(strlen(v->name) + strlen(v->attribute[nAttr].name) + 2);
			 strcpy(newAttrName, v->name);
			 strcat(newAttrName, "_");
			 strcat(newAttrName, v->attribute[nAttr].name);

			 QiSCDF->vardata[QiSCDF->n_vars] = QiMakeQiVariable();   /* add new variable at end of list */
			 attrV = QiSCDF->vardata[QiSCDF->n_vars];

			 attrV->name = (char *) malloc(strlen(newAttrName)+1);
			 strcpy( attrV->name, newAttrName);
			 attrV->number = QiSCDF->n_vars;
			 attrV->data_type = CDF_CHAR;
			 attrV->rec_vary = NOVARY;
			 attrV->max_rec_num = 0;
			 attrV->num_dims = 1;
			 attrV->dim_sizes = (long *) malloc(sizeof(long)*CDF_MAX_DIMS);
			 attrV->dim_sizes[0] = v->attribute[nAttr].num_elems;
			 attrV->dim_varys = (long *) malloc(sizeof(long)*CDF_MAX_DIMS);
			 attrV->dim_varys[0] = VARY;
			 attrV->num_v_attrs = 0;
			 attrV->num_elems = maxAttrStrLen;
			 attrV->data = (void *) malloc( sizeof(char)*(v->attribute[nAttr].num_elems*maxAttrStrLen + 1) );

			 ptr = 0;
			 for(j=0; j<v->attribute[nAttr].num_elems; j++){
			 	strcpy( &(((char*)attrV->data)[j*maxAttrStrLen]), &(charData[ptr+j]) );  /* overwrite null terminator in data */
				// pad with spaces
				for( pad=strlen(&(charData[ptr+j])); pad < maxAttrStrLen; pad++){
				  ((char*)attrV->data)[j*maxAttrStrLen+pad] = ' ';
				}
				ptr += strlen(&(charData[ptr+j])); /* +j allows for null terminator in attribute data */
			 }
			 /* change attribute to point to new variable */
			 v->attribute[nAttr].num_elems = 1;
			 free(v->attribute[nAttr].data);
			 v->attribute[nAttr].data = (void *) malloc(strlen(newAttrName)+1);
			 strcpy((char *) v->attribute[nAttr].data, newAttrName);
			 free (newAttrName);
			 
			 // add attribute to specify support_data
			 attrV->attribute[attrV->num_v_attrs].number = attrV->num_v_attrs;
			 attrV->attribute[attrV->num_v_attrs].name = (char *) malloc(strlen("PARAMETER_TYPE")+1);
			 strcpy(attrV->attribute[attrV->num_v_attrs].name, "PARAMETER_TYPE");
			 attrV->attribute[attrV->num_v_attrs].data_type = CDF_CHAR;
			 attrV->attribute[attrV->num_v_attrs].num_elems = 1; // for cef 2 handling, number of strings, not strlen
			 attrV->attribute[attrV->num_v_attrs].data = QiAttrEntryMalloc( 12, CDF_CHAR );
			 strncpy( (char *) attrV->attribute[attrV->num_v_attrs].data, "Support_Data", 12);
			 /* Note NULL terminator for char is already handled by  QiAttrEntryMalloc */

			 attrV->num_v_attrs++;

			 
		      QiSCDF->n_vars++;
		   }

		 } /* end if attr type */
		 		 
      } /* end for on attrs */
	  
  } /* end for on vars */
 
}

/***********************************************************************/


int  QiFindISOaccuracy(QiCDFContents *QiSCDF){
	struct QiSCDFVariable * variable;
	char * oldData;
	char * ptr;
	int decimals;
	
	if(QiSCDF->d0_var_num < 0) return 3;
	if(QiSCDF->n_recs < 1) return 3;

	variable = QiSCDF->vardata[QiSCDF->d0_var_num];
	if(variable->data_type != ISO_TIME) return 3;
	
	oldData = (char *) variable->data;
	ptr = strstr(oldData, ".");
	if(ptr == NULL) return 0;
	
	if(strstr(ptr, "Z") == NULL) decimals = strlen(ptr) -1 ;
	else decimals = strlen(ptr) - 2;
	
	return decimals;
}

/***********************************************************************/

long  QiRemoveISO_TIME(QiCDFContents *QiSCDF){

  long n, n_entry;
  long nAttr;
  long ind;
  long mm;
  long numEntries;
  double dataValue;
  double *dataPtr16;
  double *rangePtr16;
  double * newData;
  double * newRange;
  char * ISOstring;
  char * oldData;
  char * charData;
  double * doubleData;
  int strLen;

  /* Remove ISO from Globals */

  if(QiSCDF->UserSetSecDecimal == 0) QiSCDF->numSecDecimal = QiFindISOaccuracy(QiSCDF);
  
  for (n=0; n < QiSCDF->num_g_attrs; n++){

    /* check each entry in turn */

    for (n_entry=0; n_entry < QiSCDF->g_attr[n]->num_entries;  n_entry++){

		/* test if entry present in structure, if not go on to next */

		if(QiSCDF->g_attr[n]->entry[n_entry].exists == 0) continue;

		if(QiSCDF->g_attr[n]->entry[n_entry].data_type == ISO_TIME){
			if(QiSCDF->numSecDecimal > 3){
				QiSCDF->g_attr[n]->entry[n_entry].data_type = CDF_EPOCH16;
				if(QiSCDF->g_attr[n]->entry[n_entry].data == NULL) continue;
				dataPtr16 = (double *) malloc(2*sizeof(double));
				QiISOStringToEpoch16((char *) QiSCDF->g_attr[n]->entry[n_entry].data, &(dataPtr16[0]));
				free (QiSCDF->g_attr[n]->entry[n_entry].data);
				QiSCDF->g_attr[n]->entry[n_entry].data = (void *) dataPtr16;
				QiSCDF->g_attr[n]->entry[n_entry].num_elems = 1;
			}
			else{
				QiSCDF->g_attr[n]->entry[n_entry].data_type = CDF_EPOCH;
				if(QiSCDF->g_attr[n]->entry[n_entry].data == NULL) continue;
				dataValue = QiISOStringToEpoch((char *) QiSCDF->g_attr[n]->entry[n_entry].data);
				free (QiSCDF->g_attr[n]->entry[n_entry].data);
				QiSCDF->g_attr[n]->entry[n_entry].data = (void *) malloc(sizeof(double));
				memcpy(QiSCDF->g_attr[n]->entry[n_entry].data, &dataValue, sizeof(double));
				QiSCDF->g_attr[n]->entry[n_entry].num_elems = 1;
			}
		}

		else if(QiSCDF->g_attr[n]->entry[n_entry].data_type == ISO_TIME_RANGE){
			if(QiSCDF->numSecDecimal > 3){
				QiSCDF->g_attr[n]->entry[n_entry].data_type = CDF_EPOCH16;
				if(QiSCDF->g_attr[n]->entry[n_entry].data == NULL) continue;
				rangePtr16 = (double *) malloc(4*sizeof(double));
				QiISOStringToEpochRange16((char *) QiSCDF->g_attr[n]->entry[n_entry].data, &(rangePtr16[0]), &(rangePtr16[2])); 
				free (QiSCDF->g_attr[n]->entry[n_entry].data);
				QiSCDF->g_attr[n]->entry[n_entry].data = (void *) rangePtr16;
			
				QiSCDF->g_attr[n]->entry[n_entry].num_elems = 2;
			}
			else{
				QiSCDF->g_attr[n]->entry[n_entry].data_type = CDF_EPOCH;
				if(QiSCDF->g_attr[n]->entry[n_entry].data == NULL) continue;
				newRange = (double *) malloc(2*sizeof(double));
				QiISOStringToEpochRange((char *) QiSCDF->g_attr[n]->entry[n_entry].data, &(newRange[0]), &(newRange[1])); 
				free (QiSCDF->g_attr[n]->entry[n_entry].data);
				QiSCDF->g_attr[n]->entry[n_entry].data = (void *) newRange;
			
				QiSCDF->g_attr[n]->entry[n_entry].num_elems = 2;
			}
		}

    } /* end for over entries */

 } /* end for over G attributes */

	/* Remove ISO from Variables */

	for (n=0; n < QiSCDF->n_vars; n++){

		struct QiSCDFVariable * variable = QiSCDF->vardata[n];

		/* Remove ISO from Var data */

		if( variable->data_type == ISO_TIME){
			
			if(QiSCDF->numSecDecimal > 3){
				strLen = variable->num_elems;
				variable->num_elems = 1;
				variable->data_type = CDF_EPOCH16;
				variable->sizeofentry = (long) 2*sizeof(double);

				numEntries = 1;
				for( ind=0; ind < QiSCDF->vardata[n]->num_dims; ind++){
					if(variable->dim_varys[ind] == VARY) numEntries *= variable->dim_sizes[ind];
				}

				if( variable->rec_vary == VARY) numEntries *= QiSCDF->n_recs;
				if( variable->data == NULL) {
					printf("variable %s data is NULL\n", variable->name);
					continue;
				}
				newData = (double *)malloc(2*sizeof(double)*numEntries);
 				ISOstring = (char *)malloc(sizeof(char)*(strLen + 1));
				oldData = (char *) variable->data;

				for( mm=0; mm < numEntries; mm++){
					strncpy(ISOstring, (char *)&(oldData[mm*strLen]), strLen);
					ISOstring[strLen] = '\0';
					QiISOStringToEpoch16( ISOstring, &(newData[2*mm]) );
 				}
				free(ISOstring);
 				free(variable->data);
				variable->data = newData;
				
			}
			else{
				strLen = variable->num_elems;
				variable->num_elems = 1;
				variable->data_type = CDF_EPOCH;
				variable->sizeofentry = (long) sizeof(double);

				numEntries = 1;
				for( ind=0; ind < QiSCDF->vardata[n]->num_dims; ind++){
					if(variable->dim_varys[ind] == VARY) numEntries *= variable->dim_sizes[ind];
				}

				if( variable->rec_vary == VARY) numEntries *= QiSCDF->n_recs;
				if( variable->data == NULL) {
					printf("variable %s data is NULL\n", variable->name);
					continue;
				}
				newData = (double *)malloc(sizeof(double)*numEntries);
 				ISOstring = (char *)malloc(sizeof(char)*(strLen + 1));
				oldData = (char *) variable->data;

				for( mm=0; mm < numEntries; mm++){
					strncpy(ISOstring, (char *)&(oldData[mm*strLen]), strLen);
					ISOstring[strLen] = '\0';
					newData[mm] = QiISOStringToEpoch( ISOstring );
 				}
				free(ISOstring);
 				free(variable->data);
				variable->data = newData;
			}
			
			/* fix epoch specific attributes  */
			for( nAttr=0; nAttr < variable->num_v_attrs; nAttr++){
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "UNITS") == 0) {
					if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
					charData  = (char *) malloc(sizeof(char)*3);
					if(QiSCDF->numSecDecimal > 3) strcpy(charData, "ps");
					else strcpy(charData, "ms");
					variable->attribute[nAttr].data = charData;
					variable->attribute[nAttr].data_type = CDF_CHAR;
 					variable->attribute[nAttr].num_elems = 1;
				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "SI_CONVERSION") == 0) {
					if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
					charData  = (char *) malloc(sizeof(char)*9);
					if(QiSCDF->numSecDecimal > 3) strcpy(charData, "1.0e-12>s");
					else strcpy(charData, "1.0e-3>s");
					variable->attribute[nAttr].data = charData;
					variable->attribute[nAttr].data_type = CDF_CHAR;
					variable->attribute[nAttr].num_elems = 1;
				}
 				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "FILLVAL") == 0) {
					if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						QiISOStringToEpoch16((char*) variable->attribute[nAttr].data, &(doubleData[0]));						
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else {
						doubleData  = (double *) malloc(sizeof(double));
						doubleData[0] = QiISOStringToEpoch((char*) variable->attribute[nAttr].data);						
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMIN") == 0) {
					if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						QiISOStringToEpoch16(charData, &(doubleData[0]));
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else{
						doubleData  = (double *) malloc(sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						*doubleData = QiISOStringToEpoch(charData);
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMAX") == 0) {
					if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						QiISOStringToEpoch16(charData, &(doubleData[0]));
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else{
						doubleData  = (double *) malloc(sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						*doubleData = QiISOStringToEpoch(charData);
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "SCALEMIN") == 0) {
					if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						QiISOStringToEpoch16(charData, &(doubleData[0]));
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else{
						doubleData  = (double *) malloc(sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						*doubleData = QiISOStringToEpoch(charData);
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
 				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "SCALEMAX") == 0) {
					if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						QiISOStringToEpoch16(charData, &(doubleData[0]));
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else{
						doubleData  = (double *) malloc(sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						*doubleData = QiISOStringToEpoch(charData);
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_PLUS") == 0) {
					if(variable->attribute[nAttr].data_type == CDF_DOUBLE && variable->attribute[nAttr].data != NULL){
						doubleData =  (double *) variable->attribute[nAttr].data;
           				if(QiSCDF->numSecDecimal > 3) *doubleData *= 1.e12 ; /* convert s to pico s */
						else *doubleData *= 1000 ; /* convert s to ms */
         			}
				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_MINUS") == 0) {
					if(variable->attribute[nAttr].data_type == CDF_DOUBLE && variable->attribute[nAttr].data != NULL){
						doubleData =  (double *) variable->attribute[nAttr].data;
           				if(QiSCDF->numSecDecimal > 3) *doubleData *= 1.e12 ; /* convert s to pico s */
						else *doubleData *= 1000 ; /* convert s to ms */
					}
				}
		
				/* fix other attributes if ISO_TIME */
				if( (variable->attribute)[nAttr].data_type == ISO_TIME){
					if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						QiISOStringToEpoch16(charData, &(doubleData[0]));
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else{
						doubleData  = (double *) malloc(sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						*doubleData = QiISOStringToEpoch(charData);
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
				}

				if( (variable->attribute)[nAttr].data_type == ISO_TIME_RANGE){
					if(QiSCDF->numSecDecimal > 3){
						charData = (char*)  variable->attribute[nAttr].data;
						rangePtr16 = (double *) malloc(4*sizeof(double));
						QiISOStringToEpochRange16(charData, &(rangePtr16[0]), &(rangePtr16[2])); 
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = (void *) rangePtr16;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 2;
					}
					else{
						charData = (char*)  variable->attribute[nAttr].data;
						newRange = (double *) malloc(2*sizeof(double));
						QiISOStringToEpochRange(charData, &(newRange[0]), &(newRange[1])); 
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = (void *) newRange;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 2;
					}
				}


			} /* end of for over attributes */

		} /* end if on ISO_TIME var type */
	
		/* Remove ISO TIME RANGE from Var data */

		else if( variable->data_type == ISO_TIME_RANGE){
			if(QiSCDF->numSecDecimal > 3){
			
 				strLen = variable->num_elems;
				variable->num_elems = 1;
				variable->data_type = CDF_EPOCH16;
				variable->num_dims = 1;
				variable->dim_sizes[0] = 2;
				variable->dim_varys[0] = VARY;

				variable->sizeofentry = (long) 2*sizeof(double);

				if( variable->rec_vary == VARY) numEntries = QiSCDF->n_recs;
				else numEntries = 1;
	  
				if( variable->data == NULL) {
					printf("variable %s data is NULL\n", variable->name);
					continue;
				}
				newData = (double *)malloc(4*sizeof(double)*numEntries);
				ISOstring = (char *)malloc(sizeof(char)*(strLen + 1));
				oldData = (char *) variable->data;

				for( mm=0; mm < numEntries; mm++){
					strncpy(ISOstring, (char *)&(oldData[mm*strLen]), strLen);
					ISOstring[strLen] = '\0';
					QiISOStringToEpochRange16( ISOstring, &(newData[4*mm]), &(newData[4*mm+2]) );
				}
				free(ISOstring);
				free(variable->data);
				variable->data = newData;
			}
			else{
 				strLen = variable->num_elems;
				variable->num_elems = 1;
				variable->data_type = CDF_EPOCH;
				variable->num_dims = 1;
				variable->dim_sizes[0] = 2;
				variable->dim_varys[0] = VARY;

				variable->sizeofentry = (long) sizeof(double);

				if( variable->rec_vary == VARY) numEntries = QiSCDF->n_recs;
				else numEntries = 1;
	  
				if( variable->data == NULL) {
					printf("variable %s data is NULL\n", variable->name);
					continue;
				}
				newData = (double *)malloc(2*sizeof(double)*numEntries);
				ISOstring = (char *)malloc(sizeof(char)*(strLen + 1));
				oldData = (char *) variable->data;

				for( mm=0; mm < numEntries; mm++){
					strncpy(ISOstring, (char *)&(oldData[mm*strLen]), strLen);
					ISOstring[strLen] = '\0';
					QiISOStringToEpochRange( ISOstring, &(newData[2*mm]), &(newData[2*mm+1]) );
				}
				free(ISOstring);
				free(variable->data);
				variable->data = newData;
			}
			/* fix epoch specific attributes */
			
			for( nAttr=0; nAttr < variable->num_v_attrs; nAttr++){
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "UNITS") == 0) {
					if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
					charData  = (char *) malloc(sizeof(char)*3);
					if(QiSCDF->numSecDecimal > 3) strcpy(charData, "ps");
					else strcpy(charData, "ms");
					variable->attribute[nAttr].data = charData;
					variable->attribute[nAttr].data_type = CDF_CHAR;
					variable->attribute[nAttr].num_elems = 1;
				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "SI_CONVERSION") == 0) {
					if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
					charData  = (char *) malloc(sizeof(char)*9);
					if(QiSCDF->numSecDecimal > 3) strcpy(charData, "1.0e-12>s");
					else strcpy(charData, "1.0e-3>s");
					variable->attribute[nAttr].data = charData;
					variable->attribute[nAttr].data_type = CDF_CHAR;
					variable->attribute[nAttr].num_elems = 1;
				}
		       	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "FILLVAL") == 0) {
					if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
         			if(QiSCDF->numSecDecimal > 3){
         				doubleData  = (double *) malloc(2*sizeof(double));
         				doubleData[0] = 0.;
         				doubleData[1] = 0.;
         				variable->attribute[nAttr].data = doubleData;
         				variable->attribute[nAttr].data_type = CDF_EPOCH16;
         				variable->attribute[nAttr].num_elems = 1;
					}
					else{
         				doubleData  = (double *) malloc(sizeof(double));
         				*doubleData = -1.0e-31;
         				variable->attribute[nAttr].data = doubleData;
         				variable->attribute[nAttr].data_type = CDF_EPOCH;
         				variable->attribute[nAttr].num_elems = 1;
					}
       			}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMIN") == 0) {
         			if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						QiISOStringToEpoch16(charData, &(doubleData[0]));
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else{
						doubleData  = (double *) malloc(sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						*doubleData = QiISOStringToEpoch(charData);
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMAX") == 0) {
					if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						QiISOStringToEpoch16(charData, &(doubleData[0]));
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else{
						doubleData  = (double *) malloc(sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						*doubleData = QiISOStringToEpoch(charData);
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "SCALEMIN") == 0) {
					if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						QiISOStringToEpoch16(charData, &(doubleData[0]));
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else{
						doubleData  = (double *) malloc(sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						*doubleData = QiISOStringToEpoch(charData);
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "SCALEMAX") == 0) {
					if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						QiISOStringToEpoch16(charData, &(doubleData[0]));
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else{
						doubleData  = (double *) malloc(sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						*doubleData = QiISOStringToEpoch(charData);
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
				}
				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_PLUS") == 0) {
					if(variable->attribute[nAttr].data_type == CDF_DOUBLE && variable->attribute[nAttr].data != NULL){
						doubleData =  (double *) variable->attribute[nAttr].data;
           				if(QiSCDF->numSecDecimal > 3) *doubleData *= 1.e12 ; /* convert s to pico s */
						else *doubleData *= 1000 ; /* convert s to ms */
					}
				}
 				if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_MINUS") == 0) {
					if(variable->attribute[nAttr].data_type == CDF_DOUBLE && variable->attribute[nAttr].data != NULL){
						doubleData =  (double *) variable->attribute[nAttr].data;
           				if(QiSCDF->numSecDecimal > 3) *doubleData *= 1.e12 ; /* convert s to pico s */
						else *doubleData *= 1000 ; /* convert s to ms */
					}
				}
		
				/* fix other attributes if ISO_TIME */
				if( (variable->attribute)[nAttr].data_type == ISO_TIME){
					if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						QiISOStringToEpoch16(charData, &(doubleData[0]));
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else{
						doubleData  = (double *) malloc(sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						*doubleData = QiISOStringToEpoch(charData);
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
				}

				if( (variable->attribute)[nAttr].data_type == ISO_TIME_RANGE){
					if(QiSCDF->numSecDecimal > 3){
						charData = (char*)  variable->attribute[nAttr].data;
						rangePtr16 = (double *) malloc(4*sizeof(double));
						QiISOStringToEpochRange16(charData, &(rangePtr16[0]), &(rangePtr16[2]));            
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = rangePtr16;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 2;
					}
					else{
						charData = (char*)  variable->attribute[nAttr].data;
						newRange = (double *) malloc(2*sizeof(double));
						QiISOStringToEpochRange(charData, &(newRange[0]), &(newRange[1]));            
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = newRange;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 2;
					}
				}

			} /* end of for over attributes */

		} /* end if on ISO_TIME_RANGE var type */
	
		else{
			/* Just remove ISO from V Attrs for vars that are not ISO Times themselves */
			/* its not clear we often get here as data type for Vattr usually only available for cdf input */
			/* flat file read assumes attributes are same as variable or text  */

			for( nAttr=0; nAttr < variable->num_v_attrs; nAttr++){

				if( (variable->attribute)[nAttr].data_type == ISO_TIME){
					if(QiSCDF->numSecDecimal > 3){
						doubleData  = (double *) malloc(2*sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						QiISOStringToEpoch16(charData, &(doubleData[0]));
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 1;
					}
					else{
						doubleData  = (double *) malloc(sizeof(double));
						charData = (char*)  variable->attribute[nAttr].data;
						*doubleData = QiISOStringToEpoch(charData);
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = doubleData;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 1;
					}
				}

				if( (variable->attribute)[nAttr].data_type == ISO_TIME_RANGE){
					if(QiSCDF->numSecDecimal > 3){
						charData = (char*)  variable->attribute[nAttr].data;
						rangePtr16 = (double *) malloc(4*sizeof(double));
						QiISOStringToEpochRange16(charData, &(rangePtr16[0]), &(rangePtr16[2]));            
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = rangePtr16;
						variable->attribute[nAttr].data_type = CDF_EPOCH16;
						variable->attribute[nAttr].num_elems = 2;
					}
					else{
						charData = (char*)  variable->attribute[nAttr].data;
						newRange = (double *) malloc(2*sizeof(double));
						QiISOStringToEpochRange(charData, &(newRange[0]), &(newRange[1]));            
						if (variable->attribute[nAttr].data != NULL) free(variable->attribute[nAttr].data);
						variable->attribute[nAttr].data = newRange;
						variable->attribute[nAttr].data_type = CDF_EPOCH;
						variable->attribute[nAttr].num_elems = 2;
					}
				}
			}
		}

	} /* end for over variables */

	return QMW_OK;

} /* end   QiRemoveISO_TIME  */

/***********************************************************************/


long  QiRemoveEpoch(QiCDFContents *QiSCDF){

  long n, n_entry;
  long nAttr;
  long ind;
  long mm;
  double *dataValue;
  char * dataStr;
  char * start;
  char * end;
  char * rangeStr;
  long numEntries;
  char * newData;
  double * oldData;
  char * charData;
  double * doubleData;

  /* Remove EPOCH from Globals */

  for (n=0; n < QiSCDF->num_g_attrs; n++){

    /* check each entry in turn */

    for (n_entry=0; n_entry < QiSCDF->g_attr[n]->num_entries;  n_entry++){

		/* test if entry present in structure, if not go on to next */

		if(QiSCDF->g_attr[n]->entry[n_entry].exists == 0) continue;
		if(QiSCDF->g_attr[n]->entry[n_entry].data == NULL) continue;

		if(QiSCDF->g_attr[n]->entry[n_entry].data_type == CDF_EPOCH){

			if(QiSCDF->g_attr[n]->entry[n_entry].num_elems == 1)
			{
				QiSCDF->g_attr[n]->entry[n_entry].data_type = ISO_TIME;
				dataValue = (double *) QiSCDF->g_attr[n]->entry[n_entry].data;
				dataStr = QiNewStr(QiEpochToISOString(*dataValue, 'T'));
				free (QiSCDF->g_attr[n]->entry[n_entry].data);
				QiSCDF->g_attr[n]->entry[n_entry].data = dataStr;
				QiSCDF->g_attr[n]->entry[n_entry].num_elems = strlen(dataStr);
			}
			else if(QiSCDF->g_attr[n]->entry[n_entry].num_elems == 2){
				QiSCDF->g_attr[n]->entry[n_entry].data_type = ISO_TIME_RANGE;
				dataValue = (double *) QiSCDF->g_attr[n]->entry[n_entry].data;
				start = QiNewStr(QiEpochToISOString(dataValue[0], 'T'));
				end = QiNewStr(QiEpochToISOString(dataValue[1], 'T'));
				rangeStr = (char *) malloc((strlen(start) + strlen(end) + 2 ) * sizeof(char));
				strcpy(rangeStr, start);
				strcat(rangeStr, "/");
				strcat(rangeStr, end);
				free (QiSCDF->g_attr[n]->entry[n_entry].data);
				free (start);
				free (end);
				
				QiSCDF->g_attr[n]->entry[n_entry].data = rangeStr;
				QiSCDF->g_attr[n]->entry[n_entry].num_elems = strlen(rangeStr);
			}
			else{     
				printf("G Attr %s array of time not handled\n", QiSCDF->g_attr[n]->name);
      			continue;
			}
		}
		else if(QiSCDF->g_attr[n]->entry[n_entry].data_type == CDF_EPOCH16){

			if(QiSCDF->g_attr[n]->entry[n_entry].num_elems == 1)
			{
				QiSCDF->g_attr[n]->entry[n_entry].data_type = ISO_TIME;
				dataValue = (double *) QiSCDF->g_attr[n]->entry[n_entry].data;
				dataStr = QiNewStr(QiEpoch16ToISOString(dataValue, 'T'));
				free (QiSCDF->g_attr[n]->entry[n_entry].data);
				QiSCDF->g_attr[n]->entry[n_entry].data = dataStr;
				QiSCDF->g_attr[n]->entry[n_entry].num_elems = strlen(dataStr);
			}
			else if(QiSCDF->g_attr[n]->entry[n_entry].num_elems == 2){
				QiSCDF->g_attr[n]->entry[n_entry].data_type = ISO_TIME_RANGE;
				dataValue = (double *) QiSCDF->g_attr[n]->entry[n_entry].data;
				start = QiNewStr(QiEpoch16ToISOString(&(dataValue[0]),  'T'));
				end = QiNewStr(QiEpoch16ToISOString(&(dataValue[2]), 'T'));
				rangeStr = (char *) malloc((strlen(start) + strlen(end) + 2 ) * sizeof(char));
				strcpy(rangeStr, start);
				strcat(rangeStr, "/");
				strcat(rangeStr, end);
				free (QiSCDF->g_attr[n]->entry[n_entry].data);
				free (start);
				free (end);
				
				QiSCDF->g_attr[n]->entry[n_entry].data = rangeStr;
				QiSCDF->g_attr[n]->entry[n_entry].num_elems = strlen(rangeStr);
			}
			else{     
				printf("G Attr %s array of time not handled\n", QiSCDF->g_attr[n]->name);
      			continue;
			}
		}

    } /* end for over entries */

 } /* end for over G attributes */

  /* Remove EPOCH from Variables */

  for (n=0; n < QiSCDF->n_vars; n++){

    struct QiSCDFVariable * variable = QiSCDF->vardata[n];

    if(variable->data == NULL){
      printf("Variable %s data is NULL\n", variable->name);
      continue;
    }
    /* Remove EPOCH from Var data */
    if( variable->data_type == CDF_EPOCH){


      numEntries = 1;
      for( ind=0; ind < QiSCDF->vardata[n]->num_dims; ind++){

        if(variable->dim_varys[ind] == VARY) numEntries *= variable->dim_sizes[ind];
      }

	  // Handle range variable and other epochs separately
	  
	  if(numEntries == 2){
	  
	  	/* Range type epoch */
		
      variable->data_type = ISO_TIME_RANGE;
      variable->sizeofentry = (long) sizeof(char);
      variable->num_dims = 0;

		
      	if( variable->rec_vary == VARY) numEntries = QiSCDF->n_recs;
		else numEntries = 1;
		
      	newData = (char *)malloc(sizeof(char)*numEntries*(2*ISO_TIME_LEN+1) ); 
      	oldData = (double *) variable->data;

      	variable->num_elems = 2*ISO_TIME_LEN+1;
      	for( mm=0; mm < numEntries; mm++){
			start = QiNewStr(QiEpochToISOString( oldData[2*mm], 'T'));
			end = QiNewStr(QiEpochToISOString( oldData[2*mm+1], 'T'));
        	strcpy(&(newData[mm*(2*ISO_TIME_LEN+1)]), start);
			strcat(&(newData[mm*(2*ISO_TIME_LEN+1)]), "/");
        	strcat(&(newData[mm*(2*ISO_TIME_LEN+1)]), end);
			free(start);
			free(end);
		}
      	free(variable->data);
      	variable->data = newData;

      	/* fix epoch specific attributes */
      	for( nAttr=0; nAttr < variable->num_v_attrs; nAttr++){
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "UNITS") == 0) {
           		free(variable->attribute[nAttr].data);
           		charData  = (char *) malloc(sizeof(char)*2);
           		strcpy(charData, "s");
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = CDF_CHAR;
           		variable->attribute[nAttr].num_elems = 1;
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "SI_CONVERSION") == 0) {
           		free(variable->attribute[nAttr].data);
           		charData  = (char *) malloc(sizeof(char)*6);
           		strcpy(charData, "1.0>s");
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = CDF_CHAR;
           		variable->attribute[nAttr].num_elems = 1;
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "FILLVAL") == 0) {
	       		if(variable->attribute[nAttr].data_type == CDF_EPOCH){
               		free(variable->attribute[nAttr].data);
               		charData  = (char *) malloc(sizeof(char)*(2*ISO_TIME_LEN+1));
               		strcpy(charData, "0000-01-01T00:00:00.000Z/0000-01-01T00:00:00.000Z"); // not obvious data will match this
               		variable->attribute[nAttr].data = charData;
               		variable->attribute[nAttr].data_type = ISO_TIME_RANGE;
               		variable->attribute[nAttr].num_elems = 1;
		  		}
		  		// if char type already, leave alone
         	} 
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMIN") == 0) {
           		doubleData  = (double *)  variable->attribute[nAttr].data;
           		charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           		strcpy(charData, QiEpochToISOString(*doubleData, 'T'));
           		free(variable->attribute[nAttr].data);
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = ISO_TIME;
           		variable->attribute[nAttr].num_elems = 1;
         	}
			if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMAX") == 0) {
           		doubleData  = (double *)  variable->attribute[nAttr].data;
           		charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           		strcpy(charData, QiEpochToISOString(*doubleData, 'T'));
           		free(variable->attribute[nAttr].data);
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = ISO_TIME;
           		variable->attribute[nAttr].num_elems = 1;
         	}
			if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_PLUS") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_DOUBLE){
             		doubleData =  (double *) variable->attribute[nAttr].data;
             		*doubleData /= 1000 ; /* convert ms to s */
           		}
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_MINUS") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_DOUBLE){
             		doubleData =  (double *) variable->attribute[nAttr].data;
             		*doubleData /= 1000 ; /* convert ms to s */
           		}
         	}
	  	} /* end for on attrs */
	  }
	  else 
	  {
	  
		variable->data_type = ISO_TIME;
		variable->sizeofentry = (long) sizeof(char);

      	if( variable->rec_vary == VARY) numEntries *= QiSCDF->n_recs;

      	newData = (char *)malloc(sizeof(char)*numEntries*ISO_TIME_LEN);
      	oldData = (double *) variable->data;

      	variable->num_elems = ISO_TIME_LEN;
      	for( mm=0; mm < numEntries; mm++){
        	strcpy(&(newData[mm*ISO_TIME_LEN]), QiEpochToISOString( oldData[mm], 'T' ));
      	}
      	free(variable->data);
      	variable->data = newData;

      	/* fix epoch specific attributes */
      	for( nAttr=0; nAttr < variable->num_v_attrs; nAttr++){
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "UNITS") == 0) {
           		free(variable->attribute[nAttr].data);
           		charData  = (char *) malloc(sizeof(char)*2);
           		strcpy(charData, "s");
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = CDF_CHAR;
           		variable->attribute[nAttr].num_elems = 1;
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "SI_CONVERSION") == 0) {
           		free(variable->attribute[nAttr].data);
           		charData  = (char *) malloc(sizeof(char)*6);
           		strcpy(charData, "1.0>s");
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = CDF_CHAR;
           		variable->attribute[nAttr].num_elems = 1;
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "FILLVAL") == 0) {
	       		if(variable->attribute[nAttr].data_type == CDF_EPOCH){
               		free(variable->attribute[nAttr].data);
               		charData  = (char *) malloc(sizeof(char)*ISO_TIME_LEN);
               		strcpy(charData, "0000-01-01T00:00:00.000Z"); // not obvious data will match this
               		variable->attribute[nAttr].data = charData;
               		variable->attribute[nAttr].data_type = ISO_TIME;
               		variable->attribute[nAttr].num_elems = 1;
		  		}
		  		// if char type already, leave alone
         	} 
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMIN") == 0) {
           		doubleData  = (double *)  variable->attribute[nAttr].data;
           		charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           		strcpy(charData, QiEpochToISOString(*doubleData, 'T'));
           		free(variable->attribute[nAttr].data);
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = ISO_TIME;
           		variable->attribute[nAttr].num_elems = 1;
         	}
			if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMAX") == 0) {
           		doubleData  = (double *)  variable->attribute[nAttr].data;
           		charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           		strcpy(charData, QiEpochToISOString(*doubleData, 'T'));
           		free(variable->attribute[nAttr].data);
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = ISO_TIME;
           		variable->attribute[nAttr].num_elems = 1;
         	}
			if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_PLUS") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_DOUBLE){
             		doubleData =  (double *) variable->attribute[nAttr].data;
             		*doubleData /= 1000 ; /* convert ms to s */
           		}
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_MINUS") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_DOUBLE){
             		doubleData =  (double *) variable->attribute[nAttr].data;
             		*doubleData /= 1000 ; /* convert ms to s */
           		}
         	}

      	} /* end for over attributes */
	  } /* end if over numEntries entries */
    } /* end if on CDF_EPOCH var type */
    else if( variable->data_type == CDF_EPOCH16){


      numEntries = 1;
      for( ind=0; ind < QiSCDF->vardata[n]->num_dims; ind++){

        if(variable->dim_varys[ind] == VARY) numEntries *= variable->dim_sizes[ind];
      }

	  // Handle range variable and other epochs separately
	  
	  if(numEntries == 2){
	  
	  	/* Range type epoch */
		
      variable->data_type = ISO_TIME_RANGE;
      variable->sizeofentry = (long) sizeof(char);
      variable->num_dims = 0;

		
      	if( variable->rec_vary == VARY) numEntries = QiSCDF->n_recs;
		else numEntries = 1;
		
      	newData = (char *)malloc(sizeof(char)*numEntries*(2*ISO_TIME_LEN+1) ); 
      	oldData = (double *) variable->data;

      	variable->num_elems = 2*ISO_TIME_LEN+1;
      	for( mm=0; mm < numEntries; mm++){
			start = QiNewStr(QiEpoch16ToISOString( &(oldData[4*mm]), 'T'));
			end = QiNewStr(QiEpoch16ToISOString( &(oldData[4*mm+2]), 'T'));
        	strcpy(&(newData[mm*(2*ISO_TIME_LEN+1)]), start);
			strcat(&(newData[mm*(2*ISO_TIME_LEN+1)]), "/");
        	strcat(&(newData[mm*(2*ISO_TIME_LEN+1)]), end);
			free(start);
			free(end);
		}
      	free(variable->data);
      	variable->data = newData;

      	/* fix epoch specific attributes */
      	for( nAttr=0; nAttr < variable->num_v_attrs; nAttr++){
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "UNITS") == 0) {
           		free(variable->attribute[nAttr].data);
           		charData  = (char *) malloc(sizeof(char)*2);
           		strcpy(charData, "s");
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = CDF_CHAR;
           		variable->attribute[nAttr].num_elems = 1;
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "SI_CONVERSION") == 0) {
           		free(variable->attribute[nAttr].data);
           		charData  = (char *) malloc(sizeof(char)*6);
           		strcpy(charData, "1.0>s");
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = CDF_CHAR;
           		variable->attribute[nAttr].num_elems = 1;
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "FILLVAL") == 0) {
	       		if(variable->attribute[nAttr].data_type == CDF_EPOCH ){
               		free(variable->attribute[nAttr].data);
               		charData  = (char *) malloc(sizeof(char)*(2*ISO_TIME_LEN+1));
               		strcpy(charData, "0000-01-01T00:00:00.000Z/0000-01-01T00:00:00.000Z"); 
               		variable->attribute[nAttr].data = charData;
               		variable->attribute[nAttr].data_type = ISO_TIME_RANGE;
               		variable->attribute[nAttr].num_elems = 1;
		  		}
	       		else if(variable->attribute[nAttr].data_type == CDF_EPOCH16){
               		free(variable->attribute[nAttr].data);
               		charData  = (char *) malloc(sizeof(char)*(2*ISO_TIME_LEN+1));
               		strcpy(charData, "0000-01-01T00:00:00.000000000000Z/0000-01-01T00:00:00.000000000000Z"); 
               		variable->attribute[nAttr].data = charData;
               		variable->attribute[nAttr].data_type = ISO_TIME_RANGE;
               		variable->attribute[nAttr].num_elems = 1;
		  		}
		  		// if char type already, leave alone
         	} 
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMIN") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_EPOCH ){
					doubleData  = (double *)  variable->attribute[nAttr].data;
           			charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           			strcpy(charData, QiEpochToISOString(*doubleData, 'T'));
           			free(variable->attribute[nAttr].data);
           			variable->attribute[nAttr].data = charData;
           			variable->attribute[nAttr].data_type = ISO_TIME;
           			variable->attribute[nAttr].num_elems = 1;
				}
				else if( variable->attribute[nAttr].data_type == CDF_EPOCH16){
					doubleData  = (double *)  variable->attribute[nAttr].data;
           			charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           			strcpy(charData, QiEpoch16ToISOString(doubleData, 'T'));
           			free(variable->attribute[nAttr].data);
           			variable->attribute[nAttr].data = charData;
           			variable->attribute[nAttr].data_type = ISO_TIME;
           			variable->attribute[nAttr].num_elems = 1;
				}
         	}
			if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMAX") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_EPOCH ){
           			doubleData  = (double *)  variable->attribute[nAttr].data;
           			charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           			strcpy(charData, QiEpochToISOString(*doubleData, 'T'));
           			free(variable->attribute[nAttr].data);
           			variable->attribute[nAttr].data = charData;
           			variable->attribute[nAttr].data_type = ISO_TIME;
           			variable->attribute[nAttr].num_elems = 1;
				}
				else if( variable->attribute[nAttr].data_type == CDF_EPOCH16){
           			doubleData  = (double *)  variable->attribute[nAttr].data;
           			charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           			strcpy(charData, QiEpoch16ToISOString(doubleData, 'T'));
           			free(variable->attribute[nAttr].data);
           			variable->attribute[nAttr].data = charData;
           			variable->attribute[nAttr].data_type = ISO_TIME;
           			variable->attribute[nAttr].num_elems = 1;
				}
         	}
			if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_PLUS") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_DOUBLE){
             		doubleData =  (double *) variable->attribute[nAttr].data;
             		*doubleData *= 1.e-12 ; /* convert ps to s */
           		}
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_MINUS") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_DOUBLE){
             		doubleData =  (double *) variable->attribute[nAttr].data;
             		*doubleData *= 1.e-12 ; /* convert ps to s */
           		}
         	}
	  	} /* end for on attrs */
	  }
	  else 
	  {
	  
		variable->data_type = ISO_TIME;
		variable->sizeofentry = (long) sizeof(char);

      	if( variable->rec_vary == VARY) numEntries *= QiSCDF->n_recs;

      	newData = (char *)malloc(sizeof(char)*numEntries*ISO_TIME_LEN);
      	oldData = (double *) variable->data;

      	variable->num_elems = ISO_TIME_LEN;
      	for( mm=0; mm < numEntries; mm++){
        	strcpy(&(newData[mm*ISO_TIME_LEN]), QiEpoch16ToISOString( &(oldData[2*mm]), 'T' ));
      	}
      	free(variable->data);
      	variable->data = newData;

      	/* fix epoch specific attributes  */
      	for( nAttr=0; nAttr < variable->num_v_attrs; nAttr++){
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "UNITS") == 0) {
           		free(variable->attribute[nAttr].data);
           		charData  = (char *) malloc(sizeof(char)*2);
           		strcpy(charData, "s");
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = CDF_CHAR;
           		variable->attribute[nAttr].num_elems = 1;
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "SI_CONVERSION") == 0) {
           		free(variable->attribute[nAttr].data);
           		charData  = (char *) malloc(sizeof(char)*6);
           		strcpy(charData, "1.0>s");
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = CDF_CHAR;
           		variable->attribute[nAttr].num_elems = 1;
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "FILLVAL") == 0) {
	       		if(variable->attribute[nAttr].data_type == CDF_EPOCH){
               		free(variable->attribute[nAttr].data);
               		charData  = (char *) malloc(sizeof(char)*ISO_TIME_LEN);
               		strcpy(charData, "0000-01-01T00:00:00.000Z"); // not obvious data will match this
               		variable->attribute[nAttr].data = charData;
               		variable->attribute[nAttr].data_type = ISO_TIME;
               		variable->attribute[nAttr].num_elems = 1;
		  		}
	       		else if( variable->attribute[nAttr].data_type == CDF_EPOCH16){
               		free(variable->attribute[nAttr].data);
               		charData  = (char *) malloc(sizeof(char)*ISO_TIME_LEN);
               		strcpy(charData, "0000-01-01T00:00:00.000000000000Z"); // not obvious data will match this
               		variable->attribute[nAttr].data = charData;
               		variable->attribute[nAttr].data_type = ISO_TIME;
               		variable->attribute[nAttr].num_elems = 1;
		  		}
		  		// if char type already, leave alone
         	} 
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMIN") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_EPOCH ){
					doubleData  = (double *)  variable->attribute[nAttr].data;
           			charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           			strcpy(charData, QiEpochToISOString(*doubleData, 'T'));
           			free(variable->attribute[nAttr].data);
           			variable->attribute[nAttr].data = charData;
           			variable->attribute[nAttr].data_type = ISO_TIME;
           			variable->attribute[nAttr].num_elems = 1;
				}
				else if( variable->attribute[nAttr].data_type == CDF_EPOCH16){
					doubleData  = (double *)  variable->attribute[nAttr].data;
           			charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           			strcpy(charData, QiEpoch16ToISOString(doubleData, 'T'));
           			free(variable->attribute[nAttr].data);
           			variable->attribute[nAttr].data = charData;
           			variable->attribute[nAttr].data_type = ISO_TIME;
           			variable->attribute[nAttr].num_elems = 1;
				}
         	}
			if( strcmp(QiToUpper(variable->attribute[nAttr].name), "VALIDMAX") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_EPOCH ){
           			doubleData  = (double *)  variable->attribute[nAttr].data;
           			charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           			strcpy(charData, QiEpochToISOString(*doubleData, 'T'));
           			free(variable->attribute[nAttr].data);
           			variable->attribute[nAttr].data = charData;
           			variable->attribute[nAttr].data_type = ISO_TIME;
           			variable->attribute[nAttr].num_elems = 1;
				}
				else if( variable->attribute[nAttr].data_type == CDF_EPOCH16){
           			doubleData  = (double *)  variable->attribute[nAttr].data;
           			charData = (char*) malloc(sizeof(char)*ISO_TIME_LEN);
           			strcpy(charData, QiEpoch16ToISOString(doubleData, 'T'));
           			free(variable->attribute[nAttr].data);
           			variable->attribute[nAttr].data = charData;
           			variable->attribute[nAttr].data_type = ISO_TIME;
           			variable->attribute[nAttr].num_elems = 1;
				}
         	}
			if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_PLUS") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_DOUBLE){
             		doubleData =  (double *) variable->attribute[nAttr].data;
             		*doubleData *= 1.e-12 ; /* convert ps to s */
           		}
         	}
         	if( strcmp(QiToUpper(variable->attribute[nAttr].name), "DELTA_MINUS") == 0) {
           		if(variable->attribute[nAttr].data_type == CDF_DOUBLE){
             		doubleData =  (double *) variable->attribute[nAttr].data;
             		*doubleData *= 1.e-12 ; /* convert ps to s */
           		}
         	}

      	} /* end for over attributes */
	  } /* end if over numEntries entries */
    } /* end if on CDF_EPOCH16 var type */
    else{
        /* Remove EPOCH from V Attrs only as data not Epoch */

      for( nAttr=0; nAttr < variable->num_v_attrs; nAttr++){

         if( (variable->attribute)[nAttr].data_type == CDF_EPOCH){
		 	
			if (variable->attribute[nAttr].num_elems == 1) {
			
           		doubleData = (double *) variable->attribute[nAttr].data;
				
           		charData = (char*)  QiNewStr(QiEpochToISOString( *doubleData, 'T'));
				
           		free(variable->attribute[nAttr].data);
				
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = ISO_TIME;
           		variable->attribute[nAttr].num_elems = ISO_TIME_LEN;
		   }
		   else if(variable->attribute[nAttr].num_elems == 2){
           		doubleData = (double *) variable->attribute[nAttr].data;
				
				start = QiNewStr(QiEpochToISOString( doubleData[0], 'T'));
				end = QiNewStr(QiEpochToISOString( doubleData[1], 'T'));
				rangeStr = (char *) malloc((2*ISO_TIME_LEN+1 ) * sizeof(char));
				strcpy(rangeStr, start);
				strcat(rangeStr, "/");
				strcat(rangeStr, end);
				free (variable->attribute[nAttr].data);
				free (start);
				free (end); 
				
				variable->attribute[nAttr].data = rangeStr;				
				variable->attribute[nAttr].data_type = ISO_TIME_RANGE;
				variable->attribute[nAttr].num_elems = 2*ISO_TIME_LEN+1;
			}
			else{     
				printf("V Attr %s array of time not handled\n", variable->attribute[nAttr].name);
      			continue;
			}
         }
         else if( (variable->attribute)[nAttr].data_type == CDF_EPOCH16){
		 	
			if (variable->attribute[nAttr].num_elems == 1) {
			
           		doubleData = (double *) variable->attribute[nAttr].data;
				
           		charData = (char*)  QiNewStr(QiEpoch16ToISOString( doubleData, 'T'));
				
           		free(variable->attribute[nAttr].data);
				
           		variable->attribute[nAttr].data = charData;
           		variable->attribute[nAttr].data_type = ISO_TIME;
           		variable->attribute[nAttr].num_elems = ISO_TIME_LEN;
		   }
		   else if(variable->attribute[nAttr].num_elems == 2){
           		doubleData = (double *) variable->attribute[nAttr].data;
				
				start = QiNewStr(QiEpoch16ToISOString( doubleData, 'T'));
				end = QiNewStr(QiEpoch16ToISOString( &(doubleData[2]), 'T'));
				rangeStr = (char *) malloc((2*ISO_TIME_LEN+1 ) * sizeof(char));
				strcpy(rangeStr, start);
				strcat(rangeStr, "/");
				strcat(rangeStr, end);
				free (variable->attribute[nAttr].data);
				free (start);
				free (end); 
				
				variable->attribute[nAttr].data = rangeStr;				
				variable->attribute[nAttr].data_type = ISO_TIME_RANGE;
				variable->attribute[nAttr].num_elems = 2*ISO_TIME_LEN+1;
			}
			else{     
				printf("V Attr %s array of time not handled\n", variable->attribute[nAttr].name);
      			continue;
			}
         }
      }
    }


  } /* end for over variables */

  return QMW_OK;

} /* end   QiRemoveEpoch  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiWriteGlobals
*
* PURPOSE: Write Global Attributes to new cdf file.
*
* DESCRIPTION: Writes all global attribute entries from metadata object into cdf.
*              Existing entries from the skeleton are overwritten if present in
*              metadata object.
*
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED: CDFlib
*
*
* ALGORITHM:  For each global attribute entry in metadata object:
*               If entry present.
*                 Try to select attribute of that name
*                 if unsuccessful create attribute of that name.
*                 For each entry in turn:
*                    select entry number and write entry.
*             Append software version number of this software to Software_version
*
***********************************************************************/

long QiWriteGlobals ( QiCDFContents *QiSCDF)
{

 long check=QMW_OK;
 long n, n_entry;
 long attr_num;
 CDFstatus status;

 /* loop through entries in Global attributes in structure */

 for (n=0; n < QiSCDF->num_g_attrs; n++){

    /* select or create attribute */

    status = CDFlib(SELECT_, ATTR_NAME_, QiSCDF->g_attr[n]->name,
                    NULL_);
    if(status != CDF_OK){
      status = CDFlib(CREATE_, ATTR_, QiSCDF->g_attr[n]->name,
                                      GLOBAL_SCOPE, &attr_num,
                      NULL_);
      if(status != CDF_OK){
        check = NO_GLOBAL_ATTR;
        continue;               /* skip this as we can't select or create it */
      }
    }

    /* write each entry in turn */

    for (n_entry=0; n_entry < QiSCDF->g_attr[n]->num_entries;
         n_entry++){

      /* test if entry present in structure, if not go on to next */

      if(QiSCDF->g_attr[n]->entry[n_entry].exists == 0 || 
  	     QiSCDF->g_attr[n]->entry[n_entry].num_elems == 0) continue;

      status = CDFlib(SELECT_, gENTRY_, n_entry, NULL_);
        if(status != CDF_OK){
          QiStatusHandler(status,"on next global entry");
          check = FAIL_TO_SELECT_ENTRY;
          continue;
        }

      status = CDFlib(PUT_, gENTRY_DATA_,
                      QiSCDF->g_attr[n]->entry[n_entry].data_type,
                      QiSCDF->g_attr[n]->entry[n_entry].num_elems,
                      QiSCDF->g_attr[n]->entry[n_entry].data,
                      NULL_);
      if (status != CDF_OK) {
         check = FAIL_G_ATTR_PUT;
      }

    } /* end for over entries */

 } /* end for over G attributes */

    /* Append the software version of this module */

    status = CDFlib(SELECT_, ATTR_NAME_, "Software_version", NULL_);
    if(status != CDF_OK){
      status = CDFlib(CREATE_, ATTR_, "Software_version",
                                      GLOBAL_SCOPE, &attr_num,
                      NULL_);
      if(status != CDF_OK){
        return NO_GLOBAL_ATTR;
      }
    }

    status = CDFlib(GET_, ATTR_MAXgENTRY_, &n_entry, NULL_);
    if (status != CDF_OK){
      return FAIL_GET_MAX_ENTRY;
    }
    n_entry++;
    status = CDFlib(SELECT_, gENTRY_, n_entry, NULL_);

    status = CDFlib(PUT_, gENTRY_DATA_, CDF_CHAR,
                                        strlen(QIE_VERSION),
                                        QIE_VERSION,
                    NULL_);
    if(status != CDF_OK) return FAIL_VERSION_WRITE;

    return check;

} /* end QiWriteGlobals */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiWriteVariable
*
* PURPOSE: Write Variables and Variable Attributes to new cdf file.
*
* DESCRIPTION: Create variable of appropriate name. Write data. Find matching
*              metadata variable attribute  (same name and data type) and write.
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED:  CDFlib
*                  QiTestMetadata
*                  QiWriteVarAttrs
*
* ALGORITHM: call CDFlib to extend records in cdf to take data.
*            call CDFlib to select intervals and sizes for variable.
*            call CDFlib to write all data via hyperput
*            call QiTestMetadata to locate matching attributes in metadata
*            call QiWriteVarAttrs to write variable attributes
*
***********************************************************************/

long QiWriteVariable ( QiCDFContents *QiSCDF,
                       long n)
{
  long check;
  CDFstatus status;
  long var_num;
  long j;
  long dim_indices[CDF_MAX_DIMS];
  long dim_intervals[CDF_MAX_DIMS];

	/* ensure at least one record exists (create 1 rec of fills) */
	
	QiEnsureVariable(QiSCDF, n);

	
    /* create variable in CDF file */
   
    status = CDFlib(CREATE_, zVAR_, QiSCDF->vardata[n]->name,
                                    QiSCDF->vardata[n]->data_type,
                                    QiSCDF->vardata[n]->num_elems,
                                    QiSCDF->vardata[n]->num_dims,
                                    QiSCDF->vardata[n]->dim_sizes,
                                    QiSCDF->vardata[n]->rec_vary,
                                    QiSCDF->vardata[n]->dim_varys,
                                    &var_num,
                    NULL_);

    QiStatusHandler(status, " on create");


  /* allocate space in cdf */

  status = CDFlib(PUT_, zVAR_EXTENDRECS_, QiSCDF->vardata[n]->max_rec_num + 1,
                  NULL_);

  /* select parameters for hyper write */

  for (j=0; j < QiSCDF->vardata[n]->num_dims; j++){
    dim_indices[j] = 0;
    dim_intervals[j] = 1;
  }

    /* write out data */
  status = CDFlib (SELECT_, zVAR_RECNUMBER_, 0,
                            zVAR_RECCOUNT_, QiSCDF->vardata[n]->max_rec_num + 1,
                            zVAR_RECINTERVAL_, 1,
                            zVAR_DIMCOUNTS_, QiSCDF->vardata[n]->dim_sizes,
                            zVAR_DIMINDICES_, dim_indices,
                            zVAR_DIMINTERVALS_, dim_intervals,
                   NULL_);
  QiStatusHandler(status, " on select");

  if (status != CDF_OK) return CDF_VAR_SELECT_ERR;

  if(QiSCDF->vardata[n]->data == NULL){
    printf("Variable %s data is NULL\n", QiSCDF->vardata[n]->name);
    return QMW_WARNING;
  }
  
   
  status = CDFlib(PUT_, zVAR_HYPERDATA_, QiSCDF->vardata[n]->data, NULL_);
    if(status != CDF_OK) return FAIL_WRITE_DATA;

  /* write variable attributes */

  check = QiWriteVarAttrs(QiSCDF, n);


  return check;


}  /* end QiWriteVariable */


void QiEnsureVariable ( QiCDFContents *QiSCDF,
                       long n)
{
	long dim_indices[CDF_MAX_DIMS];
	long dim_intervals[CDF_MAX_DIMS];
	double * dbl_ptr;
	float * flt_ptr;
	long * long_ptr;
	unsigned long * ulong_ptr;
	short * short_ptr;
	char * char_ptr;
	
	long len = 1;
	int i;
	
	if(QiSCDF->vardata[n]->max_rec_num >= 0) return; /* Data exists, OK to return */
	
	QiSCDF->vardata[n]->max_rec_num = 0;
	if(QiSCDF->n_recs == 0) QiSCDF->n_recs = 1;
	
	/* ensure at least one record exists (create 1 rec of fills) */
	
	for( i=0; i<QiSCDF->vardata[n]->num_dims; i++){
		if(QiSCDF->vardata[n]->dim_varys[i] == VARY) len *= QiSCDF->vardata[n]->dim_sizes[i];
	}
	
	len *= QiSCDF->vardata[n]->num_elems;
	
	switch (QiSCDF->vardata[n]->data_type){
	
		case CDF_EPOCH: case CDF_REAL8: case CDF_DOUBLE:
		{
			dbl_ptr = (double *) malloc(len*sizeof(double));
			for (i=0; i<len; i++){
				dbl_ptr[i] = 0.;
			}
			QiSCDF->vardata[n]->data = dbl_ptr;
			break;
		}
		case CDF_EPOCH16: 
		{
			dbl_ptr = (double *) malloc(2*len*sizeof(double));
			for (i=0; i<len; i++){
				dbl_ptr[i] = 0.;
				dbl_ptr[i+1] = 0.;
			}
			QiSCDF->vardata[n]->data = dbl_ptr;
			break;
		}
		case CDF_REAL4: case CDF_FLOAT:
      	{
			flt_ptr = (float *)malloc(len*sizeof(float));
			for (i=0; i<len; i++){
				flt_ptr[i] = -1.0e31;
			}
			QiSCDF->vardata[n]->data = flt_ptr;
			break;
		}
    	case CDF_INT4: 
      	{
			long_ptr = (long *) malloc(len*sizeof(long));
			for (i=0; i<len; i++){
				long_ptr[i] = (long) -2147483647-1;
			}
			QiSCDF->vardata[n]->data = long_ptr;
			break;
		}
    	case CDF_UINT4:
      	{
			ulong_ptr = (unsigned long*) malloc(len*sizeof(unsigned long));
			for (i=0; i<len; i++){
				ulong_ptr[i] = (unsigned long) 4294967295.;  /* cast from float to get past compiler! */
			}
			QiSCDF->vardata[n]->data = ulong_ptr;
			break;
		}
		case CDF_INT2: 
		{
			short_ptr = (short *) malloc(len*sizeof(short));
			for (i=0; i<len; i++){
				short_ptr[i] = -32768;
			}
			QiSCDF->vardata[n]->data = short_ptr;
			break;
		}
		case CDF_UINT2:
		{
			short_ptr = (short *) malloc(len*sizeof(short));
			for (i=0; i<len; i++){
				short_ptr[i] = 65535;
			}
			QiSCDF->vardata[n]->data = short_ptr;
			break;
		}
		case CDF_INT1: case CDF_BYTE:
		{
			char_ptr = (char *) malloc(len*sizeof(char));
			for (i=0; i<len; i++){
				char_ptr[i] = -128;
			}
			QiSCDF->vardata[n]->data = char_ptr;
			break;
		}
		case CDF_UINT1: 
		{
			char_ptr = (char *) malloc(len*sizeof(char));
			for (i=0; i<len; i++){
				char_ptr[i] = 255;
			}
			QiSCDF->vardata[n]->data = char_ptr;
			break;
		}
		case CDF_CHAR: case CDF_UCHAR:
		{
			/* allow for Null terminator */
			char_ptr = (char *) malloc((len+QiSCDF->vardata[n]->num_elems)*sizeof(char));
			for (i=0; i<len; i+= QiSCDF->vardata[n]->num_elems+1){
				strcpy(&(char_ptr[i]), " ");
			}
			QiSCDF->vardata[n]->data = char_ptr;
			break;
		}
		case ISO_TIME:
		{
			/* NOT USED IN WRITING CDF FILE */
			
			/* allow for Null terminator */
			char_ptr = (char *) malloc((len+QiSCDF->vardata[n]->num_elems)*sizeof(char));
			for (i=0; i<len; i+= QiSCDF->vardata[n]->num_elems+1){
				strcpy(&(char_ptr[i]), "9999-12-31T23:59:59Z");
			}
			QiSCDF->vardata[n]->data = char_ptr;
			break;
		}
		case ISO_TIME_RANGE:
		{
			/* NOT USED IN WRITING CDF FILE */
			
			/* allow for Null terminator */
			char_ptr = (char *) malloc((len+QiSCDF->vardata[n]->num_elems)*sizeof(char));
			for (i=0; i<len; i+= QiSCDF->vardata[n]->num_elems+1){
				strcpy(&(char_ptr[i]), "9999-12-31T23:59:59Z/9999-12-31T23:59:59Z");
			}
			QiSCDF->vardata[n]->data = char_ptr;
			break;
		}
	}
                                    
				
}

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiWriteVarAttrs
*
* PURPOSE: Write out the attributes associated with variable.
*
* DESCRIPTION: writes out all variable attributes associated with current variable
*              in metadata, creating it if not already present in cdf file.
*
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED: CDFlib
*
*
* ALGORITHM: Select V attr entry for variable.
*            For each V attribute entry in metadata at current variable:
*              Select attrbute by name, if not present create it.
*              write entry.
*
***********************************************************************/

 long QiWriteVarAttrs( QiCDFContents *QiSCDF,
                       long var_num)
{
 long n;
 long check = QMW_OK;
 long attr_num;
 CDFstatus status;
 int i;
 long attrStrLen;
 char * concatStrAttr;
 int j_ptr;
 
 /* select entry appropriate for variable */

    status = CDFlib(SELECT_, zENTRY_NAME_,
                          QiSCDF->vardata[var_num]->name,
                   NULL_);
      if(status != CDF_OK) return BAD_VARIABLE_NAME;

 /* loop through entries in var attributes in structure */

 for (n=0; n < QiSCDF->vardata[var_num]->num_v_attrs; n++){

   if(QiSCDF->vardata[var_num]->attribute[n].data != NULL){

     /* select or create attribute */

     status = CDFlib(SELECT_, ATTR_NAME_,
                          QiSCDF->vardata[var_num]->attribute[n].name,
                   NULL_);
     if(status != CDF_OK){
       status = CDFlib(CREATE_, ATTR_,
                          QiSCDF->vardata[var_num]->attribute[n].name,
                                      VARIABLE_SCOPE, &attr_num,
                      NULL_);
       if(status != CDF_OK){
         check = NO_VAR_ATTR;
         continue;              /* skip this as we can't select or create it */
       }
     }

     /* write  entry */

     if(QiSCDF->vardata[var_num]->attribute[n].data_type == CDF_CHAR ||
        QiSCDF->vardata[var_num]->attribute[n].data_type == CDF_UCHAR ){
        char * charData = (char *)QiSCDF->vardata[var_num]->attribute[n].data;

        if(QiSCDF->vardata[var_num]->attribute[n].num_elems > 1) {
          printf( "Should NOT get here: CDF cannot write multiple entries for char type V attribute %s of variable %s\n",
                QiSCDF->vardata[var_num]->attribute[n].name, QiSCDF->vardata[var_num]->name );
        }
        /* num_elems for a Vattr is the number of strings in attr array : Changed for CEF2*/
        attrStrLen=0;
        for ( i=0; i<QiSCDF->vardata[var_num]->attribute[n].num_elems; i++){
          attrStrLen += strlen(&(charData[attrStrLen]));
        }
        /* allow space for each string + 2 spaces per string for "' " This also allows for terminator */
        concatStrAttr = (char *) malloc (sizeof(char)*((attrStrLen+2)*QiSCDF->vardata[var_num]->attribute[n].num_elems));
        j_ptr = 0;
        strcpy(concatStrAttr, &(charData[j_ptr]));
        for( i=1; i<QiSCDF->vardata[var_num]->attribute[n].num_elems; i++){
          j_ptr += strlen(&(charData[j_ptr]))+1;
          strcat(concatStrAttr, ", ");
          strcat(concatStrAttr, &(charData[j_ptr]));
        }
		if(strcmp(concatStrAttr,"") == 0) concatStrAttr[0] = ' '; /* write space rather than null string as CDFlib fails */

        status = CDFlib(PUT_, zENTRY_DATA_,
                        QiSCDF->vardata[var_num]->attribute[n].data_type,
                        (long)strlen(concatStrAttr),
                        (void *)concatStrAttr,
                        NULL_);

        if (status != CDF_OK)  {
          printf( "CDF error writing  char V attribute %s of variable %s\n",
                QiSCDF->vardata[var_num]->attribute[n].name, QiSCDF->vardata[var_num]->name );
          check = FAIL_V_ATTR_PUT;
        }
     }
     else{
        status = CDFlib(PUT_, zENTRY_DATA_,
                        QiSCDF->vardata[var_num]->attribute[n].data_type,
                        QiSCDF->vardata[var_num]->attribute[n].num_elems,
                        QiSCDF->vardata[var_num]->attribute[n].data,
                      NULL_);

       if (status != CDF_OK) {
       printf( "CDF error writing  V attribute %s of variable %s\n",
                QiSCDF->vardata[var_num]->attribute[n].name, QiSCDF->vardata[var_num]->name  );
        check = FAIL_V_ATTR_PUT;
       }

     }

   } /* end if on data not NULL */

 } /* end for over V attributes */

    return check;

} /* end QiWriteVarAttrs */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QIFF
*
* MODULE:  Qie
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiStatusHandler
*
* PURPOSE: Process response status of CDF library function calls.
*
* DESCRIPTION: Determine status returned by CDF library calls. Use returned
*              status to determine nature of warnings/errors and initiate
*              error processing when appropriate.
*
* ARGUMENTS:
*    IN:  CDFstatus  status
*    IN:  char       *note
*
* GLOBALS:  FILE *logfile
*
* RETURN:
*    TYPE:   void
*    VALUES: none
*
* FUNCTIONS_USED:
*         CDFerror
*         QiErrorHandler
*
* ALGORITHM:
*       If status is OK return.
*       Otherwise output note and message returned from CDFerror to log file.
*       Pass warning or error to QiErrorHandler.
*
***********************************************************************/

long QiStatusHandler (CDFstatus status, const char *note)

{
  char message[CDF_STATUSTEXT_LEN+1];
  long check;

  if (status == CDF_OK){
    /* do nothing */
    return QMW_OK;
  }

  /* else print location note then fetch CDF message and print */

  fprintf(stderr,"%s\n", note);

  if (status < CDF_WARN){
     check = QMW_ERROR;
  }
  else
     if (status < CDF_OK){
        fprintf (stderr,"Warning message from CDF...\n");
        check = QMW_WARNING;
     }
     else
     {
        fprintf (stderr, "CDF function executed OK, but...\n");
        check = QMW_OK;
     }

     CDFerror (status, message);
     fprintf (stderr, " CDF> %s\n",message);

     return check;

}  /* end QiStatusHandler */

/**********************************************************************
************************ START OF CDF DATA IMPORT FUNCTIONS ***********
***********************************************************************/

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiGetCSDSgenCDF
*
* PURPOSE: Get data for whole CDF file. Metadata are retrieved separately.
*
* DESCRIPTION: Opens named file and reads each variable in turn.
*              r-variables are coerced into z-variables on the fly.
*
* GLOBALS:
*
* RETURN:
*    TYPE:   int
*    VALUES: QMW_OK         Executed successfully.
*            CDF_OPEN_ERR   Unable to open named file.
*            CDF_N_VAR_ERR  Unable to get number of variables.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

int QiGetCSDSgenCDF(const char *file_name,     /* name of cdf file with path */
                 QiCDFContents *Scdf_data)  /* ptr to struct for data return */

{

  CDFstatus status;
  CDFid id;
  long n_vars, n;
  long max_rec;
  long check;
  long time_var;
  long epoch_num[100];
  int epochCount=0;
  
   /* open named cdf file */

   status = CDFlib(OPEN_, CDF_, file_name, &id, NULL_);
     if (status != CDF_OK) return CDF_OPEN_ERR;

   /* set to z mode to convert r to z variables */

    CDFlib(SELECT_, CDF_zMODE_, zMODEon2, NULL_);

   /* For each variable in file get data */

   status = CDFlib(GET_, CDF_NUMzVARS_, &n_vars,
                         zVARs_MAXREC_, &max_rec,
                   NULL_);
     if (status != CDF_OK) return CDF_N_VAR_ERR;

   /* keep record of number of variables and number of records */

   Scdf_data->n_vars = n_vars;
   Scdf_data->n_recs = max_rec + 1;

   /* allocate memory for variable pointers */

   Scdf_data->vardata = (QiCDFVariable **) QiMakeQiVariablePtrs( MAX_N_VARS);

   for(n=0; n<n_vars; n++){

   	 	status = CDFlib (SELECT_, zVARs_RECNUMBER_, 0, NULL_);
     	if (status != CDF_OK) return CDF_SELECT_ERR;

   		/* allocate memory for variable structure */

   		Scdf_data->vardata[n] = (QiCDFVariable *) QiMakeQiVariable( );

     	check = QiGetCDFVar( Scdf_data->vardata[n], n, &time_var);

     	if(check != CDF_OK) {
        	Scdf_data->vardata[n] = NULL; /* lose partially filled object */
     	}
     	else{
       		if ( time_var == KEEP ){
				if (Scdf_data->d0_var_num == -1) Scdf_data->d0_var_num = n;
				if(epochCount < 99) {
					epoch_num[epochCount] = n;
					epochCount++;
				}
			}
		}
	}

   /* retrieve metadata */

   check = QiGetCDFmetadata(Scdf_data );
     if(check != QMW_OK) return check;


   status = CDFlib(CLOSE_, CDF_, NULL_);
     if (status != CDF_OK) return QMW_WARNING;

   // In case this is a Themis file trap virtual DEPEND_O epoch and calculate true epoch
   QiFixThemisEpoch( Scdf_data, &(epoch_num[0]), epochCount);
   
   return QMW_OK;


} /* end QiGetCSDSgenCDF */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiGetCDFmetadata
*
* PURPOSE: Fills metadata into QiCDFContents structure.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK         Executed successfully.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

long QiGetCDFmetadata (QiCDFContents *Scdf_data)  /* ptr to struct for data return */

{
long num_attrs;
long attr_scope;
long i, j;
int m;

  /* find number of G attrs and malloc pointers */

  if( CDFlib( GET_, CDF_NUMgATTRS_, &(Scdf_data->num_g_attrs), NULL_) != CDF_OK )
    return FAIL_ON_CDF_READ;

  Scdf_data->g_attr =
          (QiGlobalAttribute **) QiMakeQiGAttrPtrs( Scdf_data->num_g_attrs);

  /* loop through all attributes and get G attrs */

  if( CDFlib( GET_, CDF_NUMATTRS_, &num_attrs, NULL_) != CDF_OK )
    return FAIL_ON_CDF_READ;

  j=0;
  for( i=0 ; i < num_attrs ; i++ )
  {
    if( CDFlib( SELECT_, ATTR_, i, NULL_) != CDF_OK )
      return FAIL_ON_SELECT_ATTR;

    if( CDFlib( GET_, ATTR_SCOPE_,  &attr_scope, NULL_) != CDF_OK )
      return FAIL_ON_CDF_READ;

    if( attr_scope == GLOBAL_SCOPE )
    {

    Scdf_data->g_attr[j] = (QiGlobalAttribute *) QiMakeQiGAttr();

      if( QiGetGlobalAttr( j, Scdf_data->g_attr[j] ) == QMW_OK )
        j++;
      else{
        /* free up attribute allocation and re-use */
         if(Scdf_data->g_attr[j] != NULL) {
           if(!QistrNULL( Scdf_data->g_attr[j]->name) )
             free(Scdf_data->g_attr[j]->name);
           if( Scdf_data->g_attr[j]->entry != NULL) {
             for ( m=0; m<Scdf_data->g_attr[j]->num_entries; m++){
               if(( Scdf_data->g_attr[j]->entry+m)->data != NULL)
                 free((Scdf_data->g_attr[j]->entry+m)->data );
             }
             free(Scdf_data->g_attr[j]->entry);
           }

           free(Scdf_data->g_attr[j]);
         }
       } /* end else on global read OK */

    }   /* else variable scope and ignore here */

  } /* end loop over attributes */

  /* correct num G attrs for number read OK */

  Scdf_data->num_g_attrs = j;

  return QMW_OK;

}  /* end QiGetCDFmetadata */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiGetGlobalAttr
*
* PURPOSE: Read in global attribute entries for attribute with specified number.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   CDFstatus
*    VALUES: QMW_OK         Executed successfully.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

CDFstatus QiGetGlobalAttr( long attr_num,
                           QiGlobalAttribute *g_attr )
{

char attribute_name[ CDF_ATTR_NAME_LEN +1 ];
CDFstatus status;

long i;
long check;

g_attr->number = attr_num;

if(  CDFlib( GET_,
             ATTR_NAME_, attribute_name,
             ATTR_MAXgENTRY_, &(g_attr->num_entries),
             NULL_ )!= CDF_OK ){
     QieAlertBox("warning","Failed to get name for attribute");
     return FAIL_ON_G_ATTR_READ;
 }

/* actually what comes back is the maximum gEntry number, but we want
** to use the number of gEntries _including_ any gEntries which
** have no data (ie ones which "don't exist")
** This is why we do not use ATTR_NUMgENTRIES_
**
** therefore increment ..
**/

g_attr->num_entries++;

if(g_attr->num_entries <= 0) return QMW_ERROR;

g_attr->name = (char *) malloc(strlen(attribute_name) +1);
strcpy(g_attr->name, attribute_name);

g_attr->entry = ( QiGAttrEntry *) QiMakeEntries(g_attr->num_entries);

for( i = 0; i < g_attr->num_entries; i++ )
{
  if( CDFlib( CONFIRM_, gENTRY_EXISTENCE_, i, NULL_) == CDF_OK )
  {
    g_attr->entry[i].exists = 1;
    status = CDFlib( SELECT_,
                       gENTRY_, i,
                     GET_,
                       gENTRY_DATATYPE_, &(g_attr->entry[i].data_type),
                       gENTRY_NUMELEMS_, &(g_attr->entry[i].num_elems),
                     NULL_ );

    if( status != CDF_OK ){
      QieAlertBox(g_attr->name, "- Failed to get entry info for attribute");
      return FAIL_ON_G_ATTR_READ;
    }

    g_attr->entry[i].data = QiAttrEntryMalloc( g_attr->entry[i].num_elems,
                              g_attr->entry[i].data_type );

    status = CDFlib( GET_, gENTRY_DATA_, g_attr->entry[i].data, NULL_ );


  } else
  {
    g_attr->entry[i].exists = 0;
  }

} /* end loop over entries */

check = QiStatusHandler(status,g_attr->name);
return check;

}  /* end QiGetGlobalAttr */
/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiGetCDFVar
*
* PURPOSE: Fills data into QiCDFVariable structure.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK         Executed successfully.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

long QiGetCDFVar(QiCDFVariable *vardata, /* ptr to var data struct */
                long number,             /* var number in cdf & struct array */
                long * time_flag)        /* flag for time line id */
{
  CDFstatus status;
  char name[CDF_VAR_NAME_LEN+1];
  size_t length;
  int size_of_datatype;
  long dim_indices[CDF_MAX_DIMS];
  long dim_counts[CDF_MAX_DIMS];
  long n;

  *time_flag = IGNORE_TIME;

  /* select variable */

  status = CDFlib(SELECT_, zVAR_, number,
                           zENTRY_, number,
                           NULL_);
  if (status != CDF_OK) return CDF_VAR_N_ERR;


  /* fill in info on variable */

  vardata->number = number;

  status = CDFlib(GET_, zVAR_NAME_, &name[0], NULL_);
  if (status != CDF_OK) return CDF_VAR_NAME_ERR;
  vardata->name = (char *) malloc( sizeof(char)*(strlen(name)+1) );
  strcpy(vardata->name, name);

  status = CDFlib(GET_, zVAR_DATATYPE_, &(vardata->data_type), NULL_);
  if (status != CDF_OK) return CDF_VAR_TYPE_ERR;

  status = CDFlib(GET_, zVAR_RECVARY_, &(vardata->rec_vary), NULL_);
  if (status != CDF_OK) return CDF_VAR_RVARY_ERR;

  status = CDFlib(GET_, zVAR_NUMELEMS_, &(vardata->num_elems), NULL_);
  if (status != CDF_OK) return CDF_VAR_ELEMS_ERR;

  status = CDFlib(GET_, zVAR_NUMDIMS_, &(vardata->num_dims), NULL_);
  if (status != CDF_OK) return CDF_VAR_DIMS_ERR;

  status = CDFlib(GET_, zVAR_MAXREC_, &(vardata->max_rec_num), NULL_);
  if (status != CDF_OK) return CDF_VAR_DIMS_ERR;

  if (vardata->max_rec_num < 0) {
     vardata->max_rec_num = 0;
     vardata->rec_vary = NOVARY;
     /* -1 returned for virtual records! */
    QieAlertBox( &name[0], ": variable is virtual");
  }

  if (vardata->num_dims > 0 ){

    status = CDFlib(GET_, zVAR_DIMSIZES_, &(vardata->dim_sizes[0]), NULL_);
    if (status != CDF_OK) return CDF_VAR_SIZES_ERR;

    status = CDFlib(GET_, zVAR_DIMVARYS_, &(vardata->dim_varys[0]), NULL_);
    if (status != CDF_OK) return CDF_VAR_DVARY_ERR;

  }

  /* Allocate memory for data and indices and counts for reading */

  switch(vardata->data_type){
    case CDF_CHAR:
    case CDF_UCHAR:
    case CDF_UINT1:
    case CDF_INT1:
    case CDF_BYTE:
    {
      size_of_datatype = 1;
      break;
    }
    case CDF_INT2:
    case CDF_UINT2:
    {
      size_of_datatype = 2;
      break;
    }
    case CDF_INT4:
    case CDF_UINT4:
    case CDF_REAL4:
    case CDF_FLOAT:
    {
      size_of_datatype = 4;
      break;
    }
    case CDF_REAL8:
    case CDF_DOUBLE:
    {
      size_of_datatype = 8;
      break;
    }
    case CDF_EPOCH:
    {
      size_of_datatype = 8;
      *time_flag = KEEP;
      break;
    }
    case CDF_EPOCH16:
    {
      size_of_datatype = 16;
      *time_flag = KEEP;
      break;
    }
    default:
    {
    return BAD_CDF_DATATYPE;
    }
  }

  length = (size_t) vardata->num_elems * (vardata->max_rec_num +1) * size_of_datatype;

  for (n=0; n < vardata->num_dims; n++){

    dim_indices[n] = 0;

    if(vardata->dim_varys[n] == VARY){
      length = length * vardata->dim_sizes[n];
      dim_counts[n] = vardata->dim_sizes[n];
    }
    else{
      dim_counts[n] = 1;
    }
  }

  vardata->data = (void *) malloc (length);
  

  /* get data array in same ordering as cdf file */

  status = CDFlib (SELECT_, zVAR_DIMCOUNTS_, dim_counts,
                            zVAR_DIMINDICES_, dim_indices,
                            zVAR_RECCOUNT_, vardata->max_rec_num + 1,
                   NULL_);
  if (status < CDF_WARN) {
     fprintf(stderr,"CDFlib err code %ld", (long)status);
     return CDF_VAR_SELECT_ERR;
  }

  status = CDFlib (GET_, zVAR_HYPERDATA_, vardata->data, NULL_);
  if (status < CDF_WARN)  {
     fprintf(stderr, "CDFlib err code %ld", status);
     return CDF_VAR_GET_DATA_ERR;
  }

  /* get V attrs */

  if( QiGetVariableAttr( number, vardata ) != QMW_OK )  return FAIL_ON_V_ATTR_READ;

  return QMW_OK;

} /* end QiGetCDFVar */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiAttrEntryMalloc
*
* PURPOSE: Allocate memory for attribute entry
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   void *
*    VALUES: pointer to new memory allocated for entry.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

void *QiAttrEntryMalloc( long num_elems, long data_type )
{
void *p;
size_t size;

if( num_elems < 1 ) num_elems = 1;   /* some defensive programming ... */

switch (data_type) {
  case CDF_BYTE: case CDF_INT1: case CDF_UINT1:
  {
    size = num_elems;
    break;
  }
  case CDF_INT2: case CDF_UINT2:
  {
    size = 2 * num_elems;
    break;
  }
  case CDF_INT4: case CDF_UINT4: case CDF_REAL4: case CDF_FLOAT:
  {
    size = 4 * num_elems;
    break;
  }

  case CDF_REAL8:  case CDF_EPOCH: case CDF_DOUBLE:
  {
    size = 8 * num_elems;
    break;
  }

  case CDF_EPOCH16:
  {
    size = 16 * num_elems;
    break;
  }

  case CDF_CHAR: case CDF_UCHAR:
  {
    size = num_elems + 1;
    break;
  }
  default: size = 8 * num_elems;
}

p = (void *)malloc( size );

if( data_type == CDF_CHAR || data_type == CDF_UCHAR )
  ((char *)p)[num_elems] = '\0' ;

return p;

}   /* end QiAttrEntryMalloc  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiGetVariableAttr
*
* PURPOSE: Read all variable attribute entries for specified variable
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED:
*
*
*
* ALGORITHM:
*
*
***********************************************************************/

long QiGetVariableAttr( long var_num,
                        QiCDFVariable * var)
{

CDFstatus status;

long i;
long max_num_attrs;
long attr_num;
long attr_scope;
char attribute_name[ CDF_ATTR_NAME_LEN +1 ];


  if(CDFlib( GET_, CDF_NUMATTRS_, &max_num_attrs, NULL_ ) != CDF_OK )
    return FAIL_ON_SELECT_ATTR;


 /* first  test we have enough space */
  if(max_num_attrs > MAX_N_ATTRS) max_num_attrs = MAX_N_ATTRS;

  /* count entries as we go, zENTRY_ already selected by calling routine */

  attr_num = 0;

 /* read attribute entries: if we fail, try next in list */

for( i=0 ; i < max_num_attrs ; i++ )
{
  if( CDFlib( SELECT_, ATTR_, i, NULL_) != CDF_OK ) continue;

  if( CDFlib( GET_, ATTR_SCOPE_,  &attr_scope, NULL_) != CDF_OK ) continue;

  if( CDFlib( CONFIRM_, zENTRY_EXISTENCE_, var_num, NULL_) != CDF_OK ) continue;

  if( attr_scope == VARIABLE_SCOPE ){

    if( CDFlib( GET_, ATTR_NAME_, attribute_name, NULL_ )
        != CDF_OK ) continue;

    var->attribute[attr_num].number = attr_num;
    var->attribute[attr_num].name = (char *) malloc(strlen(attribute_name)+1);
    strcpy(var->attribute[attr_num].name, attribute_name);

    if( CDFlib( GET_,
                zENTRY_DATATYPE_, &(var->attribute[attr_num].data_type),
                zENTRY_NUMELEMS_, &(var->attribute[attr_num].num_elems),
                NULL_ )
       != CDF_OK ) continue;


/* allocate entry space */

      var->attribute[attr_num].data =
        QiAttrEntryMalloc( var->attribute[attr_num].num_elems,
                           var->attribute[attr_num].data_type );

      if( ( status = CDFlib(
                  GET_,  zENTRY_DATA_, var->attribute[attr_num].data,
                  NULL_ ) )
          != CDF_OK ) continue;

       /* Change for CEF 2 handling, char Vattrs have num_elems = number of strings, not strlen */
       if(var->attribute[attr_num].data_type == CDF_CHAR ||
          var->attribute[attr_num].data_type == CDF_UCHAR )   var->attribute[attr_num].num_elems = 1;

      /* Note NULL terminator for char is already handled by  QiAttrEntryMalloc */

       /* if we get here the attribute is read OK, so increment counter */

       attr_num++;


  }  /* end if on VARIABLE_SCOPE */

} /* end for loop over attributes */

  var->num_v_attrs = attr_num;

  return QMW_OK;

} /* end QiGetVariableAttr */
