/***********************************************************************
*           See Accompanying licence file QSAS_licence                 *
***********************************************************************/
/* With adjustments for _WIN32 and __VMS compilers -- Patrick Daly & Rob Wilson*/

#include "qie_ft.h"
#include "qie.h"
#include "qie_local.h"

/* #define DEBUG_ME */  /* defining DEBUG_ME will show msecs accumulating */

static char end_of_line_mark;
static char QicMark; /* comment marker */
static long year_offset;
static long monotonic;
static long yp_last;
static FILE *includedFiles[MAX_N_INC_FILES];
static int includedFrom[MAX_N_INC_FILES];
static fpos_t includedFromPosn[MAX_N_INC_FILES];
static char *fileNames[MAX_N_INC_FILES];
static char * incReadPath;
static int fpInc;
static int fpNow;
static int isCAA;

/*********************************************************************/
/**************** START OF FLAT FILE WRITING ROUTINES ****************/
/*********************************************************************/


long QiWriteCSDSgenFlat (QiCDFContents *QiSCDF, QiOptions *QiSOpt)
{

   long check = QMW_OK;
   long n, j;
   FILE *fp;
   FILE *fp_hdr;
   QiRecord_format *QiSfmt;
   char message[LINE_LEN];
   long ns[MAX_N_VARS]; /* n sorted to put time tags first*/

   QiSfmt = (QiRecord_format *) QiMakeFmtObj();

  /* test input data structure to ensure it is useable */

  if(QiSOpt->f_type != CAA) QiRemoveISO_TIME(QiSCDF);

  /* ensure file name and path exist */
  QiEnsureFileName(QiSCDF, QiSOpt);

  /* remove any bad variables */
  if( QiMakeSafe(QiSCDF) == NULL) {
     QieAlertBox("Error", "A variable or data is missing in structure after file read");
     return BAD_STRUCT;
  }

  check = QiTestStructure(QiSCDF);
  if (check != QMW_OK) {
     QieAlertBox("Error", "Sparse records. Some data is not being found");
     return BAD_STRUCT;
  }

  switch (QiSOpt->f_type){

  case TABULAR: case UNSET:  /* unknown defaults to tabular */
    {
      QiSCDF->io_f_extn = QiSOpt->EXTN_QFT;
      QiSOpt->f_type = TABULAR;
      break;
    }
  case DELIMITED:
    {
      QiSCDF->io_f_extn = QiSOpt->EXTN_QFD;
      break;
    }

  case EXCHANGE:
    {
      QiSCDF->io_f_extn = QiSOpt->EXTN_CEF;
      break;
    }

  case CAA:
    {
      QiSCDF->io_f_extn = QiSOpt->EXTN_CEF;
      QiSOpt->time_sep_data = 'T';
      QiSOpt->time_sep_attr = 'T';
      QiRemoveEpoch(QiSCDF); /* remove CDF_EPOCH data type and replace with ISO_TIME */
      break;
    }

  default:
    {
      return QIE_FILE_TYPE_UNKNOWN;
    }
  }

     /* open ascii data file */

  if(QiSOpt->priority == REPLACE){
    fp = QiOpenFile((char*)"wb", QiSCDF->io_f_path, QiSCDF->io_f_name, QiSCDF->io_f_extn);
    if (fp == NULL) return CANNOT_OPEN_FILE;
  }
  else{
    /* test file existence first */
    fp = QiOpenFile((char*)"wb-", QiSCDF->io_f_path, QiSCDF->io_f_name, QiSCDF->io_f_extn);
    if (fp == NULL) return FILE_EXISTS;
  }


  /* set header file pointer */

  switch (QiSOpt->header){

  case NO_HEADER:
    {
      fp_hdr = QiSOpt->fp_null;
      break;
    }
  case ATTACHED:
    {
      fp_hdr = fp;
      break;
    }
  case DETACHED:
    {
       QiEnsureHeader(QiSCDF, QiSOpt);

       if(QiSOpt->priority == REPLACE){
         fp_hdr = QiOpenFile((char*)"wb", QiSOpt->header_path, QiSOpt->header_name,
                             QiSOpt->EXTN_QHD);
       }
       else{
         /* test file existence first */
         fp_hdr = QiOpenFile((char*)"wb-", QiSOpt->header_path, QiSOpt->header_name,
                             QiSOpt->EXTN_QHD);
       }
       if (fp_hdr == NULL) return CANNOT_OPEN_FILE;
       break;
     }
  default:
    {
      QiDisplayMessage("QIE Warning: unknown option no header written", QiSOpt);
      fp_hdr = QiSOpt->fp_null;
    }
  }

   /* write Header, note fp_hdr -> /dev/null if NO_HEADER set */

   check = WriteHeaderFlat(QiSCDF, fp_hdr, QiSOpt);
   if (check != QMW_OK) return check;

   /* initialise record format object */

   QiSfmt->col_start = (int *)malloc(sizeof(int)*QiSCDF->n_vars);
   QiSfmt->col_width = (int *)malloc(sizeof(int)*QiSCDF->n_vars);
   QiSfmt->n_items = (long *)malloc(sizeof(long)*QiSCDF->n_vars);

   if(QiSOpt->rec_numbering == NUM_ON){
     QiSfmt->total_width = REC_N_WIDTH ;
   }
   else{
     QiSfmt->total_width = 0;
   }

  if(QiSOpt->f_type == EXCHANGE || QiSOpt->f_type == CAA){

    /* for cef ensure time var is first */
    j=0;
    if(QiSCDF->d0_var_num > -1) {
      ns[0] = QiSCDF->d0_var_num;
      j++;
    }
    for(n=0; n < QiSCDF->n_vars; n++){
      /* add rest of variables */
      if(n != QiSCDF->d0_var_num){
        ns[j] = n;
        j++;
      }
    }
  }

   /* Write Variable metadata for each variable IN ORDER */

   for(n=0; n < QiSCDF->n_vars; n++){

     /* write variable identifier object */

     if(QiSOpt->f_type == EXCHANGE || QiSOpt->f_type == CAA){
       check = QiWriteVarObjects(QiSCDF, ns[n], QiSOpt, QiSfmt, fp_hdr);
       if(check != QMW_OK) return check;
     }
     else{
       check = QiWriteVarObjects(QiSCDF, n, QiSOpt, QiSfmt, fp_hdr);
       if(check != QMW_OK) return check;
     }
   }

 /* write records */

    switch (QiSOpt->f_type) {
      case TABULAR:
      {
         check = QiWriteRecsTabular(QiSCDF, QiSfmt, QiSOpt, fp);
         break;
      }
      case DELIMITED:
      {
         check = QiWriteRecsParsed(QiSCDF, QiSfmt, QiSOpt, fp);
         break;
      }
      case EXCHANGE:
      {
         check = QiWriteRecsExchange(QiSCDF, QiSfmt, QiSOpt, fp);
         break;
      }
      case CAA:
      {
         check = QiWriteRecsCAA(QiSCDF, QiSfmt, QiSOpt, fp);
         break;
      }
      default:
      {
         sprintf(message, "Unknown file type '%c'", QiSOpt->f_type);
         QiDisplayMessage(message, QiSOpt);
         check = QIE_FILE_TYPE_UNKNOWN;
         break;
      }
     } /* end switch */

  /* free up formatting space */

   QiSfmt = QiFreeFormatSpace(QiSfmt);

  /* close files and exit cleanly */

  /* keep header even if write failed, for diagnostic */
  if(fp != fp_hdr) QiCloseFile(QMW_OK, QiSCDF->io_f_path, QiSCDF->io_f_name, QiSOpt->EXTN_QHD, fp_hdr);

  QiCloseFile(check, QiSCDF->io_f_path, QiSCDF->io_f_name, QiSCDF->io_f_extn, fp);


   return check;

} /* end  QiWriteCSDSgenFlat  */

/*********************************************************************/

long WriteHeaderFlat(QiCDFContents *QiSCDF,   /* ptr to struct of contents */
                     FILE *fp,                /* ptr to output file */
                     QiOptions *QiSOpt)        /* io write options */
{

  long n;


  switch(QiSOpt->f_type){
  case TABULAR:
    {
  /* magic number */
  fprintf(fp, "!@%s@%s\n", QiSCDF->io_f_extn, QieVersion());
  fprintf(fp, "!------------------- QSAS ASCII File -------------------|\n");
  fprintf(fp, "! Fixed column width table padded with CSDS fill values |\n");
  fprintf(fp, "! Column position in record counted from 0              |\n");
  fprintf(fp, "!                                                       |\n");
  fprintf(fp, "! ASCII Format                                          |\n");
  fprintf(fp, "! Native C ordering, last index varies fastest          |\n");
  fprintf(fp, "! Blank lines are ignored                               |\n");
  fprintf(fp, "! \"!\" escapes rest of line as comment                   |\n");
  fprintf(fp, "!-------------------------------------------------------|\n");
  fprintf(fp, "File_type = t\n");
  break;
    }
  case DELIMITED:
    {
  /* magic number */
  fprintf(fp, "!@%s@%s\n", QiSCDF->io_f_extn, QieVersion());
  fprintf(fp, "!------------------- QSAS ASCII File -------------------|\n");
  fprintf(fp, "! Free width data entries separated by delimiters       |\n");
  fprintf(fp, "!                                                       |\n");
  fprintf(fp, "! ASCII Format                                          |\n");
  fprintf(fp, "! Native C ordering, last index varies fastest          |\n");
  fprintf(fp, "! Blank lines are ignored                               |\n");
  fprintf(fp, "! \"!\" escapes rest of line as comment                   |\n");
  fprintf(fp, "!-------------------------------------------------------|\n");
  fprintf(fp, "File_type = d\n");
  break;
    }
  case EXCHANGE:
    {
  /* magic number */
  fprintf(fp, "!@%s1@%s\n", QiSCDF->io_f_extn, QieVersion());
  fprintf(fp, "!--------------- Cluster Exchange ASCII File -----------|\n");
  fprintf(fp, "! Free width data entries separated by delimiters       |\n");
  fprintf(fp, "!                                                       |\n");
  fprintf(fp, "! ASCII Format                                          |\n");
  fprintf(fp, "! Native C ordering, last index varies fastest          |\n");
  fprintf(fp, "! Blank lines are ignored                               |\n");
  fprintf(fp, "! \"!\" escapes rest of line as comment                   |\n");
  fprintf(fp, "!-------------------------------------------------------|\n");
  break;
    }
  case CAA:
    {
  /* magic number */
  fprintf(fp, "!@%s2@%s\n", QiSCDF->io_f_extn, QieVersion());
  fprintf(fp, "!-------- Cluster Exchange Version 2 ASCII File --------|\n");
  fprintf(fp, "! Free width data entries separated by delimiters       |\n");
  fprintf(fp, "!                                                       |\n");
  fprintf(fp, "! ASCII Format                                          |\n");
  fprintf(fp, "! Native C ordering, last index varies fastest          |\n");
  fprintf(fp, "! Blank lines are ignored                               |\n");
  fprintf(fp, "! \"!\" escapes rest of line as comment                   |\n");
  fprintf(fp, "!-------------------------------------------------------|\n!\n");
  break;
    }
  default:
  return QIE_FILE_TYPE_UNKNOWN;
  }

  if(QiSOpt->header == ATTACHED){
    fprintf(fp, "FILE_NAME = %s%s\n!\n", QiSCDF->io_f_name, QiSCDF->io_f_extn);
  }
  else{
    fprintf(fp, "FILE_NAME = %s%s\n!\n", QiSOpt->header_name, QiSOpt->EXTN_QHD);
  }

  /* ensure that space is not used in time field if delim is space */
  if(QiSOpt->attr_delim == ' ') QiSOpt->time_sep_attr = 'T';

  if(QiSOpt->f_type != CAA) {
    /* CAA CEF 2 files are always comma delimited */
    fprintf(fp,"Attribute_delimiter = " );
    QiPutDelim(fp, QiSOpt->attr_delim);fprintf(fp, "\n");
  }
  else {
    fprintf(fp,"FILE_FORMAT_VERSION = " ); fprintf(fp,"%s\n!\n", CAA_VERSION );
  }

  if(QiSOpt->f_type == DELIMITED || QiSOpt->f_type == EXCHANGE) {
    /* NOTE CAA (CEF 2) files are always comma delimited */
      fprintf(fp,"Data_delimiter = " );
      QiPutDelim(fp, QiSOpt->data_delim);fprintf(fp, "\n");
  }

  if( QiSOpt->f_type == EXCHANGE  ) {
     if(QiSOpt->rec_end != '\n'){
        fprintf(fp,"End_of_record_marker = " );
        QiPutDelim(fp, QiSOpt->rec_end);fprintf(fp, "\n");
     }
  }

  if( QiSOpt->f_type == CAA ) {
  	if( QiSOpt->force_eor_write == 1 || QiSOpt->rec_end != '\n'){
		fprintf(fp,"End_of_record_marker = \"" );
		QiPutDelim(fp, QiSOpt->rec_end);fprintf(fp, "\"\n");
	}
  }

  if( QiSOpt->f_type != CAA){
    if(QiSOpt->rec_numbering == NUM_ON ){
      fprintf(fp, "Record_numbering = on\n");
    }
    else
    {
      fprintf(fp, "Record_numbering = off\n");
    }
  }
  fprintf(fp,"!\n");

  /* write global attributes in turn if requested*/

  if(QiSCDF->g_attr != NULL){

    for (n=0; n<QiSCDF->num_g_attrs; n++){
      QiWriteGlobalFlat(QiSCDF, QiSOpt, n, fp);
    }

  } /* end if on g_attr exists */

  return QMW_OK;


} /* end WriteHeaderFlat */

/***********************************************************************/

long QiWriteVarObjects(QiCDFContents *QiSCDF,
                       long var_num,
                       QiOptions *QiSOpt,
                       QiRecord_format *QiSfmt,
                       FILE *fp)
{
 long m;
 char size_str[LINE_LEN];
 char str[LINE_LEN];

    /* find number of items for each var in record */

   QiSfmt->n_items[var_num] = 1;
   if(QiSCDF->vardata[var_num]->num_dims > 0 )
   {
     QiStrcpy(size_str,"SIZES = ", LINE_LEN);

     sprintf(str," %ld",QiSCDF->vardata[var_num]->dim_sizes[0]);
     QiSfmt->n_items[var_num] = QiSfmt->n_items[var_num] *
                                  QiSCDF->vardata[var_num]->dim_sizes[0];
     strcat(size_str,str);

     for (m=1; m<QiSCDF->vardata[var_num]->num_dims; m++){
       sprintf(str,"%c %ld", QiSOpt->attr_delim, QiSCDF->vardata[var_num]->dim_sizes[m]);
       QiSfmt->n_items[var_num] = QiSfmt->n_items[var_num] *
                                  QiSCDF->vardata[var_num]->dim_sizes[m];
       strcat(size_str,str);
     }
   } /* end if on num dimensions  */
   else{
     strcpy(size_str, "");
   }

   /* find column starting points and widths */

   QiSfmt->col_width[var_num] = (long)
      ( (long) strlen(QiSCDF->vardata[var_num]->name)
        / QiSfmt->n_items[var_num] ) +1 ;

   switch(QiSCDF->vardata[var_num]->data_type)
   {
    case CDF_REAL4: case CDF_FLOAT:
     {
     if(QiSfmt->col_width[var_num] < REAL_WIDTH )
           QiSfmt->col_width[var_num] = REAL_WIDTH;
     break;
     }
    case CDF_DOUBLE: case CDF_REAL8:
     {
     if(QiSfmt->col_width[var_num] < DOUBLE_WIDTH )
           QiSfmt->col_width[var_num] = DOUBLE_WIDTH;
     break;
     }
    case CDF_EPOCH: case CDF_EPOCH16:
     {
     if(QiSfmt->col_width[var_num] < EPOCH_WIDTH )
           QiSfmt->col_width[var_num] = EPOCH_WIDTH;
     break;
     }
    case CDF_UINT1: case CDF_INT1: case CDF_BYTE:
     {
     if(QiSfmt->col_width[var_num] < BYTE_WIDTH )
           QiSfmt->col_width[var_num] = BYTE_WIDTH;
     break;
     }
    case CDF_CHAR: case CDF_UCHAR:
     {
     if(QiSfmt->col_width[var_num] < QiSCDF->vardata[var_num]->num_elems +1 )
           QiSfmt->col_width[var_num] = QiSCDF->vardata[var_num]->num_elems +1;
     break;
     }
    case ISO_TIME:
     {
     if(QiSfmt->col_width[var_num] < QiSCDF->vardata[var_num]->num_elems +1 )
           QiSfmt->col_width[var_num] = QiSCDF->vardata[var_num]->num_elems +1;
     break;
     }
    case ISO_TIME_RANGE:
     {
     if(QiSfmt->col_width[var_num] < QiSCDF->vardata[var_num]->num_elems +1 )
           QiSfmt->col_width[var_num] = QiSCDF->vardata[var_num]->num_elems +1;
     break;
     }
    case CDF_INT2: case CDF_UINT2:
     {
     if(QiSfmt->col_width[var_num] < SHORT_WIDTH )
           QiSfmt->col_width[var_num] = SHORT_WIDTH;
     break;
     }
     case CDF_INT4: case CDF_UINT4:
     {
     if(QiSfmt->col_width[var_num] < LONG_WIDTH )
           QiSfmt->col_width[var_num] = LONG_WIDTH;
     break;
     }
    default:
    {
      return BAD_CDF_DATATYPE;
    }
   }

   /* column starts where last one left off (note we started at 0) */

   QiSfmt->col_start[var_num] = QiSfmt->total_width;



   /* write metadata object */

   fprintf(fp,"START_VARIABLE = %s\n", QiSCDF->vardata[var_num]->name);

     if(QiSCDF->vardata[var_num]->rec_vary == NOVARY &&
        QiSCDF->vardata[var_num]->novary_opt == WRITE_ONCE){
          /* skip it */
     }
     else{

       /* increment total */

       QiSfmt->total_width += QiSfmt->col_width[var_num] * QiSfmt->n_items[var_num];
       if(QiSOpt->f_type == TABULAR){
         fprintf(fp,"Column_start = %d\n", QiSfmt->col_start[var_num]);
         fprintf(fp,"Column_width = %d\n", QiSfmt->col_width[var_num]);
       }
     }

     if(strcmp(size_str, "") != 0)  fprintf(fp,"%s\n", size_str);

     switch (QiSCDF->vardata[var_num]->data_type)
     {
        case CDF_DOUBLE: case CDF_REAL8:
        {
          fprintf(fp,"VALUE_TYPE = DOUBLE\n");
          break;
        }
        case ISO_TIME:
        {
          fprintf(fp,"VALUE_TYPE = ISO_TIME\n");
          break;
        }
        case ISO_TIME_RANGE:
        {
          fprintf(fp,"VALUE_TYPE = ISO_TIME_RANGE\n");
          break;
        }
        case CDF_EPOCH: case CDF_EPOCH16:
        {
        if (QiSOpt->f_type == CAA ){
            fprintf(fp,"VALUE_TYPE = ISO_TIME\n");      }
        else{
            fprintf(fp,"VALUE_TYPE = EPOCH\n");
            fprintf(fp,"TIME_FORMAT = ISO\n");
        }
          break;
        }
        case CDF_FLOAT: case CDF_REAL4:
        {
          fprintf(fp,"VALUE_TYPE = FLOAT\n");
          break;
        }
        case CDF_INT4: case CDF_UINT4:
        {
          fprintf(fp,"VALUE_TYPE = INT\n");
          break;
        }
        case CDF_CHAR: case CDF_UCHAR:
        {
          fprintf(fp,"VALUE_TYPE = CHAR\n");
        if(QiSOpt->f_type != CAA ){
            fprintf(fp,"MAXSTRLEN = %ld\n", QiSCDF->vardata[var_num]->num_elems);
        }
          break;
        }
        case CDF_UINT1: case CDF_INT1: case CDF_BYTE:
        {
          fprintf(fp,"VALUE_TYPE = BYTE\n");
          break;
        }
        case CDF_UINT2: case CDF_INT2:
        {
          fprintf(fp,"VALUE_TYPE = INT\n");
          break;
        }

     }

     /* add data for rec_vary == NOVARY  */
     if(QiSCDF->vardata[var_num]->rec_vary == NOVARY  &&
        QiSCDF->vardata[var_num]->novary_opt == WRITE_ONCE){
          fprintf(fp,"DATA = ");
          QiWriteVarDataNRV(fp, QiSCDF->vardata, var_num,
                             QiSfmt->n_items[var_num],
                       0, 0, QiSOpt, QiSOpt->attr_delim);
          fprintf(fp,"\n");
     }

     if (QiSCDF->vardata[var_num]->attribute != NULL){
        QiAttrs_to_Text(QiSCDF, var_num, fp, QiSOpt);

     }
   fprintf(fp,"END_VARIABLE = %s\n", QiSCDF->vardata[var_num]->name);
   fprintf(fp,"\n");


  return QMW_OK;

} /* end QiWriteVarObjects */

/***********************************************************************/

long QiAttrs_to_Text(QiCDFContents *QiSCDF,
                       long Vnum,
                       FILE *fp,
                       QiOptions *QiSOpt)
{
 long n, nn;
 float *float_data;
 long *long_data;
 char * char_data;
 unsigned char * uchar_data;
 double *double_data;
 int *short_data;
 long nOtherVar;
 int lineTotal;
 int n_ptr;

  QiCDFVariable *variable = QiSCDF->vardata[Vnum];


 int isCEF2;
 if(QiSOpt->f_type == CAA) isCEF2 = 1;
 else isCEF2 = 0;

   for(n=0; n< variable->num_v_attrs; n++)
   {
       /* defensive programming: skip bad attributes */
       if (variable->attribute[n].data == NULL) break;

       char_data = (char *)variable->attribute[n].data;
       nOtherVar = QiGetVarByName(QiSCDF, char_data);
       if( nOtherVar != -1 && nOtherVar != Vnum){
         /* attribute points to another var in this structure */
          fprintf(fp, "%s = ", variable->attribute[n].name);
          fprintf(fp, "%s\n", &(char_data[0]));
          continue;
       }
	  
	  if(strcmp(QiToUpper(variable->attribute[n].name), "DATA_TYPE") == 0) continue; // skip this as an attribute

       fprintf(fp, "%s = ", variable->attribute[n].name);

       switch(variable->attribute[n].data_type)
       {
         case CDF_REAL4: case CDF_FLOAT:
         {
           float_data = (float *)variable->attribute[n].data;
           fprintf(fp, "%g", (double)float_data[0]);
         lineTotal = 0;
           for (nn=1; nn<variable->attribute[n].num_elems; nn++){
           lineTotal++;
             fprintf(fp, "%c", QiSOpt->attr_delim);
           if(lineTotal > MAX_STR_LEN_CEF2/(REAL_WIDTH+1)  && isCEF2) {
             fprintf(fp, "\\ \n");
             lineTotal = 0;
           }
             fprintf(fp, "%g", (double)float_data[nn]);
           }
           break;
         }
         case CDF_DOUBLE: case CDF_REAL8:
         {
          double_data = (double *)variable->attribute[n].data;
          fprintf(fp, "%g", double_data[0]);
        lineTotal = 0;
          for (nn=1; nn<variable->attribute[n].num_elems; nn++){
          lineTotal++;
            fprintf(fp, "%c", QiSOpt->attr_delim);
          if(lineTotal > MAX_STR_LEN_CEF2/(DOUBLE_WIDTH+1)  && isCEF2) {
            fprintf(fp, "\\ \n");
            lineTotal = 0;
          }
            fprintf(fp, "%g", double_data[nn]);
          }
           break;
         }
         case CDF_EPOCH:
         {
           double_data = (double *)  variable->attribute[n].data;
           fprintf(fp, "%s", QiEpochToISOString(double_data[0], QiSOpt->time_sep_attr) );
         lineTotal = 0;
           for (nn=1; nn<variable->attribute[n].num_elems; nn++){
           lineTotal++;
             fprintf(fp, "%c", QiSOpt->attr_delim);
           if(lineTotal > MAX_STR_LEN_CEF2/(EPOCH_WIDTH+1)  && isCEF2){
             fprintf(fp, "\\ \n");
             lineTotal = 0;
           }
             fprintf(fp, "%s", QiEpochToISOString(double_data[nn], QiSOpt->time_sep_attr));
           }
           break;
         }
         case CDF_EPOCH16:
         {
           double_data = (double *)  variable->attribute[n].data;
           fprintf(fp, "%s", QiEpoch16ToISOString(&(double_data[0]), QiSOpt->time_sep_attr) );
           lineTotal = 0;
           for (nn=1; nn<variable->attribute[n].num_elems; nn++){
             lineTotal++;
             fprintf(fp, "%c", QiSOpt->attr_delim);
             if(lineTotal > MAX_STR_LEN_CEF2/(EPOCH_WIDTH+1)  && isCEF2){
               fprintf(fp, "\\ \n");
               lineTotal = 0;
             }
             fprintf(fp, "%s", QiEpoch16ToISOString(&(double_data[2*nn]), QiSOpt->time_sep_attr));
           }
           break;
         }
         case CDF_INT1:
         {
          char_data = ( char *)variable->attribute[n].data;
          fprintf(fp, "%u", ( int)char_data[0]);
        lineTotal = 0;
          for (nn=1; nn<variable->attribute[n].num_elems; nn++){
          lineTotal++;
            fprintf(fp, "%c", QiSOpt->attr_delim);
          if(lineTotal > MAX_STR_LEN_CEF2/(BYTE_WIDTH+1)  && isCEF2) {
            fprintf(fp, "\\ \n");
            lineTotal = 0;
          }
            fprintf(fp, "%u", ( int)char_data[nn]);
          }
           break;
         }
         case CDF_UINT1: case CDF_BYTE:
         {
          uchar_data = (unsigned char *)variable->attribute[n].data;
          fprintf(fp, "%u", (unsigned int)uchar_data[0]);
        lineTotal = 0;
          for (nn=1; nn<variable->attribute[n].num_elems; nn++){
          lineTotal++;
            fprintf(fp, "%c", QiSOpt->attr_delim);
          if(lineTotal > MAX_STR_LEN_CEF2/(BYTE_WIDTH+1)  && isCEF2) {
            fprintf(fp, "\\ \n");
            lineTotal = 0;
          }
            fprintf(fp, "%u", (unsigned int)char_data[nn]);
          }
           break;
         }
         case ISO_TIME:
         {
           char_data = (char *) variable->attribute[n].data;
           n_ptr = 0;
           lineTotal = 0;
           fprintf(fp, "%s", &(char_data[n_ptr]));
           for (nn=1; nn<variable->attribute[n].num_elems;  nn++){
           n_ptr += strlen(&(char_data[n_ptr]))+1;
             fprintf(fp, "%c", QiSOpt->attr_delim);
           lineTotal += strlen(&(char_data[n_ptr]))+1;
           if(lineTotal >= MAX_STR_LEN_CEF2 && isCEF2 ){
               /* break excessive lines */
              fprintf(fp, "\\ \n");
              lineTotal = 0;
           }
             fprintf(fp, "%s", &(char_data[n_ptr]));
           }
           break;
         }
         case ISO_TIME_RANGE:
         {
           char_data = (char *) variable->attribute[n].data;
           n_ptr = 0;
           lineTotal = 0;
           fprintf(fp, "%s", &(char_data[n_ptr]));
           for (nn=1; nn<variable->attribute[n].num_elems;  nn++){
           n_ptr += strlen(&(char_data[n_ptr]))+1;
             fprintf(fp, "%c", QiSOpt->attr_delim);
           lineTotal += strlen(&(char_data[n_ptr]))+1;
           if(lineTotal >= MAX_STR_LEN_CEF2 && isCEF2 ){
               /* break excessive lines */
              fprintf(fp, "\\ \n");
              lineTotal = 0;
           }
             fprintf(fp, "%s", &(char_data[n_ptr]));
           }
           break;
         }
         case CDF_CHAR:
         {
           if(isCEF2){
           char_data = (char *) variable->attribute[n].data;
           n_ptr = 0;
           lineTotal = 0;
             fprintf(fp, "\"%s\"", &(char_data[n_ptr]));
             for (nn=1; nn<variable->attribute[n].num_elems;  nn++){
             n_ptr += strlen(&(char_data[n_ptr]))+1;
               fprintf(fp, "%c", QiSOpt->attr_delim);
             lineTotal += strlen(&(char_data[n_ptr]))+1;
             if(lineTotal >= MAX_STR_LEN_CEF2){
               /* break excessive lines */
                fprintf(fp, "\\ \n");
              lineTotal = 0;
             }
               fprintf(fp, "\"%s\"",  &(char_data[n_ptr]));
             }
         }
         else{
           char_data = (char *) variable->attribute[n].data;
           n_ptr = 0;
             fprintf(fp, "%s", &(char_data[n_ptr]));
             for (nn=1; nn<variable->attribute[n].num_elems;  nn++){
             n_ptr += strlen(&(char_data[n_ptr]))+1;
               fprintf(fp, "%c%s", QiSOpt->attr_delim, &(char_data[n_ptr]));
             }

         }
           break;
         }
         case CDF_UCHAR:
         {
           if(isCEF2){
             char_data = ( char *) variable->attribute[n].data;
           n_ptr = 0;
           lineTotal = 0;
             fprintf(fp, "\"%s\"", (unsigned char *)&(char_data[n_ptr]));
             for (nn=1; nn<variable->attribute[n].num_elems;  nn++){
             n_ptr += strlen(&(char_data[n_ptr]))+1;
               fprintf(fp, "%c", QiSOpt->attr_delim);
             lineTotal += strlen(&(char_data[n_ptr]))+1;
             if(lineTotal >= MAX_STR_LEN_CEF2){
               /* break excessive lines */
                fprintf(fp, "\\ \n");
              lineTotal = 0;
             }
               fprintf(fp, "\"%s\"", (unsigned char*)&(char_data[n_ptr]));
             }
         }
         else{
             char_data = ( char *) variable->attribute[n].data;
           n_ptr = 0;
             fprintf(fp, "%s", (unsigned char *)&(char_data[n_ptr]));
             for (nn=1; nn<variable->attribute[n].num_elems;  nn++){
             n_ptr += strlen(&(char_data[n_ptr]))+1;
               fprintf(fp, "%c%s", QiSOpt->attr_delim, (unsigned char*)&(char_data[n_ptr]));
             }
         }
           break;
         }
         case CDF_INT2: case CDF_UINT2:
         {
          short_data = (int *)variable->attribute[n].data;
          fprintf(fp, "%d", short_data[0]);
        lineTotal = 0;
          for (nn=1; nn<variable->attribute[n].num_elems; nn++){
            fprintf(fp, "%c", QiSOpt->attr_delim);
          lineTotal++;
          if(lineTotal > MAX_STR_LEN_CEF2/(SHORT_WIDTH+1)  && isCEF2) {
            fprintf(fp, "\\ \n");
            lineTotal = 0;
          }
            fprintf(fp, "%d", short_data[nn]);
          }
           break;
         }
         case CDF_INT4: case CDF_UINT4:
         {
          long_data = (long *)variable->attribute[n].data;
          fprintf(fp, "%ld", long_data[0]);
        lineTotal = 0;
          for (nn=1; nn<variable->attribute[n].num_elems; nn++){
          lineTotal++;
            fprintf(fp, "%c", QiSOpt->attr_delim);
          if(lineTotal > MAX_STR_LEN_CEF2/(INT_WIDTH+1)  && isCEF2) {
            fprintf(fp, "\\ \n");
            lineTotal = 0;
          }
            fprintf(fp, "%ld", long_data[nn]);
          }
           break;
         }
         default:
         {
            fprintf(fp, "unreadable");
         }
       } /* end switch */

       fprintf(fp, "\n");


   } /* end for */

   return QMW_OK;

} /* end QiAttr_to_Text */


/***********************************************************************/

long QiWriteRecsTabular(QiCDFContents *QiSCDF,
                        QiRecord_format *QiSfmt,
                        QiOptions *QiSOpt,
                        FILE *fp)
{
  long n, j, m;
  int diff;
  float *float_data;
  long *long_data;
  char * char_data;
  unsigned char * uchar_data;
  double *double_data;
  short *short_data;
  long check = QMW_OK;
  long mm, jj;
  char pad = ' ';
  char message[WINDOW_LEN+1];
  int first_var;

   QiSfmt->title = (char *)malloc(QiSfmt->total_width +1);

   if(QiSOpt->rec_numbering == NUM_ON){
     QiStrcpy(QiSfmt->title, "! Record  ", QiSfmt->total_width);
   }
   else{
     QiStrcpy(QiSfmt->title, "!", QiSfmt->total_width);
   }

   /* write out title line  */

   for (n=0; n < QiSCDF->n_vars; n++){

     /* skip write once NOVARYs */
     if(QiSCDF->vardata[n]->rec_vary == NOVARY &&
        QiSCDF->vardata[n]->novary_opt == WRITE_ONCE) continue;

     /* pad to start of column */

     diff =  (int)QiSfmt->col_start[n] - strlen(QiSfmt->title);
     for(j=0; j<diff; j++){
       strcat(QiSfmt->title," ");
     }
     /* append  variable name */
     strcat(QiSfmt->title, QiSCDF->vardata[n]->name);
   }
   if(QiSOpt->header == ATTACHED) {
     fprintf(fp,"%s\n\n" ,QiSfmt->title);
     fprintf(fp, "%s = %ld ! Data records to follow\n", "Start_data",
             QiSCDF->n_recs);
   }

   /* write out data records */

   for (j=0; j < QiSCDF->n_recs; j++){

     /* write record number */

     if(QiSOpt->rec_numbering == NUM_ON){
       fprintf(fp,"%-*ld%c", (REC_N_WIDTH - 1), j, pad);
     }

     first_var = 0; /* flag to trap space pad for first variable in ea rec */

     /* write entry for each variable */

     for (n=0; n < QiSCDF->n_vars; n++){

       if (QiSCDF->vardata[n]->rec_vary == VARY) jj = j;
       else {
         jj = 0;
         if(QiSCDF->vardata[n]->novary_opt == WRITE_ONCE) continue;
         /* skip writing if write once */
       }

       /* trap missing data */
       if (QiSCDF->vardata[n]->data == NULL){
        sprintf(message, "QIE Error: data not available for var number %ld", n);
         QiDisplayMessage(message, QiSOpt);
         return DATA_NOT_SET;
       }

         mm = QiSfmt->n_items[n];
         for(m=0; m < mm; m++){

    /* add space to ensure columns separate (hence use {col_width -1} below) */
           if (first_var == 0 ) first_var = 1;
           else{
             fprintf(fp, "%c", pad);
           }

          switch(QiSCDF->vardata[n]->data_type)
          {
           case CDF_REAL4: case CDF_FLOAT:
           {
             float_data = (float *)QiSCDF->vardata[n]->data;
             fprintf(fp, "%-*g", QiSfmt->col_width[n] -1,
                 (double)float_data[mm*jj + m]);
             break;
           }
           case CDF_DOUBLE: case CDF_REAL8:
           {
             double_data = (double *)QiSCDF->vardata[n]->data;
             fprintf(fp, "%-*g", QiSfmt->col_width[n]-1,
                   double_data[mm*jj+m]);
             break;
           }
           case CDF_EPOCH:
          {
             double_data = (double *)QiSCDF->vardata[n]->data;
             fprintf(fp, "%-*s", QiSfmt->col_width[n] -1,
             QiEpochToISOString(double_data[mm*jj+m], QiSOpt->time_sep_data) );
             break;
           }
           case CDF_EPOCH16:
          {
             double_data = (double *)QiSCDF->vardata[n]->data;
             fprintf(fp, "%-*s", QiSfmt->col_width[n] -1,
             QiEpoch16ToISOString(&(double_data[2*mm*jj+m]), QiSOpt->time_sep_data) );
             break;
           }
           case CDF_UINT1: case CDF_INT1: case CDF_BYTE:
           {
             uchar_data = (unsigned char *)QiSCDF->vardata[n]->data;
             fprintf(fp, "%-*u", QiSfmt->col_width[n] -1,
                      (unsigned int)uchar_data[mm*jj + m]);
             break;
           }
           case ISO_TIME:
           {
             char_data = (char *)QiSCDF->vardata[n]->data;
             fprintf(fp, "%-*.*s",
                   (int)QiSCDF->vardata[n]->num_elems,
                   (int)QiSCDF->vardata[n]->num_elems,
                   &(char_data[(mm*jj+m)*QiSCDF->vardata[n]->num_elems]) );
             break;
           }
           case ISO_TIME_RANGE:
           {
             char_data = (char *)QiSCDF->vardata[n]->data;
             fprintf(fp, "%-*.*s",
                   (int)QiSCDF->vardata[n]->num_elems,
                   (int)QiSCDF->vardata[n]->num_elems,
                   &(char_data[(mm*jj+m)*QiSCDF->vardata[n]->num_elems]) );
             break;
           }
           case CDF_CHAR: case CDF_UCHAR:
           {
             char_data = (char *) QiSCDF->vardata[n]->data;
             fprintf(fp, "%-*.*s",
                     (int)QiSCDF->vardata[n]->num_elems,
                     (int)QiSCDF->vardata[n]->num_elems,
                     &(char_data[(mm*jj+m)*QiSCDF->vardata[n]->num_elems]) );
             break;
           }
           case CDF_INT2: 
           {
             short_data = (short *)QiSCDF->vardata[n]->data;
             fprintf(fp, "%-*hd", QiSfmt->col_width[n] -1,
                     short_data[mm*jj + m]);
             break;
           }
          case CDF_UINT2:
           {
             short_data = (short *)QiSCDF->vardata[n]->data;
             fprintf(fp, "%-*hu", QiSfmt->col_width[n] -1,
                     short_data[mm*jj + m]);
             break;
           }
           case CDF_INT4: case CDF_UINT4:
           {
             long_data = (long *)QiSCDF->vardata[n]->data;
             fprintf(fp, "%-*ld", QiSfmt->col_width[n] -1,
                     long_data[mm*jj + m]);
             break;
           }
           default:
           {
              fprintf(fp, "??");
           }
         } /* end switch */

        } /* end for over dimensions */
     }  /* end for over vars */

     fprintf(fp,"\n");

   } /* end for over records */
  return check;

} /* end QiWriteRecsTabular */

/*********************************************************************/

long QiWriteRecsParsed(QiCDFContents *QiSCDF,
                       QiRecord_format *QiSfmt,
                       QiOptions *QiSOpt,
                       FILE *fp)
{
  long n, j;
  long check;
  long jj;
  char pad[2];
  char message[WINDOW_LEN+1];
  long call;
  int beSafe=5;

    check = QMW_OK;

    pad[0] = QiSOpt->data_delim;
    pad[1] = '\0';


   /* ensure that space is not used in time field if delim is space */
   if(QiSOpt->data_delim == ' ') QiSOpt->time_sep_data = 'T';

   QiSfmt->title = (char *)malloc(QiSfmt->total_width +1+beSafe);

   if(QiSOpt->rec_numbering == NUM_ON){
     QiStrcpy(QiSfmt->title, "!Record", QiSfmt->total_width+beSafe);
     strcat(QiSfmt->title, pad);
   }
   else{
     QiStrcpy(QiSfmt->title, "!", QiSfmt->total_width+beSafe);
   }

   /* assemble title line  */

   strncat(QiSfmt->title, QiSCDF->vardata[0]->name, QiSfmt->col_width[0]-1);
   for (n=1; n < QiSCDF->n_vars; n++){
     strcat(QiSfmt->title, pad);
     strncat(QiSfmt->title, QiSCDF->vardata[n]->name, QiSfmt->col_width[n]-1);
   }

   /* write out title line  */

   if(QiSOpt->header == ATTACHED) {
     fprintf(fp,"%s\n\n" ,QiSfmt->title);
     fprintf(fp, "%s = %ld ! Data records to follow\n", "Start_data",
             QiSCDF->n_recs);
   }

   /* write out data records */

   for (j=0; j < QiSCDF->n_recs; j++){

     call = 0;

     /* write record number */

     if(QiSOpt->rec_numbering == NUM_ON){
       fprintf(fp,"%ld", j);
       call = 1;
     }

     /* write entry for each variable */

     for (n=0; n < QiSCDF->n_vars; n++){

       /* trap bad data records */
       if( QiSCDF->vardata[n]->data == NULL){
         sprintf(message, "QIE Error: data not available, var number %ld", n);
         QiDisplayMessage(message, QiSOpt);
         return DATA_NOT_SET;
       }

       if (QiSCDF->vardata[n]->rec_vary == VARY) jj = j;
       else {
         jj = 0;  /* repeat same value, or... */
         if(QiSCDF->vardata[n]->novary_opt == WRITE_ONCE) continue;
         /* skip writing if write once */
       }


       QiWriteVarData(fp, QiSCDF->vardata, n,
                      QiSfmt->n_items[n], jj,
                  call, QiSOpt, QiSOpt->data_delim);
       call = 1;

     }

     fprintf(fp,"\n");

   } /* end for over records */

  return check;

} /* end QiWriteRecsParsed */


/*********************************************************************/

long QiWriteRecsExchange(QiCDFContents *QiSCDF,
                       QiRecord_format *QiSfmt,
                       QiOptions *QiSOpt,
                       FILE *fp)
{
  long n, j;
  long check;
  long jj;
  long nv=0;
  char pad[3];
  char message[WINDOW_LEN+1];
  long nv_count;
  long ns[MAX_N_VARS]; /* n sorted to put time tags first*/

   check = QMW_OK;

  /* ensure time var is first */
  j=0;
  if(QiSCDF->d0_var_num > -1) {
    ns[0] = QiSCDF->d0_var_num;
    j++;
  }
  for(n=0; n < QiSCDF->n_vars; n++){
    /* add rest of variables */
    if(n != QiSCDF->d0_var_num){
       ns[j] = n;
       j++;
    }
  }

/* choose appropriate formatting character for record delim */

   if( QiSOpt->rec_end == '\n' ) QiSOpt->row_end = ' ';
   else if( QiSOpt->rec_end == '\t' ) QiSOpt->row_end = ' ';
   else QiSOpt->row_end = '\n';


    pad[0] = QiSOpt->data_delim;
    pad[1] = ' ';
    pad[2] = '\0';

   /* ensure that space is not used in time field if delim is space */
   if(QiSOpt->data_delim == ' ') QiSOpt->time_sep_data = 'T';


   if(QiSfmt->total_width > 0){
     // allocate plenty of space for variable name line
     QiSfmt->title = (char *)malloc((QiSfmt->total_width)*2 +1);

     if(QiSOpt->rec_numbering == NUM_ON){
       QiStrcpy(QiSfmt->title, "!Record", (QiSfmt->total_width)*2);
       strcat(QiSfmt->title, pad);
     }
     else{
       QiStrcpy(QiSfmt->title, "! ", (QiSfmt->total_width)*2);
     }

     /* assemble title line  */

     strncat(QiSfmt->title, QiSCDF->vardata[ns[0]]->name, QiSfmt->col_width[ns[0]]-1);
     for (n=1; n < QiSCDF->n_vars; n++){
       if(QiSCDF->vardata[ns[n]]->rec_vary == VARY){
         strcat(QiSfmt->title, pad);
         strncat(QiSfmt->title, QiSCDF->vardata[ns[n]]->name, QiSfmt->col_width[ns[n]]-1);
       }
     }

     /* write out title line  */

     if(QiSOpt->header == ATTACHED) {
       fprintf(fp,"%s\n\n" ,QiSfmt->title);
       fprintf(fp, "%s = %ld ! Data records to follow\n", "Start_data",
             QiSCDF->n_recs);
     }
   }

   /* count number of variables in record line */

   for (n=0; n < QiSCDF->n_vars; n++)
      if(QiSCDF->vardata[ns[n]]->rec_vary == VARY) nv++;

   /* write out data records */

   for (j=0; j < QiSCDF->n_recs; j++){


     /* write record number */

     if(QiSOpt->rec_numbering == NUM_ON){
       fprintf(fp,"%ld", j);
     }

     /* write entry for each variable */

     nv_count = nv;
     for (n=0; n < QiSCDF->n_vars; n++){

       /* trap bad data records */
       if( QiSCDF->vardata[ns[n]]->data == NULL){
         sprintf(message, "QIE Error: data not available, var number %ld", ns[n]);
         QiDisplayMessage(message, QiSOpt);
         return DATA_NOT_SET;
       }

       if (QiSCDF->vardata[ns[n]]->rec_vary == VARY) jj = j;
       else {
         jj = 0;  /* repeat same value, or... */
         if(QiSCDF->vardata[ns[n]]->novary_opt == WRITE_ONCE) continue;
         /* skip writing if write once */
       }
       if(nv_count == nv)
          fprintf(fp,"%c", QiSOpt->row_end);
       else
          fprintf(fp,"%c   ", QiSOpt->row_end);

       nv_count--;
       QiWriteVarBlocked(fp, QiSCDF->vardata, ns[n], nv_count,
                      QiSfmt->n_items[ns[n]], jj,
                  QiSOpt, QiSOpt->data_delim);
     }

     fprintf(fp,"%c%c", QiSOpt->row_end, QiSOpt->rec_end);
     if(QiSOpt->rec_end != '\n') fprintf(fp,"\n");
   } /* end for over records */

  return check;

} /* end QiWriteRecsExchange */

/*********************************************************************/

long QiWriteRecsCAA(QiCDFContents *QiSCDF,
                       QiRecord_format *QiSfmt,
                       QiOptions *QiSOpt,
                       FILE *fp)
{
  long n, j;
  long check;
  long jj;
  long nv=0;
  char pad[3];
  char message[WINDOW_LEN+1];
  long nv_count;
  long ns[MAX_N_VARS]; /* n sorted to put time tags first*/

   check = QMW_OK;

  /* ensure time var is first */
  j=0;
  if(QiSCDF->d0_var_num > -1) {
    ns[0] = QiSCDF->d0_var_num;
    j++;
  }
  for(n=0; n < QiSCDF->n_vars; n++){
    /* add rest of variables */
    if(n != QiSCDF->d0_var_num){
       ns[j] = n;
       j++;
    }
  }

/* choose appropriate formatting character for record delim */

   if( QiSOpt->rec_end == '\n' ) QiSOpt->row_end = ' ';
   else if( QiSOpt->rec_end == '\t' ) QiSOpt->row_end = ' ';
   else QiSOpt->row_end = '\n';


    pad[0] = QiSOpt->data_delim;
    pad[1] = ' ';
    pad[2] = '\0';

   /* ensure that space is not used in time field if delim is space */
   if(QiSOpt->data_delim == ' ') QiSOpt->time_sep_data = 'T';


   if(QiSfmt->total_width > 0){
     // allocate plenty of space for variable name line
     QiSfmt->title = (char *)malloc((QiSfmt->total_width)*2 +1);

     QiStrcpy(QiSfmt->title, "! ", (QiSfmt->total_width)*2);
     /* assemble title line  */

     strncat(QiSfmt->title, QiSCDF->vardata[ns[0]]->name, QiSfmt->col_width[ns[0]]-1);
     for (n=1; n < QiSCDF->n_vars; n++){
       if(QiSCDF->vardata[ns[n]]->rec_vary == VARY){
         strcat(QiSfmt->title, pad);
         strncat(QiSfmt->title, QiSCDF->vardata[ns[n]]->name, QiSfmt->col_width[ns[n]]-1);
       }
     }

     /* write out title line  */

     if(QiSOpt->header == ATTACHED) {
       fprintf(fp,"%s\n\n" ,QiSfmt->title);
       fprintf(fp,"DATA_UNTIL = \"End_of_Data\"\n");
     }
   }

   /* count number of variables in record line */

   for (n=0; n < QiSCDF->n_vars; n++)
      if(QiSCDF->vardata[ns[n]]->rec_vary == VARY) nv++;

   /* write out data records */

   for (j=0; j < QiSCDF->n_recs; j++){


     /* write entry for each variable */

     nv_count = nv;
     for (n=0; n < QiSCDF->n_vars; n++){

       /* trap bad data records */
       if( QiSCDF->vardata[ns[n]]->data == NULL){
         sprintf(message, "QIE Error: data not available, var number %ld", ns[n]);
         QiDisplayMessage(message, QiSOpt);
         return DATA_NOT_SET;
       }

       if (QiSCDF->vardata[ns[n]]->rec_vary == VARY) jj = j;
       else {
         jj = 0;  /* repeat same value, or... */
         if(QiSCDF->vardata[ns[n]]->novary_opt == WRITE_ONCE) continue;
         /* skip writing if write once */
       }
       if(nv_count == nv)
          fprintf(fp,"%c", QiSOpt->row_end);
       else
          fprintf(fp,"%c   ", QiSOpt->row_end);

       nv_count--;
       QiWriteVarBlocked(fp, QiSCDF->vardata, ns[n], nv_count,
                      QiSfmt->n_items[ns[n]], jj,
                  QiSOpt, QiSOpt->data_delim);
     }

     fprintf(fp,"%c%c", QiSOpt->row_end, QiSOpt->rec_end);
     if(QiSOpt->rec_end != '\n') fprintf(fp,"\n");
   } /* end for over records */

  fprintf(fp,"End_of_Data\n");
  return check;

} /* end QiWriteRecsCAA */

/**********************************************************************/

long  QiWriteGlobalFlat(QiCDFContents * QiSCDF,
                        QiOptions * QiSOpt,             /* i/o options */
                        long  n_G,
                        FILE *fp)

  {
   long n_entries;
   long n;
   char * entry;
   long current_datatype = CDF_CHAR;
   int typeCAA=0;

   if(QiSOpt->f_type == CAA) typeCAA = 1;

   /* write start line */

   if ( QiSCDF->g_attr[n_G] == NULL) return QMW_WARNING;

   fprintf(fp, "Start_meta = %s\n", QiSCDF->g_attr[n_G]->name);

   if(strcmp(QiSCDF->g_attr[n_G]->name, "Software_version") ==0){
     n_entries = QiSCDF->g_attr[n_G]->num_entries +1;
   }
   else{
     n_entries = QiSCDF->g_attr[n_G]->num_entries;
   }
  if(!typeCAA) {
    fprintf(fp, "Number_of_entries = %ld\n", n_entries);
  }

   for (n=0; n < n_entries; n++){

     if ( (QiSCDF->g_attr[n_G]->entry+n)->exists != EXISTS ) continue;

     /* set datatype if different from current type, default is char */
     if ( current_datatype != (QiSCDF->g_attr[n_G]->entry+n)->data_type ) {
       if( typeCAA && 
	     ((QiSCDF->g_attr[n_G]->entry+n)->data_type == CDF_EPOCH || (QiSCDF->g_attr[n_G]->entry+n)->data_type == CDF_EPOCH16)){
            entry = (char *)malloc( sizeof(char) * 9);
            strcpy(entry, "ISO_TIME");
       }
       else{
          entry = QiCDFDataType( (QiSCDF->g_attr[n_G]->entry+n)->data_type);
       }
       fprintf(fp, "G_Attr_Data_type = %s\n", entry );
       free (entry);
       current_datatype = (QiSCDF->g_attr[n_G]->entry+n)->data_type;
     }

     /* write entry */
     QiWriteEntry( (QiSCDF->g_attr[n_G]->entry+n), QiSOpt, fp);

   }
   if(strcmp(QiSCDF->g_attr[n_G]->name, "Software_version") == 0){
     if(typeCAA) fprintf(fp, "Entry = \"%s\"\n", QieVersion());
     else        fprintf(fp, "Entry = %s\n", QieVersion());
   }
   fprintf(fp, "End_meta = %s\n", QiSCDF->g_attr[n_G]->name);
   fprintf(fp,"!\n");

   return QMW_OK;

  } /* end   QiWriteGlobalFlat */

/**********************************************************************/

long  QiWriteEntry( QiGAttrEntry * entry,
                   QiOptions *QiSOpt,             /* i/o options */
                   FILE *fp)

 {
   float *float_data;
   int *int_data;
   char * char_data;
   double *double_data;
   int *short_data;
   long n;


      /* trap NULL entries */
      if (entry->data == NULL){
        /* skip entry */
      }
      else{
      switch(entry->data_type){
         case CDF_REAL4: case CDF_FLOAT:
         {
           float_data = (float *) entry->data;
           fprintf(fp, "Entry = %g", (double)float_data[0]);
           for (n=1; n<entry->num_elems; n++){
             fprintf(fp, "%c%g", QiSOpt->attr_delim, (double)float_data[n]);
           }
           break;
         }
         case CDF_DOUBLE: case CDF_REAL8:
         {
          double_data = (double *) entry->data;
          fprintf(fp, "Entry = %g", double_data[0]);
           for (n=1; n<entry->num_elems; n++){
            fprintf(fp, "%c%g", QiSOpt->attr_delim, double_data[n]);
          }
           break;
         }
         case CDF_EPOCH:
         {
           double_data = (double *) entry->data;
           fprintf(fp, "Entry = %s",
               QiEpochToISOString(*double_data, QiSOpt->time_sep_attr) );
           break;
         }
         case CDF_EPOCH16:
         {
           double_data = (double *) entry->data;
           fprintf(fp, "Entry = %s",
               QiEpoch16ToISOString(double_data, QiSOpt->time_sep_attr) );
           break;
         }
         case CDF_UINT1: case CDF_INT1: case CDF_BYTE:
         {
          char_data = (char *) entry->data;
          fprintf(fp, "Entry = %u", (int)char_data[0]);
           for (n=1; n<entry->num_elems; n++){
            fprintf(fp, "%c%d", QiSOpt->attr_delim, (int)char_data[n]);
          }
           break;
         }
         case CDF_CHAR: case CDF_UCHAR:
         {
           char_data = (char *) entry->data;
           if(QiSOpt->f_type == CAA) fprintf(fp, "Entry = \"%.*s\"", (int)entry->num_elems, char_data);
         else fprintf(fp, "Entry = %.*s", (int)entry->num_elems, char_data);
           break;
         }
         case ISO_TIME:
         {
           char_data = (char *) entry->data;
           fprintf(fp, "Entry = %.*s", (int)entry->num_elems, char_data);
           break;
         }
         case ISO_TIME_RANGE:
         {
           char_data = (char *) entry->data;
           fprintf(fp, "Entry = %.*s", (int)entry->num_elems, char_data);
           break;
         }
         case CDF_INT2: case CDF_UINT2:
         {
          short_data = (int *) entry->data;
          fprintf(fp, "Entry = %hd", short_data[0]);
           for (n=1; n<entry->num_elems; n++){
            fprintf(fp, "%c%hd", QiSOpt->attr_delim, short_data[n]);
          }
           break;
         }
         case CDF_INT4: case CDF_UINT4:
         {
          int_data = (int *) entry->data;
          fprintf(fp, "Entry = %d", int_data[0]);
           for (n=1; n<entry->num_elems; n++){
            fprintf(fp, "%c%d", QiSOpt->attr_delim, int_data[n]);
          }
           break;
         }
         default:
         {
            fprintf(fp, "unreadable");
         }

   } /* end switch */

   } /* end if-else */

   /* terminate line */

   fprintf(fp, "\n");

   return QMW_OK;

 } /* end QiWriteEntry */

/**********************************************************************
**************** START OF FLAT FILE READING FUNCTIONS *****************
**********************************************************************/


long QiGetCSDSgenFlat (QiCDFContents *QiSCDF, QiOptions *QiSOpt)
{

   long check = QMW_OK;
   long checkH = QMW_OK;
   FILE *fp;
   FILE *fp_hdr;
   QiRecord_format *QiSfmt;
   char message[LINE_LEN];
   int j;
   int i;

   end_of_line_mark = '\n';
   
   QiSfmt = (QiRecord_format *) QiMakeFmtObj();

  /* initialise defaults */
  QicMark = '!';       /* this is a global */
  fp_hdr = NULL;
  for( j=0; j< MAX_N_INC_FILES; j++) {
    includedFiles[j] = NULL;
    includedFrom[j] = -1;
    fileNames[j] = NULL;
  }

  incReadPath = QiSCDF->io_f_path;
   fpInc = -1; /* keeping track of files that have been included */

   /* open data file */

   fp = QiOpenFile((char*)"rb", QiSCDF->io_f_path, QiSCDF->io_f_name, QiSCDF->io_f_extn);
   if(fp == NULL) {
      sprintf(message,"Cannot open data file... \n%s%s\n", QiSCDF->io_f_name, QiSCDF->io_f_extn);
      QieAlertBox ("ERROR", message);
      return CANNOT_OPEN_FILE;
   }

   /* allocate memory for variable pointers */

   QiSCDF->vardata = (QiCDFVariable **) QiMakeQiVariablePtrs(MAX_N_VARS);

   /* flat files may have a few global attributes */

   QiSCDF->g_attr = (QiGlobalAttribute  **) QiMakeQiGAttrPtrs(MAX_N_ATTRS);
   QiSCDF->num_g_attrs = 0;


   if(QiSOpt->header == ATTACHED){

     checkH =  QiReadHeader(QiSCDF, QiSOpt, QiSfmt, fp);
     /* ensure header is attached, if not clean up and look for detached header */
     if (checkH != QMW_OK) {
       printf("%s\n",QiErrStr(checkH));
       QiSfmt = QiFreeFormatSpace(QiSfmt);       /* remove partially filled object */
       QiSfmt = (QiRecord_format *) QiMakeFmtObj(); /* make new one */
       QiSOpt->header = DETACHED;
       QieAlertBox( "warning","can't read attached header, looking for detached");
     }
	 
   }

   /* if header flagged as detached, or not found attached open detached header */

   if(QiSOpt->header == DETACHED) {
      /* look for header if not set, look for header in same dir */
      if( QistrNULL(QiSOpt->header_name) ){

        QiSOpt->header_path = QiNewStr(QiSCDF->io_f_path);
        QiSOpt->header_name = QiFindHeader(QiSOpt->header_path, QiSOpt);

        if( QistrNULL(QiSOpt->header_name) ) return NO_HEADER_FOUND;
      }

      fp_hdr = QiOpenFile((char*)"rb", QiSOpt->header_path, QiSOpt->header_name, QiSOpt->EXTN_QHD);
      if(fp_hdr == NULL) {
        sprintf(message,"Cannot open header file \n%s%s\n",
               QiSOpt->header_name, QiSOpt->EXTN_QHD);
        QieAlertBox ("ERROR", message);
        checkH = CANNOT_OPEN_FILE;
      }
      else{
        /* read header */
        checkH =  QiReadHeader(QiSCDF, QiSOpt, QiSfmt, fp_hdr);
     }
   }

   if (checkH == QMW_OK)  {
      if( QiSOpt->f_type == UNSET ){
        /* file type not set in header or menu, so use best guess */
        QiSOpt->f_type = QiSOpt->type_guess;
        if (QiSOpt->type_guess == DELIMITED) {
          sprintf(message, "Trying to open ascii data file with entries delimited by \"%c\" ", QiSOpt->data_delim);
        }
        if (QiSOpt->type_guess == TABULAR) {
          sprintf(message, "Trying to open ascii data file as flat tabular entries");
        }
        if (QiSOpt->type_guess == EXCHANGE) {
          sprintf(message, "Trying to open file as Cluster Exchange format");
        }
        if (QiSOpt->type_guess == CAA) {
          sprintf(message, "Trying to open file as Cluster Exchange format 2 (CAA)");
        }
      else{
          sprintf(message, "Trying to open unknown file type");
      }
        QiDisplayMessage(message, QiSOpt);
      }

      /* if header is detached close it and all included files */

      if (fp_hdr != NULL) {
        for ( i=0; i<fpInc; i++){
          if(includedFiles[i] != NULL) fclose(includedFiles[i]);
          includedFiles[i] = NULL;
          if(fileNames[i] != NULL) free (fileNames[i]);
        }
        fpInc = 0;
        /* reset root file to data file */
        includedFiles[fpInc] = fp;
	   fpNow = 0;
      }

      /* Read records */

      check = QiReadData (QiSCDF, QiSfmt, QiSOpt);

   }
   else{
     check = checkH;
   }
   /* Close files and exit cleanly */

   for ( i=0; i<fpInc; i++){
     if(i=0 && strcmp(QiSCDF->io_f_extn, ".cef.gz") == 0){
#if defined _WIN32 || defined WIN32
  /* cannot get here on windows, no popen/pclose */
#else
	   if(includedFiles[0] != NULL) pclose(includedFiles[0]);
#endif
	   includedFiles[0] = NULL;
       if(fileNames[0] != NULL) free (fileNames[0]);
	 }
	 else{
       if(includedFiles[i] != NULL) fclose(includedFiles[i]);
       includedFiles[i] = NULL;
       if(fileNames[i] != NULL) free (fileNames[i]);
     }
   }

   QiSfmt = QiFreeFormatSpace(QiSfmt);

   return check;

} /* end  QiGetCSDSgenFlat  */

/*********************************************************************/

long QiReadHeader (QiCDFContents *QiSCDF,
                    QiOptions *QiSOpt,
                    QiRecord_format *QiSfmt,
                    FILE *fp_hdr)
{

   long check = QMW_OK;
   long n;
   long istabular=1;

   fpInc = 0;
   fpNow = 0;
   /* set root file to primary input file */
   includedFiles[fpInc] = fp_hdr;

   /* read Header */
   check = QiConfigureRead(QiSCDF, QiSOpt, QiSfmt);
   if (check != QMW_OK) return check;

   /* validate cef 2 inputs found */
   if(QiSOpt->f_type == CAA){
     if( QiSCDF->n_recs != 0) {
        QiCEFmsg( 0, "START_DATA not used in CEF 2 syntax" , " ", " ");
        QiSCDF->n_recs = 0;
     }
     if( strcmp(QiSOpt->data_until, STR_NULL) == 0)
        QiCEFmsg( 1, "No DATA_UNTIL parameter found in header. ", " Continuing to attempt read" , " ");
     if( strcmp(QiSOpt->start_after, STR_NULL) != 0)
        QiCEFmsg( 0, "START_AFTER parameter is ignored. ", " Continuing to attempt read" , " ");
     if(QicMark != '!')
        QiCEFmsg( 1, "CEF 2 format failure: Comment marker not exclamation mark '!'. ", " Continuing to attempt read" , " ");
     if(QiSOpt->attr_delim != ',')
        QiCEFmsg( 1, "CEF 2 format failure: Attribute delimiter must be a comma. ", " Continuing to attempt read" , " ");
    if( QiSOpt->data_delim != ',')
        QiCEFmsg( 1, "CEF 2 format failure: Data delimiter must be a comma. ", " Continuing to attempt read" , " ");
    if (QiSOpt->rec_numbering != NUM_OFF)
        QiCEFmsg( 0, "CEF 2 format warning: Record numbering is not supported in CEF 2. ", " Continuing to attempt read" , " ");
   }

   /* initialise record format object */

   QiSfmt->col_start = (int *)malloc(sizeof(int)*(QiSCDF->n_vars));
   QiSfmt->col_width = (int *)malloc(sizeof(int)*(QiSCDF->n_vars));
   QiSfmt->n_items = (long *)malloc(sizeof(long)*(QiSCDF->n_vars));
   QiSfmt->ignore = (long *)malloc(sizeof(long)*(QiSCDF->n_vars));
   QiSfmt->ftpackets =                                    /* SJS re FT */
        (QiFTpacket *)malloc(sizeof(QiFTpacket)*(QiSCDF->n_vars));


   /* Read variable metadata and determine record format structure */
   check = QiReadVmeta(QiSCDF, QiSOpt, QiSfmt);

   if( check == NO_YEAR_OFFSET) QieAlertBox ("ERROR", "No YY_offset for 2 char year");

   if (check != QMW_OK) return check;

   /* is file inconsistent with tabular */
   for (n=0; n<QiSCDF->n_vars ; n++){
     if( QiSCDF->vardata[n]->rec_vary == NOVARY ) continue; /* skip non-rec varying */
     if( QiSfmt->col_start[n] == -1 || QiSfmt->col_width[n] == -1 ){
       istabular=0;
       break;
     }
   }

   /* check if consistent with tabular file for guessing */
   if (QiSOpt->f_type == UNSET && istabular == 1){
        QiSOpt->type_guess = TABULAR;
   }

   /* check if consistent with tabular file for requested tabular option */
   if (QiSOpt->f_type == TABULAR && istabular == 0){
     QiSOpt->f_type = UNSET;
     QiDisplayMessage("QIE Warning: Header not consistent with TABULAR file",QiSOpt);
   }

   /* if only one variable requested, set ignore flags */
   if (!QistrNULL(QiSOpt->get_only) ) QiSetVarToGet(QiSCDF, QiSfmt, QiSOpt->get_only);

   return QMW_OK;

} /* end  QiReadHeader  */

/*********************************************************************/

FILE * QiOpenFile( char * mode_in,
                   char * path_in,
                   char * file_stem,
                   char * extn)
{
   FILE *fp;
   char * full_name;
   char  plug[2];
   char mode[4];
   char *safename;
   char *path;

 /* copy mode_in to avoid corruption of static */

   QiStrcpy(mode, mode_in, 3);

/* ensure path is safe */

   if(QistrNULL(path_in) ){
#if defined __VMS
     path = QiNewStr("");
#else
     path = QiNewStr(".");
#endif
   }
   else path = QiNewStr(path_in);

#if defined _WIN32 || defined WIN32
   if(path[strlen(path)-1] == '\\'){
      strcpy(plug, "");
   }
   else{
      strcpy(plug, "\\");
   }
#elif defined __VMS
   if(path[strlen(path)-1] == ']'){
      strcpy(plug, "");
   }
   else{
      strcpy(plug, "");
   }
#else
   if(path[strlen(path)-1] == '/'){
      strcpy(plug, "");
   }
   else{
      strcpy(plug, "/");
   }
#endif

   /* open named file */

   full_name = (char *) malloc(strlen(path)+strlen(file_stem)+strlen(extn)+2);
   strcpy(full_name, path);strcat(full_name, plug);
   strcat(full_name, file_stem);
   strcat(full_name, extn);

   safename = QiSafePath(&(full_name[0]));
   free(full_name);  /* this is not used elsewhere, work by pointer alone */

   /* if file is not to be overwritten, "w-", test existence */

   if(strcmp(mode, "r") == 0){
		// read file
		if(strcmp(extn, ".cef.gz") == 0){
			char * unzipname = (char *) malloc(strlen(safename) + 13);
			strcpy(unzipname, "gzip -cd ");
			strcat(unzipname, safename);
#if defined _WIN32 || defined WIN32
			/* cannot open .gz file on Windows, no popen() */
			fp = NULL;
#else
           fp=popen(unzipname, mode);
#endif
			free(unzipname);
		}
		else{
			fp=fopen(safename, mode);
		}
 
	}
	else if( mode[strlen(mode)-1] == '-'){
		mode[strlen(mode)-1] = '\0';

		/* try reading to see if file of that name exists */
		fp=fopen(safename, "r");
		if( fp == NULL ) {
			fp=fopen(safename, mode);
		}
		else{
			/* it exists, overwrite not allowed, bale out */
			fclose(fp);
			fp = NULL;
		}
	}
	else{
		/* open file with mode supplied */

		fp=fopen(safename, mode);

		/* on read only, if failure try without .extn */

		if(mode[0] == 'r' && fp == NULL){
			safename[strlen(safename)-strlen(extn)] = '\0';
			fp=fopen(safename, mode);
		}
	}

	free (path);

	return fp;

} /* end  QiOpenFile  */

/********************************************************************/

long QiConfigureRead(QiCDFContents *QiSCDF,   /* ptr to struct of contents */
                     QiOptions *QiSOpt,       /* i/o options */
                     QiRecord_format *QiSfmt) /* ptr to record format struct */
{
  char *header_line;
  long check=QMW_OK;
  char * param;
  char * value;
  char * attr;
  char message[LINE_LEN];
  fpos_t fPos;
  int fpRewind;

  /* initialise defaults */
  QicMark = '!';       /* this is a global */
  QiSCDF->n_vars = 0;
  year_offset = -1;  /* global, and acts as trap if YY format used */
  monotonic = 0;
  yp_last=-2000000000;

  /* look for version stamp and CEF 2 identifier */
  QiFindVersion(QiSOpt);

  if(QiSOpt->f_type == CAA) {
     /* these need not be written in the file */
     QiSOpt->attr_delim = ',';
     QiSOpt->data_delim = ',';
     isCAA = 1;
  }
  
  if(Qifgetpos(&fPos, &fpRewind) == -1){
    QiDisplayMessage("In QiConfigureRead: Failed to locate position in file", QiSOpt);  
  }
      
  /* note QiReadLine skips comment lines and removes newlines, returns etc */

  while((header_line=QiReadLine(&(STR_NULL[0]))) != NULL){
    QiDisplayMessage(header_line, QiSOpt);
           /* writes line out for debug  */

    if( isdigit(header_line[0]) ) break;
       /* detects (usually) data lines with no header or Start_data for speed */

    check = QiParseParamValue(header_line, &param, &value);
    if (check != QMW_OK) {
      QiCEFmsg( 0, "unidentified input line ", param, value);
      continue;
    }
     /* pick up info from header */

    if(strncmp(QiToUpper(param), "START_DATA", 10) == 0) {
      QiSCDF->n_recs = atol(value);
      break;
    }
    if(strncmp(QiToUpper(param), "DATA_UNTIL", 10) == 0) {
      QiSOpt->type_guess = CAA;
	  if(strcmp(value, "EOF") == 0 )
	  {
		QiSOpt->data_until = QiNewStr("EOF");
	  }
	  else
	  {
		QiSOpt->data_until = QiNewStr(QiStripQuotes(value));
	  }
      break;
    }
    if(strncmp(QiToUpper(param), "START_META", 10) == 0) {
      attr = QiNewStr(value);
      check = QiReadGlobalFlat(QiSCDF, QiSOpt, attr);
      free(attr);
      if(check == QMW_ERROR) return check;
      continue;
    }
    if(strncmp(QiToUpper(param), "YY_OFFSET", 9) == 0) {
      year_offset = atol(value); /* global year_offset */
      if(year_offset == 0 && (strchr(value, '0') == NULL) ) return BAD_Y_OFFSET;
      continue;
    }
    if(strncmp(QiToUpper(param), "YEAR_INCREASING", 17) == 0) {
       if(strchr(value,'y') != 0 || strchr(value,'Y') != 0)
         monotonic = 1;
      continue;
    }
    if(strncmp(QiToUpper(param), "FILE_TYPE", 9) == 0) {
      switch (value[0]){
        case 't':
         if(QiSOpt->f_type != CAA) QiSOpt->f_type = TABULAR;
         break;
        case 'd':
         if(QiSOpt->f_type != CAA) QiSOpt->f_type = DELIMITED;
         break;
        default:
       /* ignore anything else, not valid in cef types (use magic number) */
         break;
      }
      continue;
    }
    if(strncmp(QiToUpper(param), "COMMENT_MARKER", 14) == 0) {
      QicMark = value[0];
      continue;
    }
    if(strncmp(QiToUpper(param), "START_AFTER", 11) == 0) {
      QiSOpt->start_after = (char *) malloc(strlen(value)+1);
      strcpy(QiSOpt->start_after, value);
      continue;
    }
    if(strncmp(QiToUpper(param), "ATTRIBUTE_DELIMITER", 19) == 0) {
      QiSOpt->attr_delim = QiGetDelim(value);
      continue;
    }

    if(strncmp(QiToUpper(param), "DATA_DELIMITER", 14) == 0) {
      QiSOpt->type_guess = EXCHANGE;
      QiSOpt->data_delim = QiGetDelim(value);
      continue;
    }

    if(strncmp(QiToUpper(param), "END_OF_RECORD_MARKER", 20) == 0) {
      QiSOpt->type_guess = CAA;

      if (value == NULL) {
        QiSOpt->rec_end = '\n';
      }
      else{
        char * delimValue = QiStripQuotes(value);
        if( delimValue[0] == '\n' || delimValue[0] == '\0') {
        QiSOpt->rec_end = '\n';
        }
        else QiSOpt->rec_end = QiGetDelim(delimValue);
      }
      
      continue;

    }

    if(strncmp(QiToUpper(param), "RECORD_NUMBERING", 16) == 0) {
      if(strncmp("on", value, 2) == 0 && QiSOpt->f_type != CAA ){
        QiSOpt->rec_numbering = NUM_ON;
      }
      else
      {
        QiSOpt->rec_numbering = NUM_OFF;
      }
      continue;
    }

    if(strncmp(QiToUpper(param), "FILE_NAME", 9) == 0) {
      sprintf(message, "file: %s", value );
      QiDisplayMessage(message, QiSOpt);
      continue;
    }

    /* count number of variables */

    if(strncmp(QiToUpper(param), "START_VARIABLE", 14) == 0) {
       QiSCDF->vardata[QiSCDF->n_vars] = (QiCDFVariable *) QiMakeQiVariable();
       QiSCDF->vardata[QiSCDF->n_vars]->name = QiNewStr(value);
       QiSCDF->n_vars++;
       continue;
    }

    /* Ignore other header inputs */

  }  /* end while */

  if( Qifsetpos(NULL, fpRewind) == -1){
    QiDisplayMessage("In QiConfigureRead: Failed to reset position in file", QiSOpt);
  }

  if ( QiSCDF->n_vars == 0) return BAD_HEADER;

  return QMW_OK;


} /* end QiConfigureRead */

/*********************************************************************/

long QiReadVmeta(QiCDFContents *QiSCDF,    /* ptr to struct of contents */
                     QiOptions *QiSOpt,        /* options */
                     QiRecord_format *QiSfmt) /* ptr to record format struct */
{
 long n=0;
 char *header_line;
 long check=QMW_OK;
 char * param;
 char * value;
 long attr_num=0;
 fpos_t file_posn;
 int fpRewind;
 char param_ISO[]="SI_CONVERSION";
 char value_ISO[]="\"1>s\"";
 int SIc_found=0;

 while((header_line=QiReadLine(&(STR_NULL[0]) )) != NULL){

    check = QiParseParamValue(header_line, &param, &value);
    if (check != QMW_OK) continue;

    if(strncmp(QiToUpper(param), "START_DATA", 10) == 0) {
      if(QiSOpt->f_type == CAA){
        QiCEFmsg( 0, "START_DATA detected: being ignored ", " ", " ");
        continue;
      }
      else
        break; /* end of variable setup for cef 1*/
    }

    if(strncmp(QiToUpper(param), "DATA_UNTIL", 10) == 0) break; /* end of variable setup for cef 2*/

    /* read variable metadata */

    if(strncmp(QiToUpper(param), "START_VARIABLE", 14) == 0) {
       attr_num = 0;
	   SIc_found=0;
      QiSfmt->col_start[n] = -1; /* initialise for error trap */
      QiSfmt->col_width[n] = -1;
      QiSfmt->ignore[n] = 0;
      QiSfmt->ftpackets[n].timeformat = NOT_A_TIME; /* SJS re FT */
                                               /* safe initialisation */

      QiSCDF->vardata[n]->rec_vary = VARY; /* default in flat files */

      QiSCDF->vardata[n]->num_dims = 0; /* initialise default scalar */

       QiSCDF->vardata[n]->name = (char *)malloc(strlen(value)+1);
       strcpy(QiSCDF->vardata[n]->name, value);
       QiSCDF->vardata[n]->number = n;
       QiSfmt->n_items[n] = 1;

       /* find data type then return to same place in input file */

       if (Qifgetpos(&file_posn, &fpRewind) != 0) return FILE_READ_ERR;

       check = QiGetDataType(QiSCDF, &(QiSCDF->vardata[n]->sizeofentry),
                             QiSOpt, QiSfmt, n);
       if (Qifsetpos(&file_posn, fpRewind) != 0) return FILE_POSN_ERR;

       /* get variable metadata */

       while((header_line=QiReadLine(&(STR_NULL[0]))) != NULL){

         check = QiParseParamValue(header_line, &param, &value);
         if (check != QMW_OK) break;

         if(strncmp(QiToUpper(param), "END_VARIABLE", 12) == 0) {
           if(strcmp(value, QiSCDF->vardata[n]->name) != 0){
             return VAR_NAME_CHECK_ERR;
           }
           break;
         }

         if(strncmp(QiToUpper(param), "COLUMN_START", 12) == 0) {
           QiSfmt->col_start[n] = atol(value);
           QiCEFmsg( 0, "COLUNM_START detected for  ", QiSCDF->vardata[n]->name, ": being ignored");
           continue;
         }

         if(strncmp(QiToUpper(param), "COLUMN_WIDTH", 12) == 0) {
           QiCEFmsg( 0, "COLUNM_WIDTH detected for  ", QiSCDF->vardata[n]->name, ": being ignored");
           QiSfmt->col_width[n] = atol(value);
           continue;
         }
         if(strncmp(QiToUpper(param), "IGNORE_ON_READ", 14) == 0) {
           if (value[0] == 'y' || value[0] == 'Y') QiSfmt->ignore[n] = 1;
           QiCEFmsg( 0, "IGNORE_ON_READ flag detected for  ", QiSCDF->vardata[n]->name, ": being ignored");
           continue;
         }
         if(strncmp(QiToUpper(param), "MAXSTRLEN", 9) == 0) {
           QiSCDF->vardata[n]->num_elems = atol(value);
           QiCEFmsg( 0, "MAXSTRLEN detected for  ", QiSCDF->vardata[n]->name, " not in CEF 2 syntax");
           continue;
         }

         if(strncmp(QiToUpper(param), "DATA_TYPE", 9) == 0) {
           /* already done in first scan */
           continue;
         }

         if(strncmp(QiToUpper(param), "VALUE_TYPE", 10) == 0) {
           /* already done in first scan */
           continue;
         }

         if(strcmp(QiToUpper(param), "DATA") == 0) {
           /* NOVARY variable */
           QiSCDF->vardata[n]->rec_vary = NOVARY;
	       if( QiSCDF->vardata[n]->data_type == CDF_CHAR || QiSCDF->vardata[n]->data_type == CDF_UCHAR 
	   	 	|| QiSCDF->vardata[n]->data_type == ISO_TIME || QiSCDF->vardata[n]->data_type == ISO_TIME_RANGE){
	        // count actual text lengths
	           QiSCDF->vardata[n]->num_elems = maxEntryLength(value, QiSOpt->attr_delim ); // used for NRV only
            }

			if( QiReadVarData(QiSCDF->vardata, n, value, QiSfmt->n_items[n], QiSOpt) != QMW_OK) return SHORT_RECORD;
		
            continue;
         }

         if(strcmp(QiToUpper(param), "TIME_FORMAT") == 0) {
            QiCEFmsg( 0, "TIME_FORMAT descriptor detected for  ", QiSCDF->vardata[n]->name, " not in CEF 2 syntax");
            if(strncmp(value,"FREE_TIME_FORMAT", 16 ) == 0) {
                QiSfmt->ftpackets[n].timeformat = FREE_TIME_FORMAT;
            }
          else if(strncmp(value,"ISO", 3 ) == 0){
                QiSfmt->ftpackets[n].timeformat = ISO;
            }             /* Non-time vars don't have this, and carry */
                          /* initialised NOT_A_TIME value  */
                          /* SJS re FT */
            else{
            printf(" time format %s\n", value);
               return UNKNOWN_TIME_FORMAT;
            }
            continue;
         }
         if(strncmp(QiToUpper(param), "SIZES", 5) == 0) {
           /* already done in first scan */
           continue;
         }
		 
		 if(strncmp(QiToUpper(param), "SI_CONVERSION", 13) == 0) SIc_found = 1;
		  
		 /* else read attribute */

         check = QiText_to_Attr(QiSCDF, QiSOpt, param, value, n, attr_num);
         if(check != QMW_OK) QiDisplayMessage(QiErrStr(check), QiSOpt);
         attr_num++;

       } /* end while inside variable setup */


		if(QiSCDF->vardata[n]->data_type == ISO_TIME && SIc_found==0){
			/* ensure ISO_TIME has SI_CONVERSION */
			check = QiText_to_Attr(QiSCDF, QiSOpt, &(param_ISO[0]), &(value_ISO[0]), n, attr_num);
            attr_num++;
		}
		
       /* attr_num is now total number of attrs for var  */
       QiSCDF->vardata[n]->num_v_attrs = attr_num;
		
       /* finally increment variable number */
       n++;

    } /* end if on variable */

  } /* end while */

  /* after while loop, n should now be number of variables */

  if (QiSCDF->n_vars != n) return INCONSISTENT_HEADER;

  check = QiFTinit_parsers(QiSCDF, QiSfmt);  /* initalise parsers SJS re FT */

  return check;

} /* end QiReadVmeta */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiGetDataType
*
* PURPOSE: Scan variable metadata for data type and sizes only.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiGetDataType(QiCDFContents *QiSCDF,    /* ptr to struct of contents */
                     long * sizeofentry,     /* size of entry in data records */
                     QiOptions *QiSOpt,        /* options */
                     QiRecord_format *QiSfmt, /* ptr to record format struct */
                     long n)                 /* variable number */
{

 char * header_line;
 long check=QMW_OK;
 char * param;
 char * value;
 long found=0;
 long dim_num=0;
 char * tok;
 char attr_delim[2];
 int tokVal=0;

 attr_delim[0] = QiSOpt->attr_delim;
 attr_delim[1] = '\0';

    /* default */
    QiSCDF->vardata[n]->data_type = CDF_REAL8;
    *sizeofentry = sizeof(double);

       /* scan variable metadata for datatype */

       while((header_line=QiReadLine(&(STR_NULL[0]))) != NULL){

         check = QiParseParamValue(header_line, &param, &value);
         if (check != QMW_OK) break;

         if(strncmp(QiToUpper(param), "DATA_TYPE", 9) == 0 ||
          strncmp(QiToUpper(param), "VALUE_TYPE", 10) == 0) {

           QiSCDF->vardata[n]->data_type = QiDataType(value, sizeofentry);
           if (QiSCDF->vardata[n]->data_type == CDF_EPOCH || QiSCDF->vardata[n]->data_type == CDF_EPOCH ){
             /* take first epoch variable as default timeline */
             if( QiSCDF->d0_var_num == -1) QiSCDF->d0_var_num = n;
             if(QiSOpt->f_type == CAA) QiCEFmsg(1, " epoch time not allowed in CEF 2 ", " specify ISO_TIME ", " ");
           }
           if (QiSCDF->vardata[n]->data_type == ISO_TIME ){
             /* take first epoch variable as default timeline */
             if( QiSCDF->d0_var_num == -1) QiSCDF->d0_var_num = n;
             QiSCDF->vardata[n]->num_elems = ISO_TIME_LEN;
           }
           if (QiSCDF->vardata[n]->data_type == ISO_TIME_RANGE ){
             QiSCDF->vardata[n]->num_elems = 2 * ISO_TIME_LEN + 1;
           }
           if (QiSCDF->vardata[n]->data_type == CDF_CHAR ||  QiSCDF->vardata[n]->data_type == CDF_UCHAR){
           /* set default for CEF 2 syntax, CEF 1 will overwrite this */
           QiSCDF->vardata[n]->num_elems = MAX_STR_LEN_CEF2;
           }
           found++;
           if (found == 2) return QMW_OK;
           continue;
         }
         if(strncmp(QiToUpper(param), "SIZES", 5) == 0) {
           tok = strtok(value, attr_delim);
		   tokVal =  atol(tok);
		   if(tokVal > 1){
           	 QiSfmt->n_items[n] = QiSfmt->n_items[n] * tokVal;
           	 QiSCDF->vardata[n]->dim_sizes[dim_num] = tokVal;
           	 QiSCDF->vardata[n]->dim_varys[dim_num] = VARY;
			 dim_num++;
		   }
           while( (tok=strtok(NULL, attr_delim)) != NULL){
		     tokVal =  atol(tok);
			 if(tokVal > 1){
               if(dim_num >= CDF_MAX_DIMS) return MAX_DIM_ERROR;
               QiSfmt->n_items[n] = QiSfmt->n_items[n] * tokVal;
               QiSCDF->vardata[n]->dim_sizes[dim_num] = tokVal;
               QiSCDF->vardata[n]->dim_varys[dim_num] = VARY;
           	   dim_num++;
			 }
           }
           QiSCDF->vardata[n]->num_dims = dim_num;
           found++;
           if (found == 2) return QMW_OK;
           continue;
         }
         if(strncmp(QiToUpper(param), "END_VARIABLE", 12) == 0) {
           break;
         }

  } /* end while */


    return QMW_OK;

} /* end QiGetDataType */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiText_to_Attr
*
* PURPOSE: Read attribute from flat file.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Always returned at present
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiText_to_Attr( QiCDFContents * QiSCDF,
                     QiOptions * QiSOpt,
                     char * param,
                     char * value,
                     long var_num,
                     long attr_num)
{
 float *float_data;
 int *int_data;
 char * char_data;
 double *double_data;
 int *short_data;
 long j=0;
 long num_entries;
 char * tokens[MAX_ENTRIES];
 int j_ptr;
 char * textValue;

 long value_length = strlen(value)+1;

   QiSCDF->vardata[var_num]->attribute[attr_num].name =
        (char *) malloc(strlen(param)+1);
   strcpy(QiSCDF->vardata[var_num]->attribute[attr_num].name, param);
   QiSCDF->vardata[var_num]->attribute[attr_num].number = attr_num;

   if(QiSOpt->f_type == CAA){

     if(value[0] == '\"' && value[strlen(value)-1] == '\"' ){
        QiSCDF->vardata[var_num]->attribute[attr_num].data_type = CDF_CHAR;
     }
     else if( QiVarExists(QiSCDF, value) ){
       /* pointer attribute, record var name */
       QiSCDF->vardata[var_num]->attribute[attr_num].data_type = CDF_CHAR;
       QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  1;
       QiSCDF->vardata[var_num]->attribute[attr_num].data =
           (char *) malloc(strlen(value)+1);
       strcpy((char *)QiSCDF->vardata[var_num]->attribute[attr_num].data, value);
       return QMW_OK;
     }
     else if(QiSCDF->vardata[var_num]->data_type == ISO_TIME 
		|| QiSCDF->vardata[var_num]->data_type == CDF_EPOCH 
		|| QiSCDF->vardata[var_num]->data_type == CDF_EPOCH16){

       /* Time fields may take double type attributes, e.g. Delta_plus/minus */
       if( QiIsISOtime(value) ) QiSCDF->vardata[var_num]->attribute[attr_num].data_type = ISO_TIME;
       else QiSCDF->vardata[var_num]->attribute[attr_num].data_type = CDF_DOUBLE;

     }
     else if(QiSCDF->vardata[var_num]->data_type == ISO_TIME_RANGE ){

       /* Time range fields may take double type attributes, e.g. Delta_plus/minus */
       if( QiIsISOtimeRange(value) ) QiSCDF->vardata[var_num]->attribute[attr_num].data_type = ISO_TIME_RANGE;
       else QiSCDF->vardata[var_num]->attribute[attr_num].data_type = CDF_DOUBLE;

     }
     else if(QiIsISOtime(value)){
       QiSCDF->vardata[var_num]->attribute[attr_num].data_type = ISO_TIME;
     }
     else{
       /* non character data type */

        QiSCDF->vardata[var_num]->attribute[attr_num].data_type = QiSCDF->vardata[var_num]->data_type;

     }

     switch(QiSCDF->vardata[var_num]->attribute[attr_num].data_type)
     {
       case CDF_CHAR: case CDF_UCHAR:
       {
         num_entries = QiParseQuotedEntry(QiSOpt->attr_delim, value, &tokens[0]);
         QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  num_entries;

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                (void *) malloc(sizeof(char)*value_length);  /* more than long enough as quotes and ',' removed */
         char_data = (char *) QiSCDF->vardata[var_num]->attribute[attr_num].data;

         j_ptr = 0;
         for(j=0; j< num_entries; j++){
           textValue = QiStripQuotes(tokens[j]);
           strcpy(&(char_data[j_ptr]), textValue);
           j_ptr += strlen(textValue);
	       char_data[j_ptr++] = '\0';
         }
         break;
       }

        case ISO_TIME:
       {
          num_entries = QiParseEntry(QiSOpt->attr_delim, value, &tokens[0]);
          QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  num_entries;
          QiSCDF->vardata[var_num]->attribute[attr_num].data =
                (void *) malloc(sizeof(char)*value_length);  /* more than long enough as quotes and ',' removed */
           char_data = (char *) QiSCDF->vardata[var_num]->attribute[attr_num].data;

         j_ptr = 0;
           for(j=0; j< num_entries; j++){
           strcpy(&(char_data[j_ptr]), tokens[j] );
           j_ptr += strlen(tokens[j]);
	   char_data[j_ptr++] = '\0';
           }
           break;
       }

        case ISO_TIME_RANGE:
       {
          num_entries = QiParseEntry(QiSOpt->attr_delim, value, &tokens[0]);
          QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  num_entries;
          QiSCDF->vardata[var_num]->attribute[attr_num].data =
                (void *) malloc(sizeof(char)*value_length);  /* more than long enough as quotes and ',' removed */
           char_data = (char *) QiSCDF->vardata[var_num]->attribute[attr_num].data;

         j_ptr = 0;
           for(j=0; j< num_entries; j++){
           strcpy(&(char_data[j_ptr]), tokens[j] );
           j_ptr += strlen(tokens[j]);
	   char_data[j_ptr++] = '\0';
           }
           break;
       }

       case CDF_EPOCH:
       {
          num_entries = QiParseEntry(QiSOpt->attr_delim, value, &tokens[0]);
          QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  num_entries;

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                (void *) malloc(sizeof(double)*num_entries);
         double_data = (double *)
               QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
            double_data[j] =  QiISOStringToEpoch(tokens[j]);   /* no change SJS re FT */
                            /* Free Time doesn't cope with such attributes*/
         }
         break;
       }
       case CDF_EPOCH16:
       {
          num_entries = QiParseEntry(QiSOpt->attr_delim, value, &tokens[0]);
          QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  num_entries;

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                (void *) malloc(2*sizeof(double)*num_entries);
         double_data = (double *) QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
             QiISOStringToEpoch16(tokens[j], &(double_data[2*j]));   /* no change SJS re FT */
						/* Free Time doesn't cope with such attributes*/
         }
         break;
       }
       case CDF_DOUBLE: case CDF_REAL8:
       {
         num_entries = QiParseEntry(QiSOpt->attr_delim, value, &tokens[0]);
         QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  num_entries;

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                (void *) malloc(sizeof(double)*num_entries);
         double_data = (double *)
              QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
            double_data[j] =  atof(tokens[j]);
         }
         break;
       }
       case CDF_FLOAT: case CDF_REAL4:
       {
         num_entries = QiParseEntry(QiSOpt->attr_delim, value, &tokens[0]);
         QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  num_entries;

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                        (void *) malloc(sizeof(float)*num_entries);
         float_data = (float *)
            QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
            float_data[j] =  atof(tokens[j]);
         }
         break;
       }
         case CDF_INT4: case CDF_UINT4:
       {
         num_entries = QiParseEntry(QiSOpt->attr_delim, value, &tokens[0]);
         QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  num_entries;

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                        (void *) malloc(sizeof(int)*num_entries);
         int_data = (int *)
            QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
            int_data[j] =  atoi(tokens[j]);
         }
         break;
       }
         case CDF_INT2: case CDF_UINT2:
       {
         num_entries = QiParseEntry(QiSOpt->attr_delim, value, &tokens[0]);
         QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  num_entries;

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                        (void *) malloc(sizeof( int)*num_entries);
         short_data = ( int *)
            QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
            short_data[j] =  ( int) atoi(tokens[j]);
         }
         break;
       }
         case CDF_INT1: case CDF_UINT1: case CDF_BYTE:
       {
         num_entries = QiParseEntry(QiSOpt->attr_delim, value, &tokens[0]);
         QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  num_entries;

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                        (void *) malloc(sizeof(char)*num_entries);
         char_data = (char *)
             QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
            char_data[j] =  (char) atoi(tokens[j]);
         }
         break;
       }
       default:
       {
         return BAD_CDF_DATATYPE;
       }

     } /* end switch */


     return QMW_OK;

   } /* end if on CAA (CEF 2) format */

     /* end on CEF 2 type input */
     /****************************************************************************************/

   /* else use guesswork for CEF 1 and flat files */

   /* For attributes that take same data type as variable... */

   if (strncmp(param,"SCALEMIN",8) == 0 ||
       strncmp(param,"SCALEMAX",8) == 0 ||
       strncmp(param, "FILLVAL",7) == 0 ||
       strncmp(param,"VALIDMIN",8) == 0 ||
       strncmp(param,"VALIDMAX",8) == 0 )
   {
     QiSCDF->vardata[var_num]->attribute[attr_num].data_type =
           QiSCDF->vardata[var_num]->data_type;

     /* epoch is a special case */

     if(QiSCDF->vardata[var_num]->attribute[attr_num].data_type == CDF_EPOCH){
        QiSCDF->vardata[var_num]->attribute[attr_num].num_elems = 1;

        QiSCDF->vardata[var_num]->attribute[attr_num].data =
                (void *) malloc(sizeof(double));
        double_data = (double *)
              QiSCDF->vardata[var_num]->attribute[attr_num].data;
        double_data[0] =  QiISOStringToEpoch(value); /* no change SJS re FT */
                            /* Free Time doesn't cope with such attributes*/

     }
     else if(QiSCDF->vardata[var_num]->attribute[attr_num].data_type == CDF_EPOCH16){
        QiSCDF->vardata[var_num]->attribute[attr_num].num_elems = 1;

        QiSCDF->vardata[var_num]->attribute[attr_num].data =
                (void *) malloc(2*sizeof(double));
        double_data = (double *) QiSCDF->vardata[var_num]->attribute[attr_num].data;
        QiISOStringToEpoch16(value, &(double_data[0])); /* no change SJS re FT */
                            /* Free Time doesn't cope with such attributes*/

     }
     else{

      num_entries = QiParseEntry(QiSOpt->attr_delim, value, &tokens[0]);
      QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =
                num_entries;

     switch(QiSCDF->vardata[var_num]->attribute[attr_num].data_type)
     {
       case CDF_DOUBLE: case CDF_REAL8:
       {

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                (void *) malloc(sizeof(double)*num_entries);
         double_data = (double *)
              QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
            double_data[j] =  atof(tokens[j]);
         }
         break;
       }
       case CDF_FLOAT: case CDF_REAL4:
       {

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                        (void *) malloc(sizeof(float)*num_entries);
         float_data = (float *)
            QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
            float_data[j] =  atof(tokens[j]);
         }
         break;
       }
         case CDF_INT4: case CDF_UINT4:
       {

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                        (void *) malloc(sizeof(int)*num_entries);
         int_data = (int *)
            QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
            int_data[j] =  atoi(tokens[j]);
         }
         break;
       }
         case CDF_INT2: case CDF_UINT2:
       {

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                        (void *) malloc(sizeof( int)*num_entries);
         short_data = ( int *)
            QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
            short_data[j] =  ( int) atoi(tokens[j]);
         }
         break;
       }
         case CDF_INT1: case CDF_UINT1: case CDF_BYTE:
       {

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                        (void *) malloc(sizeof(char)*num_entries);
         char_data = (char *)
             QiSCDF->vardata[var_num]->attribute[attr_num].data;

         for(j=0; j< num_entries; j++){
            char_data[j] =  (char) atoi(tokens[j]);
         }
         break;
       }
        case CDF_CHAR: case CDF_UCHAR:
       {

         QiSCDF->vardata[var_num]->attribute[attr_num].data =
                        (void *) malloc(strlen(value) +1);
         char_data = (char *)
           QiSCDF->vardata[var_num]->attribute[attr_num].data;


         QiSCDF->vardata[var_num]->attribute[attr_num].num_elems = 1 ;

         strcpy(char_data, value);
         break;
       }
        case ISO_TIME:
       {
          QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  1;
          QiSCDF->vardata[var_num]->attribute[attr_num].data =
                (void *) malloc(sizeof(char)*(strlen(value)+1));  /* more than long enough as quotes and ',' removed */
           char_data = (char *) QiSCDF->vardata[var_num]->attribute[attr_num].data;

           strcpy(char_data, value );
           break;
       }
        case ISO_TIME_RANGE:
       {
          QiSCDF->vardata[var_num]->attribute[attr_num].num_elems =  1;
          QiSCDF->vardata[var_num]->attribute[attr_num].data =
                (void *) malloc(sizeof(char)*(strlen(value)+1));  /* more than long enough as quotes and ',' removed */
           char_data = (char *) QiSCDF->vardata[var_num]->attribute[attr_num].data;

           strcpy(char_data, value );
           break;
       }
       default:
       {
         return BAD_CDF_DATATYPE;
       }

     } /* end switch */

     } /* end if on epoch */

   }
   else
   {

     /* treat all other attributes as text strings */

     QiSCDF->vardata[var_num]->attribute[attr_num].data_type = CDF_CHAR;
     QiSCDF->vardata[var_num]->attribute[attr_num].num_elems = 1;

     QiSCDF->vardata[var_num]->attribute[attr_num].data =
        (char *) malloc(strlen(value)+1);
     strcpy((char *)QiSCDF->vardata[var_num]->attribute[attr_num].data,
        value);

   }

   return QMW_OK;

} /* end QiText_to_Attr */


/***********************************************************************/

long QiReadData(QiCDFContents *QiSCDF,
                QiRecord_format *QiSfmt,
                QiOptions *QiSOpt)
{
  long j=0;
  long j_recs;
  long n;
  char text[300];
  long n_vary=0;
  long NrecsToRead =0;
  long recOK;

  /* if the header is generic without number of data records count them */
   if( QiSCDF->n_recs == 0)  QiSCDF->n_recs = QiCountRecords( QiSOpt);
 
   if(QiSCDF->n_recs == 0) QiSCDF->n_recs = 1; /* catch case of file with only NRV data */
   
   NrecsToRead = QiSCDF->n_recs;
   if(QiSOpt->Nend != -1 && QiSOpt->Nend + 1 < NrecsToRead)  NrecsToRead = QiSOpt->Nend + 1;
   if(QiSOpt->Nstart > 0 ) NrecsToRead -= QiSOpt->Nstart;

   if( QiSCDF->n_recs == 0) {
     QiDisplayMessage("zero records to read, is record range requested within file?\n", QiSOpt);
     return QMW_OK; /* no data records but return OK */
   }
   if( NrecsToRead < 0) {
     QiDisplayMessage("no records to read, is record range requested within file?\n", QiSOpt);
     return QMW_WARNING; /* no data records but return OK */
   }
    /* allocate memory for data arrays */

  for (n=0; n < QiSCDF->n_vars; n++){

    if(QiSfmt->ignore[n] == 1) continue;  /* these will not be read */

    if(QiSCDF->vardata[n]->rec_vary == VARY){
      n_vary++;
      if( QiSOpt->f_type == TABULAR) QiSCDF->vardata[n]->num_elems = QiSfmt->col_width[n]-1;

      if( QiSCDF->vardata[n]->data_type == CDF_CHAR
       || QiSCDF->vardata[n]->data_type == CDF_UCHAR
       || QiSCDF->vardata[n]->data_type == ISO_TIME
       || QiSCDF->vardata[n]->data_type == ISO_TIME_RANGE){
										   
		 QiSCDF->vardata[n]->data = (void *)malloc(QiSfmt->n_items[n] *
                                           NrecsToRead *
                                           QiSCDF->vardata[n]->sizeofentry *
                                           (QiSCDF->vardata[n]->num_elems+1) );
      }
      else
      {
        QiSCDF->vardata[n]->num_elems = 1;
        QiSCDF->vardata[n]->data = (void *)malloc(QiSfmt->n_items[n] *
                                             NrecsToRead *
                                             QiSCDF->vardata[n]->sizeofentry);
      }
    }
  }/* end for over vars */

/* set end of data line marker (GLOBAL) */

  end_of_line_mark = QiSOpt->rec_end;

/* read data record at a time */
  if(n_vary > 0){
    for(j_recs=0; j_recs < QiSCDF->n_recs ; j_recs++){
	  recOK=0;
      if(QiSOpt->Nend >= 0 && QiSOpt->Nend < j_recs) break;

      switch (QiSOpt->f_type){

        case TABULAR:
        {
          if( QiReadRecsTabular(QiSCDF, QiSfmt, QiSOpt, j) == QMW_OK) recOK=1;
	  
          break;
        }
        case DELIMITED:
        {
          if( QiReadRecsParsed(QiSCDF, QiSfmt, QiSOpt, j) == QMW_OK) recOK=1;
          break;
        }
        case EXCHANGE:
        {
          if( QiReadRecsExchange(QiSCDF, QiSfmt, QiSOpt, j) == QMW_OK) recOK=1;
          break;
        }
        case CAA:
        {
          if( QiReadRecsCAA(QiSCDF, QiSfmt, QiSOpt, j) == QMW_OK) recOK=1;
          break;
        }
        default:
        {
           return QIE_FILE_TYPE_UNKNOWN;
        }
      } /* end switch */
      if(j_recs >= QiSOpt->Nstart  && recOK == 1) j++;
    } /* end for over records */
  }
  else j=1;  /* for no varying variable, data from header */

  if (j < NrecsToRead) {
     sprintf(text,"QIE Warning: only %ld records read OK out of %ld\n", j, QiSCDF->n_recs);
     QiDisplayMessage(text, QiSOpt);
  }

  QiSCDF->n_recs = j; /* ensure we only try to write actual number of records read */

  return QMW_OK;

} /* end  QiReadData  */

/***********************************************************************/

long QiReadRecsTabular(QiCDFContents *QiSCDF,
                     QiRecord_format *QiSfmt,
                     QiOptions *QiSOpt,
                     long j)
{
  long n;
  long m;
  long mm;
  float *float_data;
  float float_value;
  int *int_data;
  int int_value;
  char * char_data;
  double *double_data;
  double double_value;
  int *short_data;
  int short_value;
  long check = QMW_OK;
  char *rec_line;
  char * entry_ptr;
  char Epoch_str[EPOCH_WIDTH+1];
  char FreeTime_str[MAX_FT_LENGTH+1];   /* SJS re FT */
  size_t position;


   if( (rec_line=QiReadLine(QiSOpt->start_after)) != NULL){
      /* read entries for each variable from end of line (because we add \0) */

     for (n=QiSCDF->n_vars -1; n >= 0 ; n--){

       if(QiSCDF->vardata[n]->rec_vary == NOVARY) continue; /* skip non-rec varying  */

       if(QiSfmt->ignore[n] == 1) continue; /* skip vars flagged for skipping */

       mm = QiSfmt->n_items[n];  /* number of entries for each variable */

       /* count backwards through entries for var number n */

       for(m=mm-1; m >= 0; m--){
         /* set/reset pointer to start of variable entry */

         position = (size_t)(QiSfmt->col_start[n] + QiSfmt->col_width[n]*m);
         if (position >= strlen(rec_line)) return SHORT_RECORD;
         entry_ptr = &(rec_line[position]);

         switch(QiSCDF->vardata[n]->data_type)
         {
         case CDF_REAL4: case CDF_FLOAT:
         {
           float_data = (float *) QiSCDF->vardata[n]->data;
		 if(strlen(entry_ptr) > QiSfmt->col_width[n]) entry_ptr[QiSfmt->col_width[n]] = '\0'; /* terminate string */
           sscanf(entry_ptr, "%g", &float_value);
           float_data[mm*j+m] = float_value;
           break;
         }
         case CDF_DOUBLE: case CDF_REAL8:
         {
           double_data = (double *)QiSCDF->vardata[n]->data;
           if(strlen(entry_ptr) > QiSfmt->col_width[n]) entry_ptr[QiSfmt->col_width[n]] = '\0'; /* terminate string */
           sscanf(entry_ptr, "%lg", &double_value );
           double_data[mm*j+m] = double_value;
           break;
         }
         case CDF_EPOCH:
         {
           switch(QiSfmt->ftpackets[n].timeformat) {
               case FREE_TIME_FORMAT:           /* SJS re FT */
               {
				double_data = (double *)QiSCDF->vardata[n]->data;
                 if(strlen(entry_ptr) > QiSfmt->col_width[n]) entry_ptr[QiSfmt->col_width[n]] = '\0'; /* terminate string */
                 QiStrcpy(FreeTime_str, entry_ptr, MAX_FT_LENGTH);

                 if(QiSfmt->ftpackets[n].AllRecordsFlag ==  All_records_time_found){
                   check = QiFTCatTime_All(FreeTime_str, QiSfmt->ftpackets[n].AllRecordsTimeString);
                   if (check != QMW_OK) return check; /*cat checks max strlen */
                 }

                 if(strlen(FreeTime_str) != strlen(QiSfmt->ftpackets[n].FreeTimeFormatString) )
                             return FORMAT_LEN_NE_TIME_STR;
                 check = QiFTtoEPOCH(&(QiSfmt->ftpackets[n]),FreeTime_str);
                 if (check == QMW_OK){
                   double_data[mm*j+m] =  QiSfmt->ftpackets[n].epoch.tsince0;
                 } 
			  else {
				return check; }
                 break;
               }
               case ISO: default: {
                double_data = (double *)QiSCDF->vardata[n]->data;
                if(strlen(entry_ptr) > QiSfmt->col_width[n]) entry_ptr[QiSfmt->col_width[n]] = '\0'; /* terminate string */
                QiStrcpy(Epoch_str, entry_ptr, EPOCH_WIDTH);
                double_data[mm*j+m] =  QiISOStringToEpoch(Epoch_str);
                if((int)(double_data[mm*j+m]) == BAD_ISO_TIME_STR)
                            return BAD_ISO_TIME_STR;
                break;
               }
            }  /* end switch inside CDF_EPOCH */
            break;
         }
         case CDF_UINT1: case CDF_INT1: case CDF_BYTE:
         {
           char_data = (char *)QiSCDF->vardata[n]->data;
           if(strlen(entry_ptr) > QiSfmt->col_width[n]) entry_ptr[QiSfmt->col_width[n]] = '\0'; /* terminate string */
           sscanf(entry_ptr, "%u",  &int_value);
           char_data[mm*j+m] = int_value;
           break;
         }
         case ISO_TIME:
         {
           char_data = (char *) QiSCDF->vardata[n]->data;
           if(strlen(entry_ptr) > QiSfmt->col_width[n]) entry_ptr[QiSfmt->col_width[n]] = '\0'; /* terminate string */
           strcpy( &( char_data[(mm*j+m)*QiSCDF->vardata[n]->num_elems] ), entry_ptr);

           break;
         }
         case ISO_TIME_RANGE:
         {
           char_data = (char *) QiSCDF->vardata[n]->data;
           if(strlen(entry_ptr) > QiSfmt->col_width[n]) entry_ptr[QiSfmt->col_width[n]] = '\0'; /* terminate string */
           strcpy( &( char_data[(mm*j+m)*QiSCDF->vardata[n]->num_elems] ), entry_ptr);

           break;
         }
        case CDF_CHAR: case CDF_UCHAR:
         {
           char_data = (char *) QiSCDF->vardata[n]->data;
           if(strlen(entry_ptr) > QiSfmt->col_width[n]) entry_ptr[QiSfmt->col_width[n]] = '\0'; /* terminate string */
           strcpy( &( char_data[(mm*j+m)
                   *QiSCDF->vardata[n]->num_elems] ), entry_ptr);
           break;
         }
         case CDF_INT2: case CDF_UINT2:
         {
           short_data = (int *)QiSCDF->vardata[n]->data;
           if(strlen(entry_ptr) > QiSfmt->col_width[n]) entry_ptr[QiSfmt->col_width[n]] = '\0'; /* terminate string */
           sscanf(entry_ptr, "%d", &short_value);
           short_data[mm*j+m] = short_value;
           break;
         }
         case CDF_INT4: case CDF_UINT4:
         {
           int_data = (int *)QiSCDF->vardata[n]->data;
           if(strlen(entry_ptr) > QiSfmt->col_width[n]) entry_ptr[QiSfmt->col_width[n]] = '\0'; /* terminate string */
           sscanf(entry_ptr, "%d", &int_value);
           int_data[mm*j+m] = int_value;
           break;
         }
         default:
         {
            printf("QiReadRecsTabular: data type ??");
            fprintf(stderr, "QiReadRecsTabular: data type ??");
            return CDF_VAR_TYPE_ERR;
         }
         } /* end switch */

       } /* end for over dimensions mm*/

       QiSCDF->vardata[n]->max_rec_num = j;

     } /* end for over vars n */

     return QMW_OK;

  } /* end if */

  return QMW_ERROR; /* only get here if we run out of data */

} /* end QiReadRecsTabular */


/**************************************************************************/

long QiReadRecsParsed(QiCDFContents *QiSCDF,
                     QiRecord_format *QiSfmt,
                     QiOptions *QiSOpt,
                     long j)
{
  long n;
  long m;
  long mm;
  float *float_data;
  float float_value;
  int *int_data;
  int int_value;
  char * char_data;
  double *double_data;
  double double_value;
  int *short_data;
  int short_value;
  long check = QMW_OK;
  char * rec_line;
  char * entry_ptr;
  char * entry_ptr_next;
  char Epoch_str[EPOCH_WIDTH+1];
  char FreeTime_str[MAX_FT_LENGTH+1];   /* SJS re FT */
  long rec_in_file;
  char message[LINE_LEN];
  char delimstr[2];

  delimstr[0] = QiSOpt->data_delim;
  delimstr[1] = '\0';


  /* read records */

    if((rec_line=QiReadLine(&(QiSOpt->start_after[0]))) != NULL){

      /* as strtok gets used by ISOstring fns, NULL is not safe later, so.. */
      entry_ptr_next = rec_line;

      /* check record number if present */

      if( QiSOpt->rec_numbering == NUM_ON ){

      entry_ptr = strtok(entry_ptr_next, delimstr);
      if(entry_ptr == NULL) {
         sprintf(message,  "QIE Error: record line too short");
         QiDisplayMessage(message, QiSOpt);
         return SHORT_RECORD;
      }
      else{
        entry_ptr_next = &(entry_ptr[strlen(entry_ptr)+1]);
      }

        rec_in_file = atoi(entry_ptr);
        if(rec_in_file != j){
          sprintf(message,
              "QIE Warning: record number in file dubious, %ld", rec_in_file);
          QiDisplayMessage(message, QiSOpt);
        }
      }

     /* read entries for each variable */

     for (n=0; n < QiSCDF->n_vars; n++){
       if(QiSCDF->vardata[n]->rec_vary == NOVARY) continue; /* skip non-rec varying */

       mm = QiSfmt->n_items[n];

       for(m=0; m < mm; m++){

         /* increment ptr to next entry */
         entry_ptr = strtok(entry_ptr_next, delimstr);
      /* as strtok gets used by ISOstring fns, NULL is not safe later, so.. */
         if(entry_ptr == NULL) {
           sprintf(message,  "QIE Error: record line too short");
           QiDisplayMessage(message, QiSOpt);
            return SHORT_RECORD;
         }
         else{
           entry_ptr_next = &(entry_ptr[strlen(entry_ptr)+1]);
         }

         /* skip entry if var flag set as ignore */
         if(QiSfmt->ignore[n] == 1) continue;  /* we still loop over each m */

        switch(QiSCDF->vardata[n]->data_type)
        {
         case CDF_REAL4: case CDF_FLOAT:
         {
           float_data = (float *) QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%g", &float_value))
                  return SHORT_RECORD;
           float_data[mm*j+m] = float_value;
           break;
         }
         case CDF_DOUBLE: case CDF_REAL8:
         {
           double_data = (double *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%lg", &double_value ))
                  return SHORT_RECORD;
           double_data[mm*j+m] = double_value;
           break;
         }
         case CDF_EPOCH: {

           switch(QiSfmt->ftpackets[n].timeformat) {

               case FREE_TIME_FORMAT:               /* SJS re FT  */
               {
                 double_data = (double *)QiSCDF->vardata[n]->data;
                 QiStrcpy(FreeTime_str, entry_ptr, MAX_FT_LENGTH);
                 if(QiSfmt->ftpackets[n].AllRecordsFlag ==  /*cat all recs */
                                          All_records_time_found){
                   check = QiFTCatTime_All(FreeTime_str,
                      QiSfmt->ftpackets[n].AllRecordsTimeString);
                   if (check != QMW_OK) return check;/*cat checks max strlen */
                 }
                 if(strlen(FreeTime_str) !=
                     strlen(QiSfmt->ftpackets[n].FreeTimeFormatString) )
                             return FORMAT_LEN_NE_TIME_STR;
                 check = QiFTtoEPOCH(&(QiSfmt->ftpackets[n]),FreeTime_str);
                 if (check == QMW_OK){
                   double_data[mm*j+m] =  QiSfmt->ftpackets[n].epoch.tsince0;
                 } 
			  else {return check; }
                 break;
               }
               case ISO: default: {
                 double_data = (double *)QiSCDF->vardata[n]->data;
                 QiStrcpy(Epoch_str, entry_ptr, EPOCH_WIDTH);
                 double_data[mm*j+m] = QiISOStringToEpoch(Epoch_str);
                 if((int)(double_data[mm*j+m]) == BAD_ISO_TIME_STR)
                            return BAD_ISO_TIME_STR;
                 break;
               }
            }  /* end switch inside CDF_EPOCH */
            break;
         }
         case CDF_UINT1: case CDF_INT1: case CDF_BYTE:
         {
           char_data = (char *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%u",  &int_value))
                  return SHORT_RECORD;
           char_data[mm*j+m] = int_value;
           break;
         }
         case CDF_CHAR: case CDF_UCHAR:
         {
           char_data = (char *) QiSCDF->vardata[n]->data;
           if ((long)strlen(entry_ptr) > QiSCDF->vardata[n]->num_elems){
             sprintf(message,
            "QIE Warning: char data string > maxstrlen (%ld) specified in file",
             QiSCDF->vardata[n]->num_elems);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "String is: '%s'", entry_ptr);
             QiDisplayMessage(message, QiSOpt);
           }
           strcpy( &(char_data[(mm*j+m)*QiSCDF->vardata[n]->num_elems]),
                   entry_ptr);
           break;
         }
         case ISO_TIME:
         {
           char_data = (char *) QiSCDF->vardata[n]->data;
           if ((long)strlen(entry_ptr) > QiSCDF->vardata[n]->num_elems){
             sprintf(message,
                     "QIE Warning: char data string > maxstrlen (%ld) specified in file",
                      QiSCDF->vardata[n]->num_elems);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "String is: '%s'", entry_ptr);
             QiDisplayMessage(message, QiSOpt);
           }
           strcpy( &(char_data[(mm*j+m)*QiSCDF->vardata[n]->num_elems]),
                   entry_ptr);
           break;
         }
         case ISO_TIME_RANGE:
         {
           char_data = (char *) QiSCDF->vardata[n]->data;
           if ((long)strlen(entry_ptr) > QiSCDF->vardata[n]->num_elems){
             sprintf(message,
                     "QIE Warning: char data string > maxstrlen (%ld) specified in file",
                      QiSCDF->vardata[n]->num_elems);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "String is: '%s'", entry_ptr);
             QiDisplayMessage(message, QiSOpt);
           }
           strcpy( &(char_data[(mm*j+m)*QiSCDF->vardata[n]->num_elems]), entry_ptr);
           break;
         }
         case CDF_INT2: case CDF_UINT2:
         {
           short_data = (int *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%d", &short_value))
                  return SHORT_RECORD;
           short_data[mm*j+m] = short_value;
           break;
         }
         case CDF_INT4: case CDF_UINT4:
         {
           int_data = (int *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%d", &int_value))
                  return SHORT_RECORD;
           int_data[mm*j+m] = int_value;
           break;
         }
         default:
         {
            fprintf(stderr, "QiReadRecsParsed: data type ??");
            return CDF_VAR_TYPE_ERR;
         }
         } /* end switch */

      } /* end for over dimensions */

      QiSCDF->vardata[n]->max_rec_num = j;

     } /* end for over vars n */

     return QMW_OK;

  } /* end if */

  return QMW_ERROR; /* only get here if we run out of data */


} /* end QiReadRecsParsed */


 /**************************************************************************/

long QiReadRecsExchange(QiCDFContents *QiSCDF,
                     QiRecord_format *QiSfmt,
                     QiOptions *QiSOpt,
                     long j)
{
  long n;
  long m;
  long mm;
  float *float_data;
  float float_value;
  int *int_data;
  int int_value;
  char * char_data;
  double *double_data;
  double double_value;
  int *short_data;
  int short_value;
  long check = QMW_OK;
  char * rec_line;
  char * entry_ptr;
  char * entry_ptr_next;
  char Epoch_str[EPOCH_WIDTH+1];
  char FreeTime_str[MAX_FT_LENGTH+1];   /* SJS re FT */
  long rec_in_file;
  char message[LINE_LEN];
  char delimstr[2];

  delimstr[0] = QiSOpt->data_delim;
  delimstr[1] = '\0';

  /* read records */

    if((rec_line=QiReadLine(&(QiSOpt->start_after[0]))) != NULL){
      /* as strtok gets used by ISOstring fns, NULL is not safe later, so.. */
      entry_ptr_next = rec_line;

      /* check record number if present */

      if( QiSOpt->rec_numbering == NUM_ON ){

      entry_ptr = strtok(entry_ptr_next, delimstr);
      if(entry_ptr == NULL) {
         sprintf(message,  "QIE Error: record line too short");
         QiDisplayMessage(message, QiSOpt);
         return SHORT_RECORD;
      }
      else{
        entry_ptr_next = &(entry_ptr[strlen(entry_ptr)+1]);
      }

        rec_in_file = atoi(entry_ptr);
        if(rec_in_file != j){
          sprintf(message,
              "QIE Warning: record number in file dubious, %ld", rec_in_file);
          QiDisplayMessage(message, QiSOpt);
        }
      }

     /* read entries for each variable */

     for (n=0; n < QiSCDF->n_vars; n++){
       if(QiSCDF->vardata[n]->rec_vary == NOVARY) continue; /* skip non-rec varying */

       mm = QiSfmt->n_items[n];

       for(m=0; m < mm; m++){

         /* increment ptr to next entry */
         entry_ptr = strtok(entry_ptr_next, delimstr);
      /* as strtok gets used by ISOstring fns, NULL is not safe later, so.. */
         if(entry_ptr == NULL) {
           sprintf(message,  "QIE Error: record line too short");
           QiDisplayMessage(message, QiSOpt);
            return SHORT_RECORD;
         }
         else{
           entry_ptr_next = &(entry_ptr[strlen(entry_ptr)+1]);
         }

         /* skip entry if var flag set as ignore */
         if(QiSfmt->ignore[n] == 1) continue;  /* we still loop over each m */

        switch(QiSCDF->vardata[n]->data_type)
        {
         case ISO_TIME:
         {
           char_data = (char *) QiSCDF->vardata[n]->data;
           if ((long)strlen(entry_ptr) > QiSCDF->vardata[n]->num_elems){
             sprintf(message,  "QIE Warning: ISO data string too long for Qtran (>%ld)", QiSCDF->vardata[n]->num_elems);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "String is: '%s'", entry_ptr);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "for Var: %s", QiSCDF->vardata[n]->name);
             QiDisplayMessage(message, QiSOpt);
           }
           strcpy( &(char_data[(mm*j+m)*QiSCDF->vardata[n]->num_elems]), entry_ptr);
           break;
         }
         case ISO_TIME_RANGE:
         {
           char_data = (char *) QiSCDF->vardata[n]->data;
           if ((long)strlen(entry_ptr) > QiSCDF->vardata[n]->num_elems){
             sprintf(message,  "QIE Warning: ISO Time Range data string too long for Qtran (>%ld)", QiSCDF->vardata[n]->num_elems);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "String is: '%s'", entry_ptr);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "for Var: %s", QiSCDF->vardata[n]->name);
             QiDisplayMessage(message, QiSOpt);
           }
           strcpy( &(char_data[(mm*j+m)*QiSCDF->vardata[n]->num_elems]), entry_ptr);
           break;
         }
        case CDF_REAL4: case CDF_FLOAT:
         {
           float_data = (float *) QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%g", &float_value))
                  return SHORT_RECORD;
           float_data[mm*j+m] = float_value;
           break;
         }
         case CDF_DOUBLE: case CDF_REAL8:
         {
           double_data = (double *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%lg", &double_value ))
                  return SHORT_RECORD;
           double_data[mm*j+m] = double_value;
           break;
         }
         case CDF_EPOCH: {
           switch(QiSfmt->ftpackets[n].timeformat) {

               case FREE_TIME_FORMAT:               /* SJS re FT  */
               {
                 double_data = (double *)QiSCDF->vardata[n]->data;
                 QiStrcpy(FreeTime_str, entry_ptr, MAX_FT_LENGTH);
                 if(QiSfmt->ftpackets[n].AllRecordsFlag ==  /*cat all recs */
                                          All_records_time_found){
                   check = QiFTCatTime_All(FreeTime_str,
                      QiSfmt->ftpackets[n].AllRecordsTimeString);
                   if (check != QMW_OK) return check;/*cat checks max strlen */
                 }
                 if(strlen(FreeTime_str) !=
                     strlen(QiSfmt->ftpackets[n].FreeTimeFormatString) )
                             return FORMAT_LEN_NE_TIME_STR;
                 check = QiFTtoEPOCH(&(QiSfmt->ftpackets[n]),FreeTime_str);
                 if (check == QMW_OK){
                   double_data[mm*j+m] =  QiSfmt->ftpackets[n].epoch.tsince0;
                 } else {return check; }
                 break;
               }
               case ISO: default: {
                 double_data = (double *)QiSCDF->vardata[n]->data;
                 QiStrcpy(Epoch_str, entry_ptr, EPOCH_WIDTH);
                 double_data[mm*j+m] = QiISOStringToEpoch(Epoch_str);
                 if((int)(double_data[mm*j+m]) == BAD_ISO_TIME_STR)
                            return BAD_ISO_TIME_STR;
                 break;
               }
            }  /* end switch inside CDF_EPOCH */
            break;
         }
         case CDF_UINT1: case CDF_INT1: case CDF_BYTE:
         {
           char_data = (char *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%u",  &int_value))
                  return SHORT_RECORD;
           char_data[mm*j+m] = int_value;
           break;
         }
         case CDF_CHAR: case CDF_UCHAR:
         {
           char_data = (char *) QiSCDF->vardata[n]->data;
           if ((long)strlen(entry_ptr) > QiSCDF->vardata[n]->num_elems){
             sprintf(message,
            "QIE Warning: char data string > maxstrlen (%ld) specified in file",
             QiSCDF->vardata[n]->num_elems);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "String is:\n%s", entry_ptr);
             QiDisplayMessage(message, QiSOpt);
           }
           strcpy( &(char_data[(mm*j+m)*QiSCDF->vardata[n]->num_elems]),
                   entry_ptr);
           break;
         }
         case CDF_INT2: case CDF_UINT2:
         {
           short_data = (int *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%d", &short_value))
                  return SHORT_RECORD;
           short_data[mm*j+m] = short_value;
           break;
         }
         case CDF_INT4: case CDF_UINT4:
         {
           int_data = (int *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%d", &int_value))
                  return SHORT_RECORD;
           int_data[mm*j+m] = int_value;
           break;
         }
         default:
         {
            fprintf(stderr, "QiReadRecsExchange: data type for variable %ld??\n",n);
            return CDF_VAR_TYPE_ERR;
         }
         } /* end switch */

      } /* end for over dimensions */

      QiSCDF->vardata[n]->max_rec_num = j;

     } /* end for over vars n */

     return QMW_OK;

  } /* end if */

  return QMW_ERROR; /* only get here if we run out of data */


} /* end QiReadRecsExchange */
 /**************************************************************************/

long QiReadRecsCAA(QiCDFContents *QiSCDF,
                     QiRecord_format *QiSfmt,
                     QiOptions *QiSOpt,
                     long j)
{
  long n;
  long m;
  long mm;
  float *float_data;
  float float_value;
  int *int_data;
  int int_value;
  char * char_data;
  double *double_data;
  double double_value;
  int *short_data;
  int short_value;
  long check = QMW_OK;
  char * rec_line;
  char * entry_ptr;
  char * entry_ptr_next;
  char Epoch_str[EPOCH_WIDTH+1];
  char FreeTime_str[MAX_FT_LENGTH+1];   /* SJS re FT */
  char message[LINE_LEN];
  char delimstr[2];
  char *ptr;
  
  delimstr[0] = ',';
  delimstr[1] = '\0';


  /* read records */

    if((rec_line=QiReadLine(&(QiSOpt->start_after[0]))) != NULL){

      /* as strtok gets used by ISOstring fns, NULL is not safe later, so.. */
      entry_ptr_next = rec_line;

      /* check record number if present */
	  
	  if(strcmp(entry_ptr_next, QiSOpt->data_until) == 0 ) return QMW_WARNING;
	  
      if( QiSOpt->rec_numbering == NUM_ON ){
         sprintf(message,  "QIE Error: record numbering not allowed for CEF 2");
         QiDisplayMessage(message, QiSOpt);
         return CEF2_REC_NUM_ON;

      }

     /* read entries for each variable */

     for (n=0; n < QiSCDF->n_vars; n++){

       if(QiSCDF->vardata[n]->rec_vary == NOVARY) continue; /* skip non-rec varying */

       mm = QiSfmt->n_items[n];

       for(m=0; m < mm; m++){

         /* increment ptr to next entry */
         entry_ptr = strtok(entry_ptr_next, delimstr);

      /* as strtok gets used by ISOstring fns, NULL is not safe later, so.. */
         if(entry_ptr == NULL) {
           sprintf(message,  "QIE Error: record line too short");
           QiDisplayMessage(message, QiSOpt);
            return SHORT_RECORD;
         }
         else{
           entry_ptr_next = &(entry_ptr[strlen(entry_ptr)+1]);
         }

       /* remove leading white space */
       while(entry_ptr[0] == ' ' && entry_ptr[1] != '\0') entry_ptr = &(entry_ptr[1]);

       /* remove trailing white space */
       while(strlen(entry_ptr) != 0 && entry_ptr[strlen(entry_ptr)-1] == ' ') entry_ptr[strlen(entry_ptr)-1] = '\0';

         /* skip entry if var flag set as ignore */
         if(QiSfmt->ignore[n] == 1) continue;  /* we still loop over each m */


        switch(QiSCDF->vardata[n]->data_type)
        {
         case CDF_REAL4: case CDF_FLOAT:
         {
           float_data = (float *) QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%g", &float_value))
                  return SHORT_RECORD;
           float_data[mm*j+m] = float_value;
           break;
         }
         case CDF_DOUBLE: case CDF_REAL8:
         {
           double_data = (double *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%lg", &double_value ))
                  return SHORT_RECORD;
           double_data[mm*j+m] = double_value;
           break;
         }
         case CDF_EPOCH: {
           switch(QiSfmt->ftpackets[n].timeformat) {

               case FREE_TIME_FORMAT:               /* SJS re FT  */
               {
                 double_data = (double *)QiSCDF->vardata[n]->data;
                 QiStrcpy(FreeTime_str, entry_ptr, MAX_FT_LENGTH);
                 if(QiSfmt->ftpackets[n].AllRecordsFlag ==  /*cat all recs */
                                          All_records_time_found){
                   check = QiFTCatTime_All(FreeTime_str,
                      QiSfmt->ftpackets[n].AllRecordsTimeString);
                   if (check != QMW_OK) return check;/*cat checks max strlen */
                 }
                 if(strlen(FreeTime_str) !=
                     strlen(QiSfmt->ftpackets[n].FreeTimeFormatString) )
                             return FORMAT_LEN_NE_TIME_STR;
                 check = QiFTtoEPOCH(&(QiSfmt->ftpackets[n]),FreeTime_str);
                 if (check == QMW_OK){
                   double_data[mm*j+m] =  QiSfmt->ftpackets[n].epoch.tsince0;
                 } else {return check; }
                 break;
               }
               case ISO: default: {
                 double_data = (double *)QiSCDF->vardata[n]->data;
                 QiStrcpy(Epoch_str, entry_ptr, EPOCH_WIDTH);
                 double_data[mm*j+m] = QiISOStringToEpoch(Epoch_str);
                 if((int)(double_data[mm*j+m]) == BAD_ISO_TIME_STR)
                            return BAD_ISO_TIME_STR;
                 break;
               }
            }  /* end switch inside CDF_EPOCH */
            break;
         }
         case CDF_UINT1: case CDF_INT1: case CDF_BYTE:
         {
           char_data = (char *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%u",  &int_value))
                  return SHORT_RECORD;
           char_data[mm*j+m] = int_value;
           break;
         }
         case ISO_TIME:
         {
          char_data = (char *) QiSCDF->vardata[n]->data;
           if ((long)strlen(entry_ptr) > QiSCDF->vardata[n]->num_elems){
		   
             sprintf(message,  "QIE Warning: ISO data string too long for Qtran (>%ld)", QiSCDF->vardata[n]->num_elems);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "String is: '%s'", entry_ptr);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "for Var: %s", QiSCDF->vardata[n]->name);
             QiDisplayMessage(message, QiSOpt);
			 ptr = strstr(entry_ptr,"Z"); /* find end of time part if possible otherwise string will be empty*/
			 if (ptr != NULL && strlen(ptr) >= 0) ptr[1] = '\0'; /* truncate and try */
			 if( strlen(entry_ptr) > QiSCDF->vardata[n]->num_elems ) entry_ptr[QiSCDF->vardata[n]->num_elems+1] = '\0';
           }
		   strcpy( &(char_data[(mm*j+m)*QiSCDF->vardata[n]->num_elems]), entry_ptr); 
           break;
         }
		 
         case ISO_TIME_RANGE:
         {
           char_data = (char *) QiSCDF->vardata[n]->data;
           if ((long)strlen(entry_ptr) > QiSCDF->vardata[n]->num_elems){
             sprintf(message,  "QIE Warning: ISO Time Range data string too long for Qtran (>%ld)", QiSCDF->vardata[n]->num_elems);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "String is: '%s'", entry_ptr);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "for Var: %s", QiSCDF->vardata[n]->name);
             QiDisplayMessage(message, QiSOpt);
			 ptr = strstr(entry_ptr,"Z"); /* find end of time part if possible otherwise string will be empty*/
			 if (ptr != NULL && strlen(ptr) >= 0) ptr[1] = '\0'; /* truncate and try */
			 if( strlen(entry_ptr) > QiSCDF->vardata[n]->num_elems ) entry_ptr[QiSCDF->vardata[n]->num_elems+1] = '\0';
           }
		   strcpy( &(char_data[(mm*j+m)*QiSCDF->vardata[n]->num_elems]), entry_ptr); 
           break;
         }
         case CDF_CHAR: case CDF_UCHAR:
         {
           char_data = (char *) QiSCDF->vardata[n]->data;
           if ((long)strlen(entry_ptr) > QiSCDF->vardata[n]->num_elems){
             sprintf(message, "QIE Warning: char data string too long for Qtran (>%ld)", QiSCDF->vardata[n]->num_elems);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "String is: '%s'", entry_ptr);
             QiDisplayMessage(message, QiSOpt);
           }
           strncpy( &(char_data[(mm*j+m)*QiSCDF->vardata[n]->num_elems]),
                   QiStripQuotes(entry_ptr), QiSCDF->vardata[n]->num_elems);
           break;
         }
         case CDF_INT2: case CDF_UINT2:
         {
           short_data = (int *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%d", &short_value))
                  return SHORT_RECORD;
           short_data[mm*j+m] = short_value;
           break;
         }
         case CDF_INT4: case CDF_UINT4:
         {
           int_data = (int *)QiSCDF->vardata[n]->data;
           if( EOF == sscanf(entry_ptr, "%d", &int_value))
                  return SHORT_RECORD;
           int_data[mm*j+m] = int_value;
           break;
         }
         default:
         {
            fprintf(stderr, "QiReadRecsCAA: data type ??");
            return CDF_VAR_TYPE_ERR;
         }
         } /* end switch */

      } /* end for over dimensions */

      QiSCDF->vardata[n]->max_rec_num = j;

     } /* end for over vars n */


     return QMW_OK;

  } /* end if */

  return QMW_ERROR; /* only get here if we run out of data */


} /* end QiReadRecsCAA */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiParseParamValue
*
* PURPOSE: Parse input line from flat file.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Always returned at present
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiParseParamValue(char * line, char ** param, char ** value)
{
 int lastChar;

 *param  = strtok(line, "="); /* note we remove white space on param later */
  if(param == NULL) {
    return BAD_SYNTAX;
  }

  /* remove leading white space in param */
  while( (*param)[0] == ' ' || (*param)[0] == '\t') (*param)++;

  /* remove possible trailing white space in param */
  lastChar = strlen(*param)-1;
  while( (*param)[lastChar] == ' ' ||
         (*param)[lastChar] == '\t'   )
  {
    (*param)[lastChar] = '\0';
    lastChar = strlen(*param)-1;
  }

 *value  = strtok(NULL,"!\n");   /* keep white spaces in value but stop at !*/
  if(*value == NULL) {
    *value = STR_NULL;
    return QMW_OK;
  }
 /* remove leading white space in value */
  while((*value)[0] == ' ' || (*value)[0] == '\t' ) (*value)++;

 /* remove trailing white space in value */
  lastChar = strlen(*value)-1;
  while( (*value)[lastChar] == ' ' ||
         (*value)[lastChar] == '\t'   )
  {
    (*value)[lastChar] = '\0';
    lastChar = strlen(*value)-1;
  }

 /* replace empty RHS with single space */
 if((*value)[0] == '\0')  strcpy((*value), " ");

 return QMW_OK;

} /* end QiParseParamValue */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiParseEntry
*
* PURPOSE: Parse and count value entries in  line from flat file.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: Number of entries found
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiParseEntry(char delim,
                  char * value,
                  char ** tok)

{
 long j=0;
 char delimiter[2]; /* to be on the safe side use a string */

    delimiter[0] = delim;
    delimiter[1] = '\0';

    tok[0] = strtok(value, delimiter);
    if(tok[0] == NULL) return 0;

    if(tok[0][0] == '!') return 0;

    for(j=1;
        ( (tok[j] = strtok(NULL, delimiter) ) != NULL) && j < MAX_ENTRIES;
        j++){
      if(tok[j][0] == '!') return j;
    }

    return j;

 } /* end QiParseEntry */


/************************************************************************/

long QiParseQuotedEntry(char delim,
                        char * value,
                        char ** tok)

{
 long j=0;

    int ptr=0;
    int Quoted = 0;
    int InLength;

    tok[0] = &(value[0]);
    if(tok[0] == NULL) return 0;
    if(tok[0][0] == '!') return 0;

    j++;
    InLength =  strlen(value);

    for(ptr =0; ptr < InLength; ptr++){
      if(value[ptr] == '\"') {
        if(Quoted == 1) Quoted = 0;
      else Quoted = 1;
      continue;
      }

      if(!Quoted && value[ptr] == delim) {
        value[ptr] = '\0';
      tok[j] = &(value[ptr+1]);
      if(j == MAX_ENTRIES || tok[j][0] == '\0') break;
      j++;
      continue;
      }

      if(!Quoted && value[ptr] == '!') {
        value[ptr] = '\0';
        return j;
      }

    }

    return j;

 } /* end QiParseQuotedEntry */


/************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiReadGlobalFlat
*
* PURPOSE: Read global attribute from flat file
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Always returned at present
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiReadGlobalFlat( QiCDFContents *QiSCDF,   /* ptr to struct of contents */
                       QiOptions *QiSOpt,       /* i/o options */
                       char * attr_name)
{
   long n_attr;
   long n;
   long check;
   long sizeofentry = sizeof(char);
   long current_datatype = CDF_CHAR;
   QiGlobalAttribute new_attr;
   char * header_line;
   char * param;
   char * value;
   long entry_num=0;
   fpos_t entry;
   char * line;
   int fpRewind;

   new_attr.name = (char *) malloc( strlen(attr_name) * sizeof(char) + 1);
   strcpy(new_attr.name, attr_name);

   new_attr.num_entries = 0;
   new_attr.entry = NULL;         /* as trap */

  /* count number of entries then re-read Meta block */
  if( Qifgetpos(&entry, &fpRewind) == -1){
    QiDisplayMessage("In QiReadGlobalFlat: Failed to locate position in file", QiSOpt);;
  }

  line=QiReadLine(&(STR_NULL[0]));
  while(line != NULL ){

     check = QiParseParamValue(line, &param, &value);
     if (check == QMW_OK) {

       if(strncmp(QiToUpper(param), "ENTRY", 5) == 0)  new_attr.num_entries++;

       if(strncmp(QiToUpper(param), "END_META", 8) == 0) line = NULL;
       else line=QiReadLine( &(STR_NULL[0]));
     }
  }

  if( Qifsetpos (&entry, fpRewind) == -1){
    QiDisplayMessage("In QiReadGlobalFlat: Failed to reset position in file", QiSOpt);
  }

  new_attr.entry = ( QiGAttrEntry *)
                             QiMakeEntries( new_attr.num_entries);

   /* read the global block again getting entries */
   while((header_line = QiReadLine(&(STR_NULL[0]))) != NULL){

     /* writes line out to debug file or NULL */
     QiDisplayMessage(header_line, QiSOpt);

     check = QiParseParamValue(header_line, &param, &value);
     if (check == QMW_OK) {

       if(strncmp(QiToUpper(param), "END_META", 8) == 0){
          if(strcmp(attr_name, value) != 0) {
             return GLOBAL_NAME_CHECK_ERR;
          }
          break;
       }

       if(strncmp(QiToUpper(param), "G_ATTR_DATA_TYPE", 16) == 0 ||
          strncmp(QiToUpper(param), "VALUE_TYPE", 10) == 0){
          current_datatype = QiDataType(value, &sizeofentry);
          continue;
       }

       if(strncmp(QiToUpper(param), "ENTRY", 5) == 0){
          if( entry_num >= new_attr.num_entries) continue;

          (new_attr.entry+entry_num)->data_type = current_datatype;

          if( QiSetEntry(new_attr.entry + entry_num, value,
                            sizeofentry, QiSOpt->attr_delim) == QMW_OK){
            (new_attr.entry+entry_num)->exists = EXISTS;
            entry_num++;
          }
          else {
            new_attr.entry->exists = MISSING;
          }

          continue;
       }
     }
  }

   check = QiCreateGlobalAttr(QiSCDF, &new_attr, &n_attr);

   for (n=0; n<new_attr.num_entries; n++){
     if ((new_attr.entry+n)->data != NULL) free((new_attr.entry+n)->data);
   }
   if (new_attr.entry != NULL) free(new_attr.entry);
   if (new_attr.name != NULL) free(new_attr.name);

   return check;

} /* end  QiReadGlobalFlat  */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiDataType
*
* PURPOSE: Determine CDF data type equivalent to flat file data type and
*          set size of one entry.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: CDF data type
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiDataType( char * value,
                 long * sizeofentry)
{
 char * valueUpper;
 long datatype = CDF_DOUBLE;   /* default on unrecognised entry */
 int ptr;
 // strip out possible quotes
 if(value[strlen(value)-1] == '\"') value[strlen(value)-1] = '\0';
 ptr = 0;
 if(value[0] == '\"') ptr = 1;
 
   valueUpper = QiToUpper(&(value[ptr]));
           if(strncmp(valueUpper,"DOUBLE", 6) == 0)
           {
             datatype = CDF_DOUBLE;
             *sizeofentry = sizeof(double);
           }
           else if(strncmp(valueUpper,"FLOAT", 5) == 0)
           {
             datatype = CDF_FLOAT;
             *sizeofentry = sizeof(float);
           }
           else if(strncmp(valueUpper,"EPOCH", 5) == 0)
           {
             datatype = CDF_EPOCH;
             *sizeofentry = sizeof(double);
           }
           else if(strncmp(valueUpper,"ISO_TIME_RANGE", 14) == 0)
           {
		     /* done first as ISO_TIME is a substring */
			 
             datatype = ISO_TIME_RANGE;
             *sizeofentry = sizeof(char);
           }
           else if(strncmp(valueUpper,"ISO_TIME", 8) == 0)
           {
             datatype = ISO_TIME;
             *sizeofentry = sizeof(char);
           }
           else if(strncmp(valueUpper,"CHAR", 4) == 0)
           {
             datatype = CDF_CHAR;
             *sizeofentry = sizeof(char);
           }
           else if(strncmp(valueUpper,"INT", 3) == 0)
           {
             datatype = CDF_INT4;
             *sizeofentry = sizeof(long);
           }
           else if(strncmp(valueUpper,"BYTE", 4) == 0)
           {
             datatype = CDF_INT1;
             *sizeofentry = sizeof(char);
           }
           else if(strncmp(valueUpper,"SHORT", 5) == 0)
           {
             datatype = CDF_INT2;
             *sizeofentry = sizeof(int);
           }

  return datatype;

} /* End QiDataType   */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiSetEntry
*
* PURPOSE: Convert CDF attribute entry into QGAttrEntry entry.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK if executed successfully
*            otherwise error code
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiSetEntry( QiGAttrEntry *entry,
                char * value,
                long size,
                char delim)

{
  char * tokens[MAX_ENTRIES];
  long n;
  double * double_data;
  float * float_data;
  int * short_data;
  long * long_data;
  char * char_data;

  entry->data = NULL;  /* as a trap */


  switch(entry->data_type){
    case CDF_CHAR: case CDF_UCHAR:
    {
       /* remove quotes */
       char * stripped = QiStripQuotes(value);
       entry->num_elems = strlen(stripped) ;
       entry->data = (char *) malloc(size * (entry->num_elems+1));
       char_data = (char *) entry->data;
       strcpy(char_data, stripped);
       char_data[entry->num_elems] = '\0';
       break;
    }
    case ISO_TIME:
    {
       entry->num_elems = strlen(value) ;
       entry->data = (char *) malloc(size * (entry->num_elems+1));
       char_data = (char *) entry->data;
       strcpy(char_data, value);
       char_data[entry->num_elems] = '\0';
       break;
    }
    case ISO_TIME_RANGE:
    {
       entry->num_elems = strlen(value) ;
       entry->data = (char *) malloc(size * (entry->num_elems+1));
       char_data = (char *) entry->data;
       strcpy(char_data, value);
       char_data[entry->num_elems] = '\0';
       break;
    }
    case CDF_FLOAT: case CDF_REAL4:
    {
       entry->num_elems = QiParseEntry(delim, value, &tokens[0]);
       entry->data = malloc(size * entry->num_elems);
       float_data = (float *) entry->data;
       for (n=0; n < entry->num_elems; n++){
         float_data[n] = (float ) atof(tokens[n]);
       }
       break;
    }
    case CDF_DOUBLE: case CDF_REAL8:
    {
       entry->num_elems = QiParseEntry(delim, value, &tokens[0]);
       entry->data = malloc(size * entry->num_elems);
       double_data = (double *) entry->data;
       for (n=0; n < entry->num_elems; n++){
         double_data[n] = (double ) atof(tokens[n]);
       }
       break;
    }
    case CDF_INT4: case CDF_UINT4:
    {
       entry->num_elems = QiParseEntry(delim, value, &tokens[0]);
       entry->data = malloc(size * entry->num_elems);
       long_data = (long *) entry->data;
       for (n=0; n < entry->num_elems; n++){
         long_data[n] = (long ) atol(tokens[n]);
       }
       break;
    }
    case CDF_INT2: case CDF_UINT2:
    {
       entry->num_elems = QiParseEntry(delim, value, &tokens[0]);
       entry->data = malloc(size * entry->num_elems);
       short_data = (int *) entry->data;
       for (n=0; n < entry->num_elems; n++){
         short_data[n] = (int ) atoi(tokens[n]);
       }
       break;
    }
    case CDF_INT1: case CDF_UINT1: case CDF_BYTE:
    {
       entry->num_elems = QiParseEntry(delim, value, &tokens[0]);
       entry->data = malloc(size * entry->num_elems);
       char_data = (char *) entry->data;
       for (n=0; n < entry->num_elems; n++){
         char_data[n] = (char ) atoi(tokens[n]);
       }
       break;
    }
    case CDF_EPOCH:
    {
       /* at present we only handle single element epochs */
       /* this avoids parsing time strings with spaces */

       entry->num_elems = 1;
       entry->data =  malloc(size * entry->num_elems);
       double_data = (double *) entry->data;
       double_data[0] = (double ) QiISOStringToEpoch(value);
       break;
    }
    case CDF_EPOCH16:
    {
       /* at present we only handle single element epochs */
       /* this avoids parsing time strings with spaces */

       entry->num_elems = 1;
       entry->data =  malloc(2*size * entry->num_elems);
       double_data = (double *) entry->data;
       QiISOStringToEpoch16(value, &(double_data[0]));
       break;
    }
    default:
      return BAD_CDF_DATATYPE;

  }


  return QMW_OK;

} /* end QiSetEntry  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiFindVersion
*
* PURPOSE: Find file type and test version compatibilities
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK if executed successfully
*            otherwise error code
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long  QiFindVersion(QiOptions *QiSOpt)
{

 char header_line[LINE_LEN];
  fpos_t fPos;
  int fpRewind;
  char * param;
  char * value;
  long check;

  if( Qifgetpos(&fPos, &fpRewind)== -1){
    QiDisplayMessage("In QiFindVersion: Failed to locate position in file", QiSOpt);
  }

  if(fgets( &(header_line[0]), LINE_LEN, includedFiles[fpNow]) != NULL){
     if (strncmp(&(header_line[0]), "!@", 2) == 0){
       if ( strncmp(&(header_line[2]), QiSOpt->EXTN_QHD, 4) == 0){
         /*  header file, ignore */
       }
       else if ( strncmp(&(header_line[2]), QiSOpt->EXTN_QFT, 4) == 0){
         QiDisplayMessage("QIE magic number in file, file type is TABULAR",QiSOpt);
         QiSOpt->f_type = TABULAR;
         QiCEFmsg( 0, "version identifier found, but does not specify cef 2 ", " ", " ");
       }
       else if ( strncmp(&(header_line[2]), QiSOpt->EXTN_QFD, 4) == 0){
         QiDisplayMessage("QIE magic number in file, file type is DELIMITED",QiSOpt);
         QiSOpt->f_type = DELIMITED;
         QiCEFmsg( 0, "version identifier found, but does not specify cef 2 ", " ", " ");
       }
       else if ( strncmp(&(header_line[2]), QiSOpt->EXTN_CEF, 4) == 0 &&
                 header_line[6] == '1'){
         QiDisplayMessage("QIE magic number in file, in CSDS Cluster Exchange Format (CEF 1)",QiSOpt);
         QiSOpt->f_type = EXCHANGE;
         QiCEFmsg( 0, "version identifier found, but specifies cef 1 ", " ", " ");
       }
       else if ( strncmp(&(header_line[2]), QiSOpt->EXTN_CEF, 4) == 0 &&
                 header_line[6] == '2'){
         QiDisplayMessage("QIE magic number in file, in CAA Cluster Exchange Format (CEF 2)",QiSOpt);
         QiSOpt->f_type = CAA;
       }
       else {
         QiCEFmsg( 0, "version identifier found, but does not specify cef 2 ", " ", " ");
       }
     }

     /* QIE version number not checked as no compatibility problems yet */
  }
  
  /* look for CEF 2 identifier, FILE_FORMAT_VERSION */
  
  while(fgets( &(header_line[0]), LINE_LEN, includedFiles[fpNow]) != NULL){
    check = QiParseParamValue(header_line, &param, &value);
    if (check != QMW_OK) continue;

    if(strcmp(QiToUpper(param), "FILE_FORMAT_VERSION") == 0) {
      if( strstr(QiToUpper(value), "CEF-2") != NULL) QiSOpt->f_type = CAA;
      break;
    }
    /* only search header part */
    if(strcmp(QiToUpper(param), "START_DATA") == 0) break;
    if(strcmp(QiToUpper(param), "DATA_UNTIL") == 0) break;
  }

  if( Qifsetpos(NULL, fpRewind) == -1){
    QiDisplayMessage("In QiFindVersion: Failed to reset position in file", QiSOpt);
  }

  return QMW_OK;

} /* end QiFindVersion  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiSetVarToGet
*
* PURPOSE: sets ignore flags to ensure only one var fetched if var_to_get is set
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK if executed successfully
*            otherwise error code
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiSetVarToGet(QiCDFContents * QiSCDF,
                   QiRecord_format * QiSfmt,
                   char * var_to_get)
{
  long n;

  for (n=0; n<QiSCDF->n_vars; n++) {
    if( strcmp(var_to_get, (QiSCDF->vardata[n])->name) == 0
        || n == QiSCDF->d0_var_num) {
      QiSfmt->ignore[n] = 0;
    }
    else{
      QiSfmt->ignore[n] = 1;
    }
  }

  return QMW_OK;

} /* end QiSetVarToGet  */


/***********************************************************************/

char * QiToUpper(const char *string)
{

int i;
static char upper[LINE_LEN+1];
int s_length;

s_length = strlen(string);

if(s_length >= LINE_LEN) {
  char strTxt[50];
  sprintf(strTxt, "Input string too long = %d", s_length);
  QieAlertBox("warning",strTxt);
  QieAlertBox("note","increase LINE_LEN in qie.h and recompile Qtran/QSAS");
  s_length = LINE_LEN;
}

for (i=0; i < s_length; i++){
  upper[i]=toupper(string[i]);
}
upper[s_length] = '\0';

return &(upper[0]);
}

/****************************************************** Date parse */

long QiParseFreeTimeFormatString_date (
      char * ft_format_str,
      QiFTParser *ftparser
) {

char *substring_ptr;


/* initialization */

ftparser->n_found_date = 0;

/* start looking for key strings in date */

/* Year first. Try YYYY, then if no result try YY */
substring_ptr = strstr (ft_format_str, "YYYY");
   if (substring_ptr != NULL)
   {
      ftparser->found_date[ftparser->n_found_date] = YYYY_FTe;
      ftparser->date_start_in_FT[ftparser->n_found_date] =
      (int)(substring_ptr - ft_format_str);   /* pos is distance betw ptrs*/
      ftparser->date_width_in_FT[ftparser->n_found_date] = 4;
      (ftparser->n_found_date)++;
   } else {
   substring_ptr = strstr (ft_format_str, "YY");
      if (substring_ptr != NULL)
      {
        ftparser->found_date[ftparser->n_found_date] = YY_FTe;
        ftparser->date_start_in_FT[ftparser->n_found_date] =
        (int)(substring_ptr - ft_format_str);   /* pos is distance betw ptrs*/
        ftparser->date_width_in_FT[ftparser->n_found_date] = 2;
        (ftparser->n_found_date)++;
      }
   }  /* end year trials */

/* Try DOY. If no result, try month/day */
/* Use string "Doy" so can't clash with YYYY, e.g. "DOYYYYY" */

substring_ptr = strstr (ft_format_str, "Doy");
   if (substring_ptr != NULL)
   {
     ftparser->found_date[ftparser->n_found_date] = DOY_FTe;
     ftparser->date_start_in_FT[ftparser->n_found_date] =
      (int)(substring_ptr - ft_format_str);   /* pos is distance betw ptrs*/
     ftparser->date_width_in_FT[ftparser->n_found_date] = 3;
     (ftparser->n_found_date)++;

   } else {       /* not Doy so get [either MON or MO] AND DD  */

     substring_ptr = strstr (ft_format_str, "MON");
      if (substring_ptr != NULL)
      {
       ftparser->found_date[ftparser->n_found_date] = MON_FTe;
       ftparser->date_start_in_FT[ftparser->n_found_date] =
       (int)(substring_ptr - ft_format_str);   /* pos is distance betw ptrs*/
       ftparser->date_width_in_FT[ftparser->n_found_date] = 3;
       (ftparser->n_found_date)++;
      } else {
       substring_ptr = strstr (ft_format_str, "MO");
         if (substring_ptr != NULL){
          ftparser->found_date[ftparser->n_found_date] = MO_FTe;
          ftparser->date_start_in_FT[ftparser->n_found_date] =
         (int)(substring_ptr - ft_format_str); /* pos is distance betw ptrs*/
          ftparser->date_width_in_FT[ftparser->n_found_date] = 2;
         (ftparser->n_found_date)++;
         }
      }
          /* finished MON or MO; go try DD */

     substring_ptr = strstr (ft_format_str, "DD");
      if (substring_ptr != NULL)
      {
        ftparser->found_date[ftparser->n_found_date] = DD_FTe;
        ftparser->date_start_in_FT[ftparser->n_found_date] =
       (int)(substring_ptr - ft_format_str);   /* pos is distance betw ptrs*/
        ftparser->date_width_in_FT[ftparser->n_found_date] = 2;
        (ftparser->n_found_date)++;

      }  /* finished trying to get DD */


   } /* end else for getting [either MON or MO] AND DD */

return QMW_OK;

}  /* end QiParseFreeTimeFormatString_date  */

/******************************************************** Time parse */

long QiParseFreeTimeFormatString_time (
      char * ft_format_str,
      QiFTParser *ftparser
) {

char *substring_ptr;
int i;

/* key_time_str and rpt_time_chr moved to qie_ft.h */

/* initialization */

ftparser->n_found_time = 0;

/* start looking for key strings in time. This is easier than */
/* Date because we collect them all, so do in loop */

for(i=0; i < N_FT_TIME_STRINGS; i++){

substring_ptr = strstr (ft_format_str, key_time_str[i]);
if (substring_ptr != NULL)
{
  ftparser->found_time[ftparser->n_found_time] = (QiFTtimestrings_e)(i);
  ftparser->time_start_in_FT[ftparser->n_found_time] =
      (int)(substring_ptr - ft_format_str);   /* pos is distance betw ptrs*/
  ftparser->time_width_in_FT[ftparser->n_found_time] =
           strspn((substring_ptr+1),rpt_time_chr[i]) + 1;
  (ftparser->n_found_time)++;
} /* end if */

} /* end for */

return QMW_OK;

} /* end QiParseFreeTimeFormatString_time */



/********************************************** Copy & calc msecs factors */

long QiFTCpyCalcmsecsFactors( QiFTParser *ftparser,
                               double t2ms_factors[] )
{

/* copy msecs per hour, min, etc. into array and then multiply */
/* by 10^-(width of input field) FOR ALL FRACTIONAL BITS*/

/* uses constant time_2_msecs_factors array defined at top of file */

int i;
double ten=10;

for(i=0; i < ftparser->n_found_time; i++)
{
   t2ms_factors[i] = time_2_msecs_factors[ftparser->found_time[i]];
   if(ftparser->found_time[i] >= d_FTe) t2ms_factors[i] *=
                        pow(ten , -ftparser->time_width_in_FT[i]);
}

return QMW_OK;

} /* end  QiFTCpyCalcmsecsFactors */



/******************************************* Apply Parser to create epoch */

long QiFTtoEPOCH ( QiFTpacket *ftpackp, char *ftstr)
{

/* convert timestring pointed to by ftstr to year, month, day , millisecs */
/* Then convert the result to epoch */

long      year, month, day;
double    msecs;
long      check;

check = QiFTGetDate(&(ftpackp->ft_parser_s),ftstr, &year, &month, &day);
                               /* set year, month, day using parser */

if (check != QMW_OK) return check;

check = QiFTGetMsecs(ftpackp,ftstr,&msecs); /* set milliseconds */

#ifdef DEBUG_ME
   printf ("Time string is\nPos 0123456789012345678901234567890\n    %s\n",ftstr);
   printf ("Resulting Date: y = %d, m = %d, d = %d\n",year,month,day);
   printf ("Resulting Msecs = %20.12g\n",msecs);
#endif

if (check != QMW_OK) return check;  /* at present GetMsecs always returns ok*/


(ftpackp->epoch).tsince0 = QiFTComputeEpoch(year,month,day,msecs);


return QMW_OK;
}


/******************************************* Get Date */

long QiFTGetDate(QiFTParser *ftparser,
                 char * ftstr,
                 long *yp,
                 long *mp,
                 long *dp)
{

int n_date_items, n_in_list,i;
char temp_str[MAX_FT_LENGTH];
long doy;
char month_str[4];
long check;

*yp = 0;
*mp  = 1; *dp = 1;   /* initialise to 1 Jan 0000 <=> tsince0=0 */

n_date_items = ftparser->n_found_date;
if (n_date_items == 0) return QMW_OK;

/* so there are some date items to work with */
/* if the year is given, it will be the first item, either YYYY or YY */

n_in_list = 0;

switch (ftparser->found_date[n_in_list]) {
   case YYYY_FTe:
       strcpy(temp_str, "");   /* initialise to null string */
       strncat(temp_str, (ftstr + ftparser->date_start_in_FT[n_in_list]),
                       ftparser->date_width_in_FT[n_in_list] );
       *yp = atol(temp_str);
       /* trap string with no digit, NOTE 0 may be a valid return */
       if(*yp == 0 && (strchr(temp_str, '0') == NULL) ) return BAD_YEAR_STR;

       n_in_list++;
       break;

   case YY_FTe:
       strcpy(temp_str, "");   /* initialise to null string */
       strncat(temp_str, (ftstr + ftparser->date_start_in_FT[n_in_list]),
                       ftparser->date_width_in_FT[n_in_list] );
       *yp = atol(temp_str);
       /* trap string with no digit, NOTE 0 may be a valid return */
       if(*yp == 0 && (strchr(temp_str, '0') == NULL) ) return BAD_YEAR_STR;

       /* trap century boundaries in YY monotonic data */
       if(monotonic == 1){  /* force monotonic */
         if(yp_last > *yp){
           QieAlertBox("Warning", "Century boundary detected");
           year_offset+=100;
         }
         yp_last = *yp;
       }

       if(year_offset > 0) {
         *yp += year_offset;
       }
       else {
           QieAlertBox("Error", "No year offset for 2 digit year");
	     return NO_YEAR_OFFSET;
	 }

       n_in_list++;
       break;

    default:
       break;
}

if (n_in_list == n_date_items) return QMW_OK; /* only year provided */
if (n_in_list > n_date_items) return OVER_RAN_DATE_ITEM_LIST;

/* See if DOY given; if so, convert to mm,dd and return */

if ( ftparser->found_date[n_in_list] == DOY_FTe)
{
       strcpy(temp_str, "");   /* initialise to null string */
       doy = atol( strncat(temp_str,
                   (ftstr + ftparser->date_start_in_FT[n_in_list]),
                       ftparser->date_width_in_FT[n_in_list] ));
       check = QiYearDoyToDate (*yp, doy, mp, dp);
       n_in_list++;
       if (n_in_list == n_date_items) /* this MUST be true, but check anyway*/
       {    return check;
       } else if (n_in_list > n_date_items) {
            return OVER_RAN_DATE_ITEM_LIST;
       } else { return DATE_COMPLETE_BUT_LIST_NOT_EXHAUSTED;}
}

/* so we need to find month and day. month should be next item */

switch (ftparser->found_date[n_in_list]) {
   case MON_FTe:

       strcpy(month_str, "");  /* initialise to null string */
       strncat(month_str,
                   (ftstr + ftparser->date_start_in_FT[n_in_list]),
                       ftparser->date_width_in_FT[n_in_list] );
       n_in_list++;
       make_uppercase(month_str);   /* compare strings in caps */
       *mp = -1;
       for(i=1; (i<13) && (*mp < 0); i++)
       {  if (strncmp(month_str,monthnames[i],3) == 0) *mp = i;
       }   /* end for */
       if (*mp < 0 )   /* not found; maybe July used JLY and not JUL */
       {  if (strncmp(month_str,"JLY",3) == 0)
          { *mp = 7; } else { return BAD_MONTH_STRING; }
       }
       break;

    case MO_FTe:

       strcpy(temp_str, "");    /* initialise to null string  */
       *mp = atol( strncat(temp_str,
                   (ftstr + ftparser->date_start_in_FT[n_in_list]),
                       ftparser->date_width_in_FT[n_in_list] ));

       if (*mp < 1 || *mp > 12) return MONTH_OUT_OF_RANGE;
       n_in_list++;
       break;

    default:
       /* return MONTH_NEEDED_NOT_FOUND; */ /* if no month, def is already set*/
       break;

}  /* end switch for month */

if (n_in_list == n_date_items)
 /*{ return LIST_EXHAUSTED_BUT_DAY_OF_MONTH_REQUIRED; }*/ /* def set already*/
   { return QMW_OK; }
else if  (n_in_list > n_date_items)
   {   return OVER_RAN_DATE_ITEM_LIST;}

/* just need to get day */

strcpy(temp_str, "");    /* initialise to null string  */
*dp = atol( strncat(temp_str,
            (ftstr + ftparser->date_start_in_FT[n_in_list]),
                       ftparser->date_width_in_FT[n_in_list] ));
if (*dp < 1 || *dp > 31) return DAY_OUT_OF_RANGE;
n_in_list++;
if (n_in_list == n_date_items)
   { return QMW_OK; }
else if  (n_in_list > n_date_items)
   {   return OVER_RAN_DATE_ITEM_LIST;}
else return QMW_ERROR;
}




/******************************************* Get Msecs */

long QiFTGetMsecs( QiFTpacket *ftpackp,
                   char       *ftstr,
                   double     *msecs)
{
/* Grab all the time elements from ftstr, use pre-calculated factors
*  to convert to milliseconds, and collect in msecs
*  Times are extracted from string using atof, which always returns
*  a value; NO ERROR checking is done on individual time elements being
*  within range, e.g. could have 127 hours.*/

int i;
QiFTtimestrings_e *foundtimes;
double time_part, *t2msecs;
char temp_str[MAX_FT_LENGTH];
QiFTParser *ftparser;

*msecs = 0;


/* set ptrs to parser, found elements, and  conv factors */

ftparser = &(ftpackp->ft_parser_s);
foundtimes = ftparser->found_time;
t2msecs = ftpackp->time2msecs_factors_adjusted;

#ifdef DEBUG_ME
  printf(" i fdtim stpos width     factor      time_part  +msecs \n\n");
#endif

for(i=0; i<ftparser->n_found_time; i++)
{
strcpy(temp_str,"");
strncat(temp_str,  (ftstr + ftparser->time_start_in_FT[i]),
                       ftparser->time_width_in_FT[i] );

time_part = atof(temp_str);

#ifdef DEBUG_ME
  printf ("%2d %3d %4d %4d %12.8g %15.8g %15.8g\n",i,foundtimes[i],
     ftparser->time_start_in_FT[i],ftparser->time_width_in_FT[i],
     t2msecs[i],time_part,time_part*t2msecs[i]);
#endif

*msecs += time_part * t2msecs[i];

}

return QMW_OK;
}




/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++  QiYearDoyToDate  ++
***
***/

/* from Qan routine of same name dated June 1996 , with return vals added
*  and all args as longs */

long QiYearDoyToDate( long year, long doy, long* month, long* day )
{
  int i, leap;
  leap = ( year%4 == 0 && year%100 != 0 || year%400 == 0 );
/*
** check that doy is in range for supplied year
*/
  if( leap && doy > 366 || !leap && doy > 365 || doy < 1 )
  {
    *month = -1; *day = -1;
    return  BAD_DOY_TO_DATE;
  }

  for(i=1; doy > qie_days_in_month[leap][i]; i++ )
     doy -= qie_days_in_month[leap][i];
  *month = i;
  *day = doy;

  return QMW_OK;
}


/* +++++++++++++++++++++++++++++++++++++++++++++++++++++++  QiFTComputeEpoch  ++
*** Taken from the following dated June 1996
*** QanComputeEpoch: source taken from CDF 2.4.11 source (epochu.c)
***/
/******************************************************************************
* Local function prototypes.
******************************************************************************/

static long JulianDay( long year, long month, long day );

/******************************************************************************
* JulianDay.
*    The year, month, and day are assumed to have already been validated.  This
* is the day since 0 AD (julian day may not be the proper term).
******************************************************************************/

static long JulianDay( long y, long m, long d )
{
  long jd;
  jd = (long)
       (367*y-7*(y+(m+9)/12)/4-3*((y+(m-9)/7)/100+1)/4+275*m/9+d+1721029);
  return (jd);
}

/******************************************************************************
* computeEPOCH.
*   Computes (and returns) an EPOCH value based on its component parts.  -1.0
* is returned if an illegal component part is detected.
double computeEPOCH (year, month, day, hour, minute, second, msec)
long year, month, day, hour, minute, second, msec;
******************************************************************************/
/* For FT use, only need year, month, day and msecs so this routine does
*  not use hours, minutes, and seconds. Also FT routines use longs for all
*  integer values and msecs is a double*/


double QiFTComputeEpoch( long year,
                         long month,
                         long day,
                         double msec)

{
  double epoch;

  if ((day < 1 || day > 31) ||
      (month < 1 || month > 12) ||
      (year < 0 || year > 9999))
    epoch = -1.0;
  else {
    epoch = (double) JulianDay (year, month, day) - 1721060;
    epoch = epoch * 86400000.0 + msec;
  }
  return epoch;
}

/********************************* ConCatTime_All_Records */

long QiFTCatTime_All ( char *record,
                       char *all )
{

/* this is just strcat but is used to concatenate both the format
*  strings and also the time strings to ensure they are put together
*  in exactly the same way */

/* the input string record is replaced by the concatenated string;
*  the input string all    is left untouched (as it is needed again!) */

char separator[2] = " ";  /* insert separator between strings */

/* make sure there's enough room for both strings, separator, and terminator */

if ( (int)(strlen(record) + strlen(all) + 2) > MAX_FT_LENGTH)
   return FT_STRINGS_TOO_LONG_TO_CAT;

strcat( strcat(record,separator), all);

return QMW_OK;
}


/****************************************** FTinit_parsers   */


long QiFTinit_parsers(QiCDFContents* QiSCDF,
                      QiRecord_format* QiSfmt)
{

long i,iatt,check;
QiFTpacket* ftpackp;
QiFTParser* ftparser;


/* step through vars and initialise ftparsers for all time variables */

for (i = 0; i< QiSCDF->n_vars; i++){

int ierr_allrecstime = 0;    /* initialise counters */
int ierr_allrecsformat = 0;
int ierr_allrecsfound = 0;
int ierr_freetimeformat = 0;

                                             /* ignore non time vars */
if(QiSCDF->vardata[i]->data_type != CDF_EPOCH) continue;

/* We have a time variable. Is it a Free Time one? */
if(QiSfmt->ftpackets[i].timeformat != FREE_TIME_FORMAT) continue;

/* Yes. Assign pointers for local use */
ftpackp = &(QiSfmt->ftpackets[i]);
ftparser = &(ftpackp->ft_parser_s);

/* Need to:
*
*  1 Set ALLRecordsFlag, AllRecordsFormatString, AllRecordsTimeString
*      and FreeTimeFormatString in ftpackp from attributes
*  2 Parse concatenated Format string and build data/time parser info
*  3 Copy and adjust time2msecs factors
*/

ftpackp->AllRecordsFlag = No_all_records_time;  /* initialise to none */

for(iatt=0; iatt< QiSCDF->vardata[i]->num_v_attrs; iatt++){
 if(strncmp(QiSCDF->vardata[i]->attribute[iatt].name,"All_records_time", 15)
     == 0)  {
     if((int)(strlen((char *)QiSCDF->vardata[i]->attribute[iatt].data))< MAX_FT_LENGTH){
        strcpy(ftpackp->AllRecordsTimeString,
                 (char *)QiSCDF->vardata[i]->attribute[iatt].data);
     } else {
     return ALL_RECORDS_STRING_TOO_LONG;
     }
     ftpackp->AllRecordsFlag = All_records_time_found;
     ierr_allrecstime++;
     ierr_allrecsfound++;
 } else if
    (strncmp(QiSCDF->vardata[i]->attribute[iatt].name,"All_records_format", 15)
     == 0)  {
     if((int)(strlen((char *)QiSCDF->vardata[i]->attribute[iatt].data))< MAX_FT_LENGTH){
        strcpy(ftpackp->AllRecordsFormatString,
                 (char *)QiSCDF->vardata[i]->attribute[iatt].data);
     } else {
     return ALL_RECORDS_STRING_TOO_LONG;
     }
     ftpackp->AllRecordsFlag = All_records_time_found;
     ierr_allrecsformat++;
     ierr_allrecsfound++;
 } else if
    (strncmp(QiSCDF->vardata[i]->attribute[iatt].name,"TIME_FORMAT_STRING", 15)
     == 0)  {
     if((int)(strlen((char *)QiSCDF->vardata[i]->attribute[iatt].data))< MAX_FT_LENGTH){
        strcpy(ftpackp->FreeTimeFormatString,
                 (char *)QiSCDF->vardata[i]->attribute[iatt].data);
     } else {
     return ALL_RECORDS_STRING_TOO_LONG;
     }
     ierr_freetimeformat++;
 } else {
   continue;    /* if not a relevant attribute, skip */
 }
} /* end for over attributes*/

/* error check that things are set correctly. Should have:
*  ierr_freetimeformat = 1
*  ierr_allrecsfound = 0 or 2*/

if(ierr_freetimeformat !=1) return FREE_TIME_FORMAT_STRING_NOT_FOUND;
switch(ierr_allrecsfound)
{
    case 0: {
       if ( (ierr_allrecsformat !=0) && (ierr_allrecstime !=0) )
           return ERROR_COUNTING_ALL_RECS_ATTRIBUTES;
       if (ftpackp->AllRecordsFlag != No_all_records_time)
           return ERROR_COUNTING_ALL_RECS_ATTRIBUTES;
       break;
    }
    case 2: {
       if ( (ierr_allrecsformat !=1) && (ierr_allrecstime !=1) )
           return ERROR_COUNTING_ALL_RECS_ATTRIBUTES;
       if (ftpackp->AllRecordsFlag != All_records_time_found)
           return ERROR_COUNTING_ALL_RECS_ATTRIBUTES;
       if ((int)(strlen(ftpackp->FreeTimeFormatString))> MAX_FT_LENGTH)
           return FREE_TIME_FORMAT_STRING_ERROR;
       if ((int)(strlen(ftpackp->AllRecordsFormatString))> MAX_FT_LENGTH)
           return FREE_TIME_FORMAT_STRING_ERROR;
       if (strlen(ftpackp->AllRecordsFormatString) !=
                         strlen(ftpackp->AllRecordsTimeString) )
           return INCOMPATIBLE_ALL_RECS_FORMAT_AND_STRING;
       break;
    }

    case 1: default: {
       return ALL_RECS_COMPONENT_MISSING;
       break;
    }
}   /* end switch */

/* if all records time, then concatenate format strings */

if (ftpackp->AllRecordsFlag == All_records_time_found){
    check = QiFTCatTime_All(ftpackp->FreeTimeFormatString,
                              ftpackp->AllRecordsFormatString);
    if (check != QMW_OK) return check;
}

/* build data and time parsing elements */

check = QiParseFreeTimeFormatString_date(ftpackp->FreeTimeFormatString,
                                             ftparser);
if (check != QMW_OK) return check;
check = QiParseFreeTimeFormatString_time(ftpackp->FreeTimeFormatString,
                                             ftparser);
if (check != QMW_OK) return check;

/* Calc time2msecs factors */
check = QiFTCpyCalcmsecsFactors(ftparser,ftpackp->time2msecs_factors_adjusted);
if (check != QMW_OK) return check;

} /* end for over variables */

return QMW_OK;

}

/**************************************** make string uppercase  */

void make_uppercase(char *string)
{

int i;

for (i=0; i< (int)(strlen(string)); i++)
  {
    string[i]=toupper(string[i]);
  }

return;
}

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiReadLine
*
* PURPOSE: read line from file and validate
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUE:  QMW_OK if valid line found
*            EOF at end of file
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

char * QiReadLine(char *skip_until){

 int i, n=0;
 long check;
 char * param;
 char * value;
 char * ptr;
 char * local_ptr;
 FILE *fp;
 static char * newLine = NULL;
 char * partLine = NULL;
 int parentFile;

 if(newLine != NULL) {
   free(newLine);
   newLine = NULL;
 }

fp = includedFiles[fpNow];

 /* global QicMark in qie.h */

 /* read file until valid full line or record found */

 while(fp != NULL){
   if( (ptr=QiGetLine(fp)) == NULL) {

     /* end of this file, back up to parent file */
     rewind(fp);
     parentFile = includedFrom[fpNow];
     fpNow = parentFile;
     if(fpNow >=0) fp = includedFiles[fpNow];
     else return NULL; /* finished input file */
     continue;
   }

   local_ptr = ptr;
   /* remove leading white space locally, full line returned */
    while(local_ptr[0] == ' ' || local_ptr[0] == '\t' ) local_ptr++;

    /* skip comment/blank lines */
   if(local_ptr[0] == '!' ||  local_ptr[0] == '\0' ||
      local_ptr[0] == QicMark ||  local_ptr[0] == '\n'  ) continue;

    /* go out to included files */

    if (strstr(QiToUpper(local_ptr), "INCLUDE") != NULL){
      if(QiParseParamValue(local_ptr, &param, &value) != BAD_SYNTAX) {
	     if(strstr(QiToUpper(param), "INCLUDE") == NULL) continue; // skip lines with "include" in text
	     // skip text with "include"
         if( QiFindAlreadyOpen( QiStripQuotes(value), &fpNow) != 1){
           /* not already open, so open it */
           fpInc++;
           if (fpInc == MAX_N_INC_FILES) {
             QieAlertBox("Error","Too many recursions in included files");
             return NULL;
           }
           includedFiles[fpInc] = QiOpenFile((char*)"rb", incReadPath, QiStripQuotes(value), (char*)"");
           if( includedFiles[fpInc] != NULL) {
             fp = includedFiles[fpInc];
             includedFrom[fpInc] = fpNow;
		     fgetpos(includedFiles[fpNow], &(includedFromPosn[fpInc]) );
             fpNow = fpInc;
              fileNames[fpInc] = QiNewStr( QiStripQuotes(value));
           }
           else {
             fpInc--;
             QieAlertBox("Failed to include file ", QiStripQuotes(value));
           }
         }
         else{
           fp = includedFiles[fpNow]; /* file already open, use it */
         }
         continue;
	  }
    }

   /* skip next n lines from '&' marker */
   if(local_ptr[0] == '&'){
      check = QiParseParamValue(local_ptr, &param, &value);
      if (check == QMW_OK) n = atoi(value);
      for (i=0; i<n; i++){
        QiGetLine(fp); /* these are to be skipped */
      }
      n=0;
      continue;
   }

   if (!QistrNULL(skip_until) ){
     if (strstr(local_ptr, skip_until) != NULL) {
       /* found token */
       strcpy(skip_until, STR_NULL);
     }
     continue;   /* search until this token found */
   }

   if(newLine != NULL) {
     partLine = (char *) malloc(strlen(newLine)+1);
     strcpy(partLine, newLine);
     free(newLine);
     newLine = (char *)malloc(strlen(ptr) + strlen(partLine) +2);
     strcpy(newLine, partLine);
     if(newLine[strlen(newLine)-1] == '\n') newLine[strlen(newLine)-1] = '\0';
     strcat(newLine, ptr);
     free (partLine);
   }
   else {
       newLine = (char *)malloc(strlen(ptr) + 1);
       strcpy(newLine, ptr);
   }

   if(newLine[strlen(newLine)-1] == end_of_line_mark) {
     newLine[strlen(newLine)-1] = '\0';
     return &(newLine[0]);
   }

 }

 return NULL;

} /* end QiReadLine */

/***********************************************************************/

char * QiGetLine(FILE *fp){


 static char * line_ptr = NULL;
 static int line_len = 0;
 fpos_t posn;
 int c;
 int n_actual;
 int count;
 int notQuoted = 1;

   /* count length of line */

   fgetpos(fp, &posn);

   count = 0;
   c = fgetc(fp);
   while (c != end_of_line_mark && c != '\n' && c != EOF){

     if(c == '\"') {
       /* toggle quote counter */
       if(notQuoted == 0) notQuoted = 1;
       else notQuoted = 0;
     }

     if(c == '\\' && notQuoted == 1 && isCAA){
       count--;
       /* continuation marker, step to end of line and append next */
        while((c = fgetc(fp)) != '\n'){
          if(c == EOF) return NULL;
        }
     }
     count++;
     c = fgetc(fp);
   }
   fsetpos(fp, &posn);

   if (count+3 > line_len){
     /* new line is larger than previous, allocate enough memory */
     if(line_ptr != NULL) free (line_ptr);
     line_ptr = (char *) malloc(count+3);  /* add extra space for terminator and \n */
   }
   line_len = count;

   notQuoted = 1; /* reset quote test */

/* read line char at a time, inc new line */
   n_actual= 0;
   c = fgetc(fp);
   while ( c != end_of_line_mark && c != '\n' && c != EOF){

     if(c == '\"') {
       /* toggle quote counter */
       if(notQuoted == 0) notQuoted = 1;
       else notQuoted = 0;
     }

     if(c == '\\' && notQuoted == 1 && isCAA){
       /* continuation marker, skip to end of line and append next */
       while( (c = fgetc(fp)) != '\n'){
          if(c == EOF) return NULL;
       }
       // now increment to remove '\n'
       c = fgetc(fp);
     }

     /* skip extra characters treated as white space */
     if(c != '\f' &&
        c != '\v' &&
        c != '\a' &&
        c != '\r'  )
          line_ptr[n_actual++] = c;
     c = fgetc(fp);
   }

/* add newline or rec marker and end of string null */

   if( c != EOF ){
     line_ptr[n_actual] = c;
     line_ptr[n_actual+1] = '\0';
   }
   else if(n_actual == 0){
     return NULL;
   }
   else
     line_ptr[n_actual] = '\0';

   return line_ptr;

} /* end QiGetLine */

/*****************************************************************/

int Qifgetpos(fpos_t *file_posn,int *fpRewind){
  /* first of all find which file we are in - INCLUDES have been followed */
  if (fpNow >= 0) *fpRewind = fpNow;
  else return -1;

  /* return position in appropriate file */
  if(includedFiles[*fpRewind] != NULL) return fgetpos(includedFiles[*fpRewind], file_posn);
  else return -1;

}

/************************************************************/

int Qifsetpos( fpos_t *file_posn, int fpRewind) {
 int i;
 int parentFile;
 
  /* set pointer to file being rewound */
  for( i=fpInc; i>=0; i-- ){

    if(fpRewind == i) {
      fpNow = i;
      /* rewind parent to location of this file */
      if(fpNow >0){
        parentFile = includedFrom[fpNow];
        fsetpos(includedFiles[parentFile], &(includedFromPosn[fpNow]));
      }
      break;
    }
    else{
      /* rewind it */
      if(includedFiles[i] != NULL) rewind(includedFiles[i]);
      if(i == 0) return -1; /* have backed up through all files without finding target file */
    }
  }

   /* reset position in appropriate file */
  if(file_posn == NULL) {
    if(includedFiles[fpRewind] != NULL) {
      rewind(includedFiles[fpRewind]);
      return 0;
    }
    else return -1;
  }
  else {
    if(includedFiles[fpRewind] != NULL) return fsetpos(includedFiles[fpRewind], file_posn);
    else return -1;
  }

}

int QiFindAlreadyOpen(const char *name, int *which_fp){
int i;
  /* go from fpNow in case file is included more than once, only checks those we've rewound past */
  for( i=fpNow; i<=fpInc; i++){
    if(fileNames[i] == NULL) continue;
    if(strcmp(name, fileNames[i]) == 0){
      *which_fp = i;
      return 1;
    }
  }
  return 0;
}
