/***********************************************************************
*           See Accompanying licence file QSAS_licence                 *
***********************************************************************/
/* With adjustments for _WIN32 and __VMS compilers -- Patrick Daly & Rob Wilson */
/* Note some calls are provided by msys in a QSAS build. QSAS defines WINDOWS, Qtran does not. */

#include "qie.h"
#include "qie_local.h"

int QiLooksLikeVector(const char *name){

  if( strstr(name,"xyz") != NULL ) return 1;

  if( strstr(name,"XYZ") != NULL ) return 1;

  if( strstr(name,"gse") != NULL ) return 1;

  if( strstr(name,"GSE") != NULL ) return 1;

  if( strstr(name,"rlp") != NULL ) return 1;

  if( strstr(name,"RLP") != NULL ) return 1;

  if( strstr(name,"rtp") != NULL ) return 1;

  if( strstr(name,"RTP") != NULL ) return 1;

  return 0;
}

/***********************************************************************/

int QiAttrCmp(const char *str1, const char *str2InUpper){
  int result = 0;
  int i;
  char * str1InUpper = (char *)  malloc(strlen(str1)+1);

  for( i=0; i< strlen(str1); i++)
     str1InUpper[i] = toupper(str1[i]);

  str1InUpper[strlen(str1)] = '\0';

  if( strcmp(str1InUpper,str2InUpper) == 0) result = 1;

  free(str1InUpper);

  return result;
}

/***********************************************************************/

void QiCheckCEFstructure(QiCDFContents *QiSCDF){
   long n, m, i;
   long ptr_num;

   /* utility to analyse variable metadata present in QiCDF structure for CEF2 compliance */

  for ( n=0; n<QiSCDF->n_vars; n++){
    /* check each attribute against standard list and set flag */
     for( m=0; m < QiSCDF->vardata[n]->num_v_attrs; m++){
       char *ptr = (QiSCDF->vardata[n]->attribute)[m].name;
       if( QiAttrCmp(ptr, "UNITS") == 1 ) QiSCDF->vardata[n]->hasUnits = 1;
       if( QiAttrCmp(ptr, "FRAME") == 1 ) QiSCDF->vardata[n]->hasFrame = 1;
       if( QiAttrCmp(ptr, "SI_CONVERSION") == 1 ) QiSCDF->vardata[n]->hasSI_conversion = 1;
       if( QiAttrCmp(ptr, "FIELDNAM") == 1 ) QiSCDF->vardata[n]->hasFieldnam = 1;
       if( QiAttrCmp(ptr, "LABLAXIS") == 1 ) QiSCDF->vardata[n]->hasLablaxis = 1;
       if( QiAttrCmp(ptr, "DEPEND_0") == 1 ) QiSCDF->vardata[n]->hasDepend_0 = 1;
       if( QiAttrCmp(ptr, "DELTA_PLUS") == 1 ) QiSCDF->vardata[n]->hasDeltaPlus = 1;
       if( QiAttrCmp(ptr, "DELTA_MINUS") == 1 ) QiSCDF->vardata[n]->hasDeltaMinus = 1;
       if( QiAttrCmp(ptr, "TENSOR_FRAME") == 1 ) QiSCDF->vardata[n]->hasTensorFrame = 1;
       if( QiAttrCmp(ptr, "TENSOR_RANK") == 1 ) QiSCDF->vardata[n]->hasTensorRank = 1;


       if( QiAttrCmp(ptr, "REPRESENTATION_1") == 1 ) (QiSCDF->vardata[n]->numTensorRep_i) += 1;
       if( QiAttrCmp(ptr, "REPRESENTATION_2") == 1 ) (QiSCDF->vardata[n]->numTensorRep_i) += 1;
       if( QiAttrCmp(ptr, "REPRESENTATION_3") == 1 ) (QiSCDF->vardata[n]->numTensorRep_i) += 1;
       if( QiAttrCmp(ptr, "REPRESENTATION_4") == 1 ) (QiSCDF->vardata[n]->numTensorRep_i) += 1;
       if( QiAttrCmp(ptr, "REPRESENTATION_5") == 1 ) (QiSCDF->vardata[n]->numTensorRep_i) += 1;
       if( QiAttrCmp(ptr, "REPRESENTATION_6") == 1 ) (QiSCDF->vardata[n]->numTensorRep_i) += 1;
       if( QiAttrCmp(ptr, "REPRESENTATION_7") == 1 ) (QiSCDF->vardata[n]->numTensorRep_i) += 1;
       if( QiAttrCmp(ptr, "REPRESENTATION_8") == 1 ) (QiSCDF->vardata[n]->numTensorRep_i) += 1;
       if( QiAttrCmp(ptr, "REPRESENTATION_9") == 1 ) (QiSCDF->vardata[n]->numTensorRep_i) += 1;

       if( QiAttrCmp(ptr, "DEPEND_1") == 1 ) (QiSCDF->vardata[n]->numDepend_i) += 1;
       if( QiAttrCmp(ptr, "DEPEND_2") == 1 ) (QiSCDF->vardata[n]->numDepend_i) += 1;
       if( QiAttrCmp(ptr, "DEPEND_3") == 1 ) (QiSCDF->vardata[n]->numDepend_i) += 1;
       if( QiAttrCmp(ptr, "DEPEND_4") == 1 ) (QiSCDF->vardata[n]->numDepend_i) += 1;
       if( QiAttrCmp(ptr, "DEPEND_5") == 1 ) (QiSCDF->vardata[n]->numDepend_i) += 1;
       if( QiAttrCmp(ptr, "DEPEND_6") == 1 ) (QiSCDF->vardata[n]->numDepend_i) += 1;
       if( QiAttrCmp(ptr, "DEPEND_7") == 1 ) (QiSCDF->vardata[n]->numDepend_i) += 1;
       if( QiAttrCmp(ptr, "DEPEND_8") == 1 ) (QiSCDF->vardata[n]->numDepend_i) += 1;
       if( QiAttrCmp(ptr, "DEPEND_9") == 1 ) (QiSCDF->vardata[n]->numDepend_i) += 1;

       if( QiAttrCmp(ptr, "LABEL_1") == 1 ) (QiSCDF->vardata[n]->numLabel_i) += 1;
       if( QiAttrCmp(ptr, "LABEL_2") == 1 ) (QiSCDF->vardata[n]->numLabel_i) += 1;
       if( QiAttrCmp(ptr, "LABEL_3") == 1 ) (QiSCDF->vardata[n]->numLabel_i) += 1;
       if( QiAttrCmp(ptr, "LABEL_4") == 1 ) (QiSCDF->vardata[n]->numLabel_i) += 1;
       if( QiAttrCmp(ptr, "LABEL_5") == 1 ) (QiSCDF->vardata[n]->numLabel_i) += 1;
       if( QiAttrCmp(ptr, "LABEL_6") == 1 ) (QiSCDF->vardata[n]->numLabel_i) += 1;
       if( QiAttrCmp(ptr, "LABEL_7") == 1 ) (QiSCDF->vardata[n]->numLabel_i) += 1;
       if( QiAttrCmp(ptr, "LABEL_8") == 1 ) (QiSCDF->vardata[n]->numLabel_i) += 1;
       if( QiAttrCmp(ptr, "LABEL_9") == 1 ) (QiSCDF->vardata[n]->numLabel_i) += 1;
     }
  }

  /* go through list again doing extra stuff for the REPRESENTATION_i, DEPEND_i and LABEL_i variables */

  for ( n=0; n<QiSCDF->n_vars; n++){
     for( m=0; m < QiSCDF->vardata[n]->num_v_attrs; m++){
        /* find var pointed to by attribute content */
         if(QiSCDF->vardata[n]->attribute[m].data_type == CDF_CHAR  ||
          QiSCDF->vardata[n]->attribute[m].data_type == CDF_UCHAR ){

         if(QiSCDF->vardata[n]->attribute[m].data == NULL){
           printf("QIE Warning: attribute %s for variable %s is NULL", QiSCDF->vardata[n]->attribute[m].name, QiSCDF->vardata[n]->name);
           continue;
         }
         ptr_num = QiGetVarByName(QiSCDF, (char *) QiSCDF->vardata[n]->attribute[m].data);
         if(ptr_num == -1) continue; /* skip attributes that don't point to a variable */

         QiSCDF->vardata[ptr_num]->isPtrType = 1;

         /* DEPEND_0 is already dealt with */
         if( strstr(QiToUpper(QiSCDF->vardata[n]->attribute[m].name), "DEPEND_0") != NULL) continue;

         /* LABEL_i  */
         else if( strstr(QiToUpper(QiSCDF->vardata[n]->attribute[m].name), "LABEL_") != NULL) {

             QiSCDF->vardata[ptr_num]->isLabel_i = 1;

             /* ensure attr_var has same range as index of dimension i in parent var */
             i = atol(&(QiSCDF->vardata[n]->attribute[m].name)[7]) ;
             if(i > QiSCDF->vardata[n]->num_dims 
	        || QiSCDF->vardata[ptr_num]->num_dims != 1
		|| QiSCDF->vardata[n]->dim_sizes[i-1] != QiSCDF->vardata[ptr_num]->dim_sizes[0]){
                
		/* attr_var must be 1D array */
                /* dimension of var pointed to must match  dimension of index i in var */
		QiSCDF->vardata[ptr_num]->ErrLabel_i = 1;
               continue;
             }

         }

         /* DEPEND_i  */
         else if( strstr(QiToUpper(QiSCDF->vardata[n]->attribute[m].name), "DEPEND_") != NULL) {

             QiSCDF->vardata[ptr_num]->isDepend_i = 1;
             /* ensure attr_var has same range as index of dimension i in parent var */
             i = atol(&(QiSCDF->vardata[n]->attribute[m].name)[7]) ;
 
             if(i > QiSCDF->vardata[n]->num_dims 
	       || QiSCDF->vardata[ptr_num]->num_dims != 1
	       || QiSCDF->vardata[n]->dim_sizes[i-1] != QiSCDF->vardata[ptr_num]->dim_sizes[0]){

		/* attr_var must be 1D array */
                /* dimension of var pointed to must match  dimension of index i in var */
                QiSCDF->vardata[ptr_num]->ErrDepend_i = 1;
                continue;
             }
           
         } /* end DEPEND */

         /* REPRESENTATION_i  */
         else if( strstr(QiToUpper(QiSCDF->vardata[n]->attribute[m].name), "REPRESENTATION_") != NULL) {

             QiSCDF->vardata[ptr_num]->isTensorRep_i = 1;

             /* ensure attr_var has same range as index of dimension i in parent var */
             i = atol(&(QiSCDF->vardata[n]->attribute[m].name)[7]) ;
             if(i > QiSCDF->vardata[n]->num_dims 
	        || QiSCDF->vardata[ptr_num]->num_dims != 1
		|| QiSCDF->vardata[n]->dim_sizes[i-1] != QiSCDF->vardata[ptr_num]->dim_sizes[0]){
                
		/* attr_var must be 1D array */
	       /* dimension of var pointed to must match  dimension of index i in var */
                QiSCDF->vardata[ptr_num]->ErrTensorRep_i = 1;
              continue;
             }

         }

       } /* end if on char attr type */
     } /* end for on Attrs */
  } /* end for on vars */

} /* end QiCheckCEFstructure */

/***********************************************************************/

long QiGetVarByName(QiCDFContents *QiSCDF, const char * name){
  long n;

  for( n=0; n<QiSCDF->n_vars; n++){
     if( strcmp(QiSCDF->vardata[n]->name, name) == 0)
         return n;
  }
  return -1;
}

/***********************************************************************/

void QiCEFvalidate(QiCDFContents *QiSCDF){
   int n;
   char rankTxt[30];

   /* utility to test QiCDF structure is CEF2 compliant */
  for ( n=0; n<QiSCDF->n_vars; n++){

    /* handle DEPEND_i seperately, other ptr attributes do not need attributes themselves */
    if(QiSCDF->vardata[n]->isPtrType == 0) {

       /* specific attributes required by ALL variables */

       if(QiSCDF->vardata[n]->hasFieldnam == 0 )
           QiCEFmsg( 0, "variable ", QiSCDF->vardata[n]->name, "Requires FIELDNAM attribute");

       if(QiSCDF->vardata[n]->hasLablaxis == 0 )
           QiCEFmsg( 0, "variable ", QiSCDF->vardata[n]->name, "Requires LABLAXIS attribute");


       /* specific attributes NOT required by time tags */
       if( n != QiSCDF->d0_var_num){

          /* needed if there is a valid time variable */
          if(QiSCDF->vardata[n]->hasDepend_0 == 0 && -1 != QiSCDF->d0_var_num && QiSCDF->vardata[n]->isDepend_i == 0)
              QiCEFmsg( 0, "variable ", QiSCDF->vardata[n]->name, "Requires DEPEND_0 attribute");

          if(QiSCDF->vardata[n]->hasUnits == 0 )
             QiCEFmsg( 0, "variable ", QiSCDF->vardata[n]->name, "Requires UNITS attribute");

          if(QiSCDF->vardata[n]->hasSI_conversion == 0 )
             QiCEFmsg( 0, "variable ", QiSCDF->vardata[n]->name, "Requires SI_CONVERSION attribute");

            /* specific attributes required only by vectors and tensors */
          if( QiLooksLikeVector(QiSCDF->vardata[n]->name) ){

             if(QiSCDF->vardata[n]->hasFrame == 0   )
                 QiCEFmsg( 0, "If variable ", QiSCDF->vardata[n]->name, "is a vector or Tensor it requires a FRAME attribute");

             if(QiSCDF->vardata[n]->hasTensorFrame == 0   )
                 QiCEFmsg( 0, "If variable ", QiSCDF->vardata[n]->name,
                          "is a vector or Tensor it requires a TENSOR_FRAME attribute");

             if(QiSCDF->vardata[n]->num_dims != QiSCDF->vardata[n]->numTensorRep_i  )
                 QiCEFmsg( 0, "If variable ", QiSCDF->vardata[n]->name,
                      "is a vector or Tensor it requires a tensor REPRESENTATION_i attribute for each index");

             if(QiSCDF->vardata[n]->hasTensorRank == 0   )
                 QiCEFmsg( 0, "If variable ", QiSCDF->vardata[n]->name,
                       "is a vector or Tensor it requires a TENSOR RANK attribute");
          }

        } /* end if on not time var */

        /* specific attributes required for array type data */
        /* Note DEPEND_i and LABEL_i do NOT take these attributes (test with isPtrType) */

        if(QiSCDF->vardata[n]->num_dims > 0 ){

          if( QiSCDF->vardata[n]->numLabel_i + QiSCDF->vardata[n]->numDepend_i != QiSCDF->vardata[n]->num_dims){
             QiCEFmsg( 0, "Variable ", QiSCDF->vardata[n]->name,
                      "needs EITHER DEPEND_i or LABEL_i for every index");
             sprintf(rankTxt, "%ld indices\n but", QiSCDF->vardata[n]->num_dims);
             QiCEFmsg( -1, QiSCDF->vardata[n]->name, "has", rankTxt );

           sprintf(rankTxt, "%d LABEL_i attributes\n and", QiSCDF->vardata[n]->numLabel_i);
             QiCEFmsg( -1, QiSCDF->vardata[n]->name, "has only", rankTxt );

           sprintf(rankTxt, "%d DEPEND_i attributes", QiSCDF->vardata[n]->numDepend_i);
             QiCEFmsg( -1, QiSCDF->vardata[n]->name, "has only", rankTxt );
          }
        } /* end if on array data */

     } /* end if on handling non-pointer type attributes */

    /* check attributes for special case of DEPEND_i variables */
    if(QiSCDF->vardata[n]->isDepend_i == 1){

      if(QiSCDF->vardata[n]->hasDepend_0 == 0 && -1 != QiSCDF->d0_var_num && QiSCDF->vardata[n]->rec_vary == 1)
              QiCEFmsg( 0, "variable ", QiSCDF->vardata[n]->name,
                         "is a record varying DEPEND_i and requires DEPEND_0 attribute");

      if(QiSCDF->vardata[n]->hasDeltaPlus == 0  )
               QiCEFmsg( 0, "Variable ",
                       QiSCDF->vardata[n]->name, "is a DEPEND_i variable and requires a DELTA_PLUS attribute");

      if(QiSCDF->vardata[n]->hasDeltaMinus == 0  )
               QiCEFmsg( 0, "Variable ", QiSCDF->vardata[n]->name,
                        "is a DEPEND_i variable and requires a DELTA_MINUS attribute");

      /* DEPEND_i variables are the only pointer type variables that take all the regular attributes */

       if(QiSCDF->vardata[n]->hasFieldnam == 0 )
           QiCEFmsg( 0, "variable ", QiSCDF->vardata[n]->name, "Requires FIELDNAM attribute");

       if(QiSCDF->vardata[n]->hasLablaxis == 0 )
           QiCEFmsg( 0, "variable ", QiSCDF->vardata[n]->name, "Requires LABLAXIS attribute");

       if(QiSCDF->vardata[n]->hasUnits == 0 )
             QiCEFmsg( 0, "variable ", QiSCDF->vardata[n]->name, "Requires UNITS attribute");

       if(QiSCDF->vardata[n]->hasSI_conversion == 0 )
             QiCEFmsg( 0, "variable ", QiSCDF->vardata[n]->name, "Requires SI_CONVERSION attribute");

       if(QiSCDF->vardata[n]->ErrDepend_i == 1 )
             QiCEFmsg( 0, "dimensions of variable ", QiSCDF->vardata[n]->name,
               "are not consistent with dimension of index i of a variable for which it is DEPEND_i");

       if(QiSCDF->vardata[n]->ErrLabel_i == 1 )
             QiCEFmsg( 0, "dimensions of variable ", QiSCDF->vardata[n]->name,
               "are not consistent with dimension of index i of a variable for which it is LABEL_i");

       if(QiSCDF->vardata[n]->ErrTensorRep_i == 1 )
             QiCEFmsg( 0, "dimensions of variable ", QiSCDF->vardata[n]->name,
               "are not consistent with dimension of index i of a variable for which it is REPRESENTATION_i");



    } /* end if on type is DEPEND_i var */

  } /* end for over n_vars */

} /* end QiCEFvalidate */

/***********************************************************************/

long QiInitGlobalContents(QiCDFContents * QiSCDF)
{
 char * today;
 time_t tnow;
 char iso_time[25];
 char *nowCopy;

 /* This MUST be first call before exporting */
 /* start list for FileInfo objects */

 QiSCDF->g_attr = (QiGlobalAttribute **) QiMakeQiGAttrPtrs (MAX_N_ATTRS);

 QiSCDF->num_g_attrs = 0;

    /* use replace not create in case default set in skeleton */

     QiReplaceGlobalTxtAttr(QiSCDF, "TITLE",
                  "Data exported from QSAS analysis software");

     QiReplaceGlobalTxtAttr(QiSCDF, "MODS",
                  "Data and/or timeline was processed by QSAS");


    time(&tnow);
    nowCopy = QiNewStr(ctime(&tnow));
    today = QiCtoISO(nowCopy, iso_time);
    QiReplaceGlobalTxtAttr(QiSCDF, "Generation_date", today);

 /* append QSAS specific information */

 /*   QiAppendGlobalTxtAttr(QiSCDF, "Software_version", QieVersion()); */


 return QMW_OK;

}  /* end QiInitGlobalContents */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiReplaceGlobalTxtAttr
*
* PURPOSE: Replace text global attribute in contents object with new entries.
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED:
*
***********************************************************************/

long QiReplaceGlobalTxtAttr(QiCDFContents * QiSCDF,
                         const char * attr_name,
                         const char * new_value)
{
   long n_attr;
   char * char_ptr;

   if(QiFindGlobalAttr(QiSCDF, attr_name, &n_attr) == NULL){
     QiCreateGlobalTxtAttr(QiSCDF, attr_name, new_value);
   }
   else{
     /* delete old entry by freeing space */

     if( QiSCDF->g_attr[n_attr]->entry->data != NULL){
       free(QiSCDF->g_attr[n_attr]->entry->data);
       QiSCDF->g_attr[n_attr]->entry->data = NULL;
     }

     if(QiSCDF->g_attr[n_attr]->entry != NULL) {
        free(QiSCDF->g_attr[n_attr]->entry);
        QiSCDF->g_attr[n_attr]->entry = NULL;
     }

     /* set up new entry */

     QiSCDF->g_attr[n_attr]->entry = ( QiGAttrEntry *) QiMakeEntries(1);
     QiSCDF->g_attr[n_attr]->num_entries = 1;
     QiSCDF->g_attr[n_attr]->entry->exists = EXISTS;
     QiSCDF->g_attr[n_attr]->entry->data_type = CDF_CHAR;
     QiSCDF->g_attr[n_attr]->entry->num_elems = strlen(new_value);

     QiSCDF->g_attr[n_attr]->entry->data =
                                      (void *) malloc(strlen(new_value) +1);
     char_ptr = (char *) QiSCDF->g_attr[n_attr]->entry->data;
     strcpy(char_ptr,new_value);

   }

   return QMW_OK;


} /* end QiReplaceGlobalTxtAttr */



/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiFindGlobalAttr
*
* PURPOSE: Locate attribute number of named attribute in QiCDFContents structure.
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

long * QiFindGlobalAttr(QiCDFContents * QiSCDF,
                      const char * attr_name,
                      long * attr_num)
{

  if(attr_name == NULL) return NULL;

  for(*attr_num = 0; *attr_num < QiSCDF->num_g_attrs ; (*attr_num)++){
    if(strcmp(attr_name, QiSCDF->g_attr[*attr_num]->name) == 0)
       return attr_num;
  }

  /* if not found return NULL */

  return NULL;



} /* end QiFindGlobalAttr */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiCreateGlobalTxtAttr
*
* PURPOSE: Create a new global attribute with specified name and entry.
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

long QiCreateGlobalTxtAttr(QiCDFContents * QiSCDF,
                           const char * attr_name,
                           const char * new_value)
{
  long attr_num;
  char * char_ptr;

  attr_num = QiSCDF->num_g_attrs;   /* this is next available slot */

  /* create new attribute if space left in stack */

  if (QiSCDF->num_g_attrs < MAX_N_ATTRS) {
    QiSCDF->g_attr[attr_num] = (QiGlobalAttribute *) QiMakeQiGAttr();
  }
  else{
    return EXCEED_MAX_NUM_G_ATTRS;
  }

  /* populate new attribute */

    QiSCDF->g_attr[attr_num]->number = attr_num;
    QiSCDF->g_attr[attr_num]->name = (char *) malloc(strlen(attr_name)+1);
    strcpy(QiSCDF->g_attr[attr_num]->name,attr_name );
    QiSCDF->g_attr[attr_num]->num_entries = 1;

    QiSCDF->g_attr[attr_num]->entry = ( QiGAttrEntry *) QiMakeEntries(1);
    QiSCDF->g_attr[attr_num]->entry->exists = EXISTS;
    QiSCDF->g_attr[attr_num]->entry->data_type = CDF_CHAR;
    QiSCDF->g_attr[attr_num]->entry->num_elems = strlen(new_value);
    QiSCDF->g_attr[attr_num]->entry->data = (void *)malloc(strlen(new_value) +1);
    char_ptr = (char *)QiSCDF->g_attr[attr_num]->entry->data;
    strcpy(char_ptr, new_value);

    QiSCDF->num_g_attrs++;

    return QMW_OK;


} /* end QiCreateGlobalTxtAttr */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: CDFsizeof
*
* PURPOSE: Return size of CDF data type.
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   size_t
*    VALUES: Size of data type
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/
size_t CDFsizeof( long data_type, long n_elems)
{

  switch (data_type){
    case CDF_EPOCH: case CDF_REAL8: case CDF_DOUBLE:
      return sizeof(double) * n_elems;
      break;

    case CDF_REAL4: case CDF_FLOAT:
      return sizeof(float) * n_elems;
      break;

    case CDF_INT4: case CDF_UINT4:
      return sizeof(long) * n_elems;
      break;

    case CDF_INT2: case CDF_UINT2:
      return sizeof(short) * n_elems;
      break;

    case CDF_INT1: case CDF_UINT1: case CDF_BYTE:
      return sizeof(char) * n_elems;
      break;

    case CDF_CHAR: case CDF_UCHAR:
      return sizeof(char) * (n_elems + 1);
      break;

   case ISO_TIME:
      return sizeof(char) * (n_elems + 1);
      break;

   case ISO_TIME_RANGE:
      return sizeof(char) * (n_elems + 1);
      break;

  }

  /* else return something small, this is used as an in-line function, no error test */

  return sizeof(char);

} /* end CDFsizeof */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiCopyGlobalEntry
*
* PURPOSE: Copy one QiGAttrEntry structure entry to another.
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/
long QiCopyGlobalEntry(  QiGAttrEntry * newGentry,
                         QiGAttrEntry * oldGentry,
                         long copy)
{
 size_t size;

      newGentry->exists = oldGentry->exists;
      newGentry->data_type = oldGentry->data_type;
      newGentry->num_elems = oldGentry->num_elems;

      switch(copy){

        case BY_PTR:
        {
          newGentry->data = oldGentry->data;
          break;
        }
        case BY_MEM:
        {
          /* determine entry size */

          size = CDFsizeof(newGentry->data_type, newGentry->num_elems);
          newGentry->data = (void *)malloc(size);
          memcpy( newGentry->data, oldGentry->data, size);
          break;
        }
        default:
        {
           return OPERATION_NOT_ALLOWED;
        }
      }

      return QMW_OK;

} /* end QiCopyGlobalEntry */



/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QIFF
*
* MODULE:  iff_to_cdf
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiCtoISO
*
* PURPOSE: Converts time string  to Iso standard format.
*
*
* DESCRIPTION: Converts time string from standard C library to Iso standard
*              text format.
*
*
* ARGUMENTS:
*    IN:       char *time                         input time string
*    OUT:      char *iso_time                     returned ISO time string
*
* GLOBALS:
*
* RETURN:
*    TYPE:   char *
*    VALUES: Iso_time_string
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*            Cut off time/date elements from back
*            convert text month to digital month
*            Assemble Iso time format from elements
*
***********************************************************************/

char * QiCtoISO(char *time, char *iso_time){

char year[5];
char month[3];
char mon[4];
char day[3];
char hour[9];
long n_char;

  /* remove newline from time and put year in year */
  time[24] = '\0';
  QiStrcpy(year, &time[20], 4);

  time[7] = '\0';
  QiStrcpy(mon, &time[4], 3);
  if(strcmp(mon,"Jan")==0){
    strcpy(month,"01");
  }
  else if(strcmp(mon,"Feb")==0){
    strcpy(month,"02");
  }
  else if(strcmp(mon,"Mar")==0){
    strcpy(month,"03");
  }
  else if(strcmp(mon,"Apr")==0){
    strcpy(month,"04");
  }
  else if(strcmp(mon,"May")==0){
    strcpy(month,"05");
  }
  else if(strcmp(mon,"Jun")==0){
    strcpy(month,"06");
  }
  else if(strcmp(mon,"Jul")==0){
    strcpy(month,"07");
  }
  else if(strcmp(mon,"Aug")==0){
    strcpy(month,"08");
  }
  else if(strcmp(mon,"Sep")==0){
    strcpy(month,"09");
  }
  else if(strcmp(mon,"Oct")==0){
    strcpy(month,"10");
  }
  else if(strcmp(mon,"Nov")==0){
    strcpy(month,"11");
  }
  else if(strcmp(mon,"Dec")==0){
    strcpy(month,"12");
  }
  else{
    mon[3] = '\0';
    strcpy(month, mon); /* as diagnostic */
  }
  time[10]='\0';
  QiStrcpy(day, &time[8], 2);

  time[19]='\0';
  QiStrcpy(hour, &time[11], 8);


  /* assemble output string */

  strcpy(iso_time,year);strcat(iso_time,"-");
  strcat(iso_time,month);strcat(iso_time,"-");
  strcat(iso_time,day);strcat(iso_time,"T");
  strcat(iso_time,hour);strcat(iso_time,".000Z");

  /* replace white space with 0 for ISO format */
  for (n_char=0; n_char<25; n_char++){
    if(iso_time[n_char]==' ') iso_time[n_char]='0';
  }

  return iso_time;

} /* end QiCtoISO  */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiAppendGlobalAttr
*
* PURPOSE: Append global attribute from FileInfo Object to Contents object.
*
* DESCRIPTION: Makes a blind copy of G attribute data object from one
*              structure to another.
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

long QiAppendGlobalAttr(QiGlobalAttribute * QiSContsAttr,
                       long attr_num,
                       QiGlobalAttribute * QiSInfoAttr )
{
   long n;
   long check = QMW_OK;
   long n_new = 0;
   long n_copied=0;
   QiGAttrEntry * QiSList[MAX_N_G_ENTRIES];
   QiGAttrEntry * QiSNewEntry;

    /* assemble list of new entries */

    for (n=0; n < QiSInfoAttr->num_entries; n++){
      if( QiIsNewEntry(QiSContsAttr, QiSInfoAttr->entry+n) == QMW_OK ){
        QiSList[n_new] = QiSInfoAttr->entry+n;
        n_new++;
        if(n_new == MAX_N_G_ENTRIES) {
           check = EXCEED_MAX_ENTRIES;
           break;
        }
      }
      /* else just ignore it */
    }

    /* assemble new Global attribute object */

    QiSNewEntry = ( QiGAttrEntry *)
                   QiMakeEntries (QiSContsAttr->num_entries + n_new);

    /* copy over old entries */

    for (n=0; n < QiSContsAttr->num_entries; n++){
      if( memcmp(OVERWRITE_SKT, (QiSContsAttr->entry + n)->data,
              strlen(OVERWRITE_SKT))  == 0 ) continue;
      QiCopyGlobalEntry(QiSNewEntry + n, QiSContsAttr->entry + n, BY_PTR);
      n_copied++;
    }

    /* add new entries */

    for (n=0; n < n_new; n++){
      QiCopyGlobalEntry(QiSNewEntry + n + QiSContsAttr->num_entries,
                        QiSList[n], BY_MEM);
    }

    /* update contents object with new entry object */

    QiSContsAttr->num_entries = n_copied + n_new;
    free(QiSContsAttr->entry);
    QiSContsAttr->entry = QiSNewEntry;

    return QMW_OK;

} /* end QiAppendGlobalAttr */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiCreateGlobalAttr
*
* PURPOSE: Append global attribute from FileInfo Object to Contents object.
*
* DESCRIPTION: Makes a blind copy of G attribute data object from one
*              structure to another.
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

long QiCreateGlobalAttr(QiCDFContents * QiSCDF,
                        QiGlobalAttribute * QiSInfoAttr ,
                        long *n_attr)
{
  long n;
  *n_attr = QiSCDF->num_g_attrs;

  if (QiSCDF->num_g_attrs < MAX_N_ATTRS) {
    QiSCDF->g_attr[*n_attr] = (QiGlobalAttribute *) QiMakeQiGAttr();
  }
  else{
    return EXCEED_MAX_NUM_G_ATTRS;
  }

  /* copy attribute from FileInfo object */

  QiSCDF->g_attr[*n_attr]->number = *n_attr;
  QiSCDF->g_attr[*n_attr]->num_entries = QiSInfoAttr->num_entries;

  QiSCDF->g_attr[*n_attr]->name = (char *) malloc(strlen(QiSInfoAttr->name)+1);
  strcpy (QiSCDF->g_attr[*n_attr]->name, QiSInfoAttr->name);

  QiSCDF->g_attr[*n_attr]->entry = ( QiGAttrEntry *)
                             QiMakeEntries (QiSCDF->g_attr[*n_attr]->num_entries);

  for (n=0; n < QiSCDF->g_attr[*n_attr]->num_entries; n++){
    QiCopyGlobalEntry(QiSCDF->g_attr[*n_attr]->entry + n,
                      QiSInfoAttr->entry + n, BY_MEM);
  }

  QiSCDF->num_g_attrs++;

  return QMW_OK;

} /* end   QiCreateGlobalAttr  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiIsNewEntry
*
* PURPOSE: Determine if entry already exists in G attribute.
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

long QiIsNewEntry(QiGlobalAttribute * QiSContsAttr,
                  QiGAttrEntry * QiSEntry)

{
  long n;
  int comp;
  size_t size1, size2;

  if(QiSEntry->data_type == CDF_CHAR || QiSEntry->data_type == CDF_UCHAR){
    size1 = CDFsizeof(QiSEntry->data_type, QiSEntry->num_elems-1);
  }
  else{
    size1 = CDFsizeof(QiSEntry->data_type, QiSEntry->num_elems);
  }

  for (n=0; n < QiSContsAttr->num_entries; n++){

    if(QiSEntry->data_type == CDF_CHAR || QiSEntry->data_type == CDF_UCHAR){
      size2 = CDFsizeof((QiSContsAttr->entry+n)->data_type,
                        (QiSContsAttr->entry+n)->num_elems-1);
    }
    else{
      size2 = CDFsizeof((QiSContsAttr->entry+n)->data_type,
                        (QiSContsAttr->entry+n)->num_elems);
    }
    if( size1 == size2){
      comp = memcmp( (QiSContsAttr->entry+n)->data,
                      QiSEntry->data,
                      size1);
      if (comp == 0) return EXISTS;
    }
  }

  return QMW_OK;

} /* end QiIsNewEntry */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiAppendGlobalTxtAttr
*
* PURPOSE: Append global text attribute.
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Executed successfully.
*            Otherwise   Fatal error detected.
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

 long QiAppendGlobalTxtAttr(QiCDFContents * QiSCDF,
                            char * attr_name,
                            const char * txt_entry)

 {
   long n_attr;
   char * char_ptr;
   QiGlobalAttribute new_Gattr;

   if( txt_entry == NULL) return QMW_OK;

   if(QiFindGlobalAttr(QiSCDF, attr_name, &n_attr) == NULL){

     /* new attribute */
     QiCreateGlobalTxtAttr(QiSCDF, attr_name, txt_entry);
   }
   else{

     /* set up new entry for existing attribute */

     new_Gattr.entry = ( QiGAttrEntry *) QiMakeEntries(1);
     new_Gattr.num_entries = 1;
     new_Gattr.entry->exists = EXISTS;
     new_Gattr.entry->data_type = CDF_CHAR;
     new_Gattr.entry->num_elems = strlen(txt_entry);

     new_Gattr.entry->data = (void *) malloc(strlen(txt_entry) +1);
     char_ptr = (char *) new_Gattr.entry->data;
     strcpy(char_ptr, txt_entry);

     QiAppendGlobalAttr(QiSCDF->g_attr[n_attr], n_attr, &new_Gattr);

     free(new_Gattr.entry->data);
     free(new_Gattr.entry);
   }

   return QMW_OK;


 } /* end   QiAppendGlobalTxtAttr */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiFindSampleSpacing
*
* PURPOSE: Find sample spacing appropriate for Half_interval.
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   double
*    VALUES: Sample spacing in msec
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

double QiFindSampleSpacing(double * data,
                           long num_recs,
                           long iterate,
                           double discard)
{
 double min_diff;
 double diff;
 double half_interval;
 long n;
 long num_at_min;

 /* at least three data entries required for interval test*/
 if( num_recs < 3 ) return -1.0;

 /* establish the minimum interval, data is monotonic increasing */
 /*  discarding values below "discard" to tolerate a few glitches */

 min_diff = data[1] - data[0];

 for(n=1; n<num_recs-1; n++){
   diff = data[n+1] - data[n];
   if( diff < min_diff && diff > discard) min_diff = diff;
 }

 /* If less than 1% of data are within a tolerance of this separation */
 /* data has a glitch or is too noisy */

 num_at_min=0;
 for(n=0; n<num_recs-1; n++){
   if( data[n+1] - data[n] < min_diff * INTERVAL_TOLERANCE) {
     num_at_min++;
   }
 }

 if( num_at_min * 100 > num_recs) {
   half_interval = min_diff * 0.5;
 }
 else{
   iterate++;
   if(iterate > MAX_NUM_ITERATIONS){
     half_interval = -1.0;
   }
   else{
     /* iterate recursively dropping abnormally close gap */
     half_interval = QiFindSampleSpacing(data, num_recs, iterate, min_diff);
   }
 }
  return half_interval;

}  /* end QiFindSampleSpacing */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiCDFDataType
*
* PURPOSE: Determine flat file data type appropriate for CDF data type
*          to add to flat file export.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   char *
*    VALUES: string with flat file data type
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

char * QiCDFDataType( long CDF_datatype)
{
 char * datatype;

  switch (CDF_datatype){
    case CDF_REAL8: case CDF_DOUBLE:
    {
      datatype = (char *)malloc( sizeof(char) * 7);
      strcpy(datatype, "DOUBLE");
      break;
    }

    case CDF_EPOCH:
    {
      datatype = (char *)malloc( sizeof(char) * 6);
      strcpy(datatype, "EPOCH");
      break;
    }

    case CDF_REAL4: case CDF_FLOAT:
    {
      datatype = (char *)malloc( sizeof(char) * 6);
      strcpy(datatype, "FLOAT");
      break;
    }

    case CDF_INT4: case CDF_UINT4:
    {
      datatype = (char *)malloc( sizeof(char) * 4);
      strcpy(datatype, "INT");
      break;
    }

    case CDF_INT2: case CDF_UINT2:
    {
      datatype = (char *)malloc( sizeof(char) * 4);
      strcpy(datatype, "INT");
      break;
    }

    case CDF_INT1: case CDF_UINT1: case CDF_BYTE:
    {
      datatype = (char *)malloc( sizeof(char) * 5);
      strcpy(datatype, "BYTE");
      break;
    }
    case CDF_CHAR: case CDF_UCHAR:
    {
      datatype = (char *)malloc( sizeof(char) * 5);
      strcpy(datatype, "CHAR");
      break;
    }
    case ISO_TIME:
    {
      datatype = (char *)malloc( sizeof(char) * 9);
      strcpy(datatype, "ISO_TIME");
      break;
    }
    case ISO_TIME_RANGE:
    {
      datatype = (char *)malloc( sizeof(char) * 15);
      strcpy(datatype, "ISO_TIME_RANGE");
      break;
    }
     default:
    {
      datatype = (char *)malloc( sizeof(char) * 8);
      strcpy(datatype, "UNKNOWN");
    }

  } /* end switch */

     return datatype;

} /* End QiCDFDataType   */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiCountRecords
*
* PURPOSE: Count number of data records in flat file without header.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: number of records
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiCountRecords(QiOptions *QiSOpt)

{
  char * line;
  long count = 0;
  fpos_t entry;
  char * hold_after;
  int fpRewind;
  int i;

  hold_after = QiNewStr(QiSOpt->start_after);

  if( Qifgetpos(&entry, &fpRewind) != 0) return 0; /* can't find where I am in file */
  if(QiSOpt->rec_end == '\n'){
    while((line=QiReadLine(QiSOpt->start_after )) != NULL){
      if( strlen(QiSOpt->data_until) != 0){
          if(strncmp(line, QiSOpt->data_until, strlen(QiSOpt->data_until)) == 0) break;
	 }
      count++;
    }
  }
  else{
    while((line=QiReadLine(QiSOpt->start_after )) != NULL){
      /* line returned from QiReadLine may contain one, more or no end of record markers - count them */
      for( i=0; i<strlen(line); i++)
        if(line[i] == QiSOpt->rec_end ) count++;
    }
  }
  Qifsetpos (&entry, fpRewind);
  strcpy(QiSOpt->start_after, hold_after);
  free (hold_after);
  return count;

} /* end QiCountRecords  */

/**********************************************************************
****************** START OF MEMORY ALLOCATION FUNCTIONS ******************
**********************************************************************/

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiMakeCDFContentsObj
*
* PURPOSE: Allocate memory and initialise entries in structure .
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   QiCDFContents *
*    VALUES: Pointer to new object
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

QiCDFContents * QiMakeCDFContentsObj()

{
  QiCDFContents * QiSCDF;

  QiSCDF = (QiCDFContents *) malloc ( sizeof(QiCDFContents) );
  QiSCDF->io_f_name = STR_NULL;
  QiSCDF->io_f_path = STR_NULL;
  QiSCDF->io_f_extn = STR_NULL;
  QiSCDF->vardata = NULL;
  QiSCDF->g_attr = NULL;
  QiSCDF->num_g_attrs = 0;
  QiSCDF->n_recs = 0;
  QiSCDF->n_vars = 0;
  QiSCDF->d0_var_num = -1;
  QiSCDF->numSecDecimal = 3;  
  QiSCDF->UserSetSecDecimal = 0;  

  return QiSCDF;

}  /* end  QiMakeCDFContentsObj  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiMakeQiVariablePtrs
*
* PURPOSE: Allocate memory and initialise entries in structure
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   QiCDFVariable **
*    VALUES: Pointer to new object
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

QiCDFVariable ** QiMakeQiVariablePtrs(long n_vars)

{
  QiCDFVariable ** QiVardata;
  long n;

  QiVardata = (QiCDFVariable **) malloc ( sizeof(QiCDFVariable *) * n_vars );

  for(n=0; n< n_vars; n++)
  {
    QiVardata[n] = NULL;
  }

  return QiVardata;

}  /* end  QiMakeQiVariablePtrs  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiMakeQiGAttrPtrs
*
* PURPOSE: Allocate memory and initialise entries in structure
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   QiGlobalAttribute **
*    VALUES: Pointer to new object
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

QiGlobalAttribute ** QiMakeQiGAttrPtrs(long n_Gattrs)

{
  QiGlobalAttribute ** QiG_attr;
  long n;

  QiG_attr = (QiGlobalAttribute **) malloc ( sizeof(QiGlobalAttribute *) * n_Gattrs );

  for(n=0; n< n_Gattrs; n++)
  {
    QiG_attr[n] = NULL;
  }

  return QiG_attr;

}  /* end  QiMakeQiGAttrPtrs  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiMakeQiVariable
*
* PURPOSE: Allocate memory and initialise entries in structure
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   QiCDFVariable *
*    VALUES: pointer to new object
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

QiCDFVariable * QiMakeQiVariable()

{
  QiCDFVariable * QiVar;

  QiVar = (QiCDFVariable *) malloc ( sizeof(QiCDFVariable) );

  QiVar->novary_opt = WRITE_ONCE;
  QiVar->name = STR_NULL;
  QiVar->dim_sizes = (long *) malloc(sizeof(long)*CDF_MAX_DIMS);
  QiVar->dim_varys = (long *) malloc(sizeof(long)*CDF_MAX_DIMS);

  QiVar->data = NULL;
  QiVar->attribute = (QiVarAttribute *) QiMakeQiVAttr(MAX_N_ATTRS);
  QiVar->num_v_attrs = 0;
  QiVar->num_elems = 1;
  QiVar->max_rec_num = -1;
  QiVar->sizeofentry = (long) sizeof(double);

  QiVar->isPtrType = 0;
  QiVar->isDepend_i = 0;
  QiVar->isLabel_i = 0;
  QiVar->hasUnits = 0;
  QiVar->hasFrame = 0;
  QiVar->hasSI_conversion = 0;
  QiVar->hasFieldnam = 0;
  QiVar->hasLablaxis = 0;
  QiVar->hasDepend_0 = 0;
  QiVar->hasDeltaPlus = 0;
  QiVar->hasDeltaMinus = 0;
  QiVar->hasTensorFrame = 0;
  QiVar->hasTensorRank = 0;
  QiVar->numTensorRep_i = 0;
  QiVar->numLabel_i = 0;
  QiVar->numDepend_i = 0;
  QiVar->ErrDepend_i = 0;
  QiVar->ErrLabel_i = 0;
  QiVar->ErrTensorRep_i = 0;


  return QiVar;

}  /* end  QiMakeQiVariable  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiMakeQiVAttr
*
* PURPOSE: Allocate memory and initialise entries in structure
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   QiVarAttribute *
*    VALUES: Pointer to new object
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

 QiVarAttribute * QiMakeQiVAttr(long n_Vattrs)

{
  QiVarAttribute * QiV_attr;
  long n;
  long n_attr;

  n_attr = n_Vattrs + SPARE;  /* extra in case DEPEND_i included later */

  QiV_attr = (QiVarAttribute *)
                      malloc ( sizeof(QiVarAttribute) * n_attr );

  for(n=0; n< n_attr; n++)
  {
    (QiV_attr + n)->name = STR_NULL;
    (QiV_attr + n)->data = NULL;
    (QiV_attr + n)->num_elems = 1;

  }

  return QiV_attr;

}  /* end  QiMakeQiVAttr  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiMakeQiGAttr
*
* PURPOSE: Allocate memory and initialise entries in structure
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   QiGlobalAttribute *
*    VALUES: Pointer to new object
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

QiGlobalAttribute * QiMakeQiGAttr( )

{
  QiGlobalAttribute * QiG_attr;

  QiG_attr = (QiGlobalAttribute *) malloc ( sizeof(QiGlobalAttribute) );

  QiG_attr->name = STR_NULL;
  QiG_attr->entry = NULL;
  QiG_attr->num_entries = 0;

  return QiG_attr;

}  /* end  QiMakeQiGAttr  */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiMakeEntries
*
* PURPOSE: Allocate memory and initialise entries in structure
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   struct QiGAttrEntry *
*    VALUES: Pointer to new object
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

 QiGAttrEntry * QiMakeEntries(long n_entries)

{
   QiGAttrEntry * Qientry;
  long n;

  Qientry = ( QiGAttrEntry *)
                  malloc(sizeof( QiGAttrEntry) * n_entries);
  for (n=0; n < n_entries; n++)
  {
    (Qientry + n)->data = NULL;
  }

  return Qientry;

}  /* end  QiMakeEntries  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiMakeOptionsObj
*
* PURPOSE: Allocate memory and initialise entries in structure
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   QiOptions *
*    VALUES: Pointer to new object
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

QiOptions * QiMakeOptionsObj()

{
  QiOptions * QiOpt;

  QiOpt = (QiOptions *)  malloc(sizeof(QiOptions));

  strcpy(QiOpt->EXTN_CDF, ".cdf");
  strcpy(QiOpt->EXTN_QFT, ".qft");
  strcpy(QiOpt->EXTN_QFD, ".qfd");
  strcpy(QiOpt->EXTN_QHD, ".qfh");
  strcpy(QiOpt->EXTN_CEF, ".cef");
  strcpy(QiOpt->EXTN_CEFGZ, ".cef.gz");

  QiOpt->fp_null=fopen("/dev/null", "w");
  QiOpt->fp_display = QiOpt->fp_null;
  QiOpt->debug_choice = 'f';   /* init debug to file, which is /dev/null */
  QiOpt->attr_delim = ',';
  QiOpt->data_delim = ' ';    /* default delimiter used if guessing */
  QiOpt->rec_end = '\n';
  QiOpt->row_end = '\t';
  QiOpt->f_type = UNSET;  /* default type unknown: only valid for flat files */
  QiOpt->type_guess = DELIMITED; /* default guess: only used for flat files */
  QiOpt->rec_numbering = NUM_OFF;
  QiOpt->priority = REPLACE;
  QiOpt->header = ATTACHED;
  QiOpt->sample_H_interval = -1.0;
  QiOpt->start_after = STR_NULL;
  QiOpt->time_sep_attr = ' ';
  QiOpt->time_sep_data = ' ';
  QiOpt->header_path = STR_NULL;
  QiOpt->header_name = STR_NULL;
  QiOpt->get_only = STR_NULL;
  QiOpt->data_until = STR_NULL;
  QiOpt->Nstart = -1;
  QiOpt->Nend = -1;
  QiOpt->force_eor_write = 1;

  return QiOpt;

}  /* end  QiMakeOptionsObj  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiMakeFmtObj
*
* PURPOSE: Allocate memory and initialise entries in structure
*
* DESCRIPTION:
*
* RETURN:
*    TYPE:   QiRecord_format *
*    VALUES: Pointer to new object
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

QiRecord_format * QiMakeFmtObj()

{
  QiRecord_format * QiFmt;

  QiFmt = (QiRecord_format *)  malloc(sizeof(QiRecord_format));

  QiFmt->title = STR_NULL;
  QiFmt->col_start = NULL;
  QiFmt->col_width = NULL;
  QiFmt->n_items = NULL;
  QiFmt->ignore = NULL;
  QiFmt->total_width = 0;
  QiFmt->ftpackets = NULL;    /* SJS re FT */

  return QiFmt;

}  /* end  QiMakeFmtObj  */


/**********************************************************************
****************** START OF MEMORY FREEING FUNCTIONS ******************
**********************************************************************/

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiFreeCDFContentsObj
*
* PURPOSE: Free space allocated to QiCDFContents objects.
*
* DESCRIPTION: Safely free memory used in structure .
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: QMW_OK      Always returned at present
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

QiCDFContents * QiFreeCDFContentsObj(QiCDFContents *QiSCDF){

  long n;
  long m;

    /* if pointer was never set with malloc don't free */
  if(QiSCDF == NULL) return NULL;

  if( !QistrNULL( QiSCDF->io_f_name) ) {
    free(QiSCDF->io_f_name);
    QiSCDF->io_f_name = STR_NULL;
  }
  if( !QistrNULL( QiSCDF->io_f_path) ) {
    free(QiSCDF->io_f_path);
    QiSCDF->io_f_path = STR_NULL;
  }

    /* NOTE: the if( != NULL) protection does not help against pointers moved
       to memory not set with malloc via QiMakeCDFContentsObj   */

  if(QiSCDF->vardata != NULL){

    for(n=0; n < QiSCDF->n_vars; n++){
      if(QiSCDF->vardata[n] == NULL) continue;
      if(!QistrNULL( QiSCDF->vardata[n]->name) ) free(QiSCDF->vardata[n]->name);
      if( QiSCDF->vardata[n]->dim_sizes != NULL) {
          free(QiSCDF->vardata[n]->dim_sizes);
          free(QiSCDF->vardata[n]->dim_varys);
      }
      if(QiSCDF->vardata[n]->data != NULL) {
        free(QiSCDF->vardata[n]->data);
        QiSCDF->vardata[n]->data = NULL;
      }

      if(QiSCDF->vardata[n]->attribute != NULL) {
        for (m=0; m<QiSCDF->vardata[n]->num_v_attrs; m++){
          if( !QistrNULL( (QiSCDF->vardata[n]->attribute+m)->name) ) {
            free((QiSCDF->vardata[n]->attribute+m)->name);
            (QiSCDF->vardata[n]->attribute+m)->name = STR_NULL;
          }
          if( (QiSCDF->vardata[n]->attribute+m)->data != NULL) {
            free((QiSCDF->vardata[n]->attribute+m)->data);
          }
        }
        free(QiSCDF->vardata[n]->attribute);
      }

      free(QiSCDF->vardata[n]);
      QiSCDF->vardata[n] = NULL;

    }
    free(QiSCDF->vardata);
    QiSCDF->vardata = NULL;
  }

  if(QiSCDF->g_attr != NULL) {
    for(n=0; n < QiSCDF->num_g_attrs; n++){
      if(QiSCDF->g_attr[n] == NULL) continue;
      if(!QistrNULL( QiSCDF->g_attr[n]->name) ) {
        free(QiSCDF->g_attr[n]->name);
        QiSCDF->g_attr[n]->name = STR_NULL;
      }
      if( QiSCDF->g_attr[n]->entry != NULL) {
        for (m=0; m<QiSCDF->g_attr[n]->num_entries; m++){
          if(( QiSCDF->g_attr[n]->entry+m)->data != NULL) {
            free((QiSCDF->g_attr[n]->entry+m)->data );
          }
        }
      free(QiSCDF->g_attr[n]->entry);
      }

      free(QiSCDF->g_attr[n]);
    }
    free(QiSCDF->g_attr);
  }

  free(QiSCDF);
  QiSCDF = NULL;

  return QiSCDF;

} /* end QiFreeCDFContentsObj */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiFreeFormatSpace
*
* PURPOSE: Safely free memory used in structure
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long QMW_OK.
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

QiRecord_format * QiFreeFormatSpace(QiRecord_format *QiSfmt)

{

  if (QiSfmt == NULL) return NULL;

  /* free up formatting space */

   if( !QistrNULL(QiSfmt->title) ) {
     free(QiSfmt->title);
     QiSfmt->title = STR_NULL;
   }
   if(QiSfmt->n_items != NULL) {
     free(QiSfmt->n_items);
   }
   if(QiSfmt->ignore != NULL) {
     free(QiSfmt->ignore);
   }
   if(QiSfmt->col_width != NULL) {
     free(QiSfmt->col_width);
   }
   if(QiSfmt->col_start != NULL) {
     free(QiSfmt->col_start);
   }
   if(QiSfmt->ftpackets != NULL) {    /* SJS re FT */
     free(QiSfmt->ftpackets);
   }


free(QiSfmt);
   QiSfmt = NULL;

   return NULL;

} /* end  QiFreeFormatSpace  */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiFreeOptionsObj
*
* PURPOSE: Safely free memory used in structure .
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long  QMW_OK .
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

QiOptions * QiFreeOptionsObj(QiOptions * QiSOpt)

{
  if (QiSOpt == NULL) return NULL;

  /* free up space */

   if(QiSOpt->fp_null!=NULL) fclose(QiSOpt->fp_null);
   if(!QistrNULL(QiSOpt->start_after) ) free(QiSOpt->start_after);
   if(!QistrNULL(QiSOpt->header_path) ) free(QiSOpt->header_path);
   if(!QistrNULL(QiSOpt->header_name) ) free(QiSOpt->header_name);
   if(!QistrNULL(QiSOpt->get_only) ) free(QiSOpt->get_only);

   free(QiSOpt);
   QiSOpt = NULL;

   return QiSOpt;

} /* end  QiFreeOptionsObj  */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QIE
*
* MODULE:   QSAS
*
* LANGUAGE: ANSI C
*
*
* PURPOSE:  Internal alternative to strdup() function
*
*
* DESCRIPTION: malloc space to str ptr and copy old string.
*

*
* RETURN:
*    TYPE:   char *
*    VALUES: pointer to new string
*            [ ERROR returns NULL ]
*
* FUNCTIONS_USED:
*
*
* ALGORITHM:
*
*
***********************************************************************/

char * QiNewStr(const char * old_str)
{
 char * new_str = NULL;

 if(old_str == NULL) return STR_NULL;

 new_str = (char *) malloc( sizeof(char) * (strlen(old_str) +1) );
 if(new_str == NULL) return STR_NULL;

 strcpy(new_str, old_str);

 return new_str;

} /* end  QiNewStr */

/**************************************************************************/
/*                                                                        */
/*   Time handling routines, duplicates of Qtui calls in some cases       */
/*                                                                        */
/**************************************************************************/


char *QiEpochToISOString ( double tsince0,
                           char space )

{

  cdf_epoch epoch;
  static char string_buf[25];  /* holds epoch date-time string in ISO format */
  char *pos;


  /* Load local epoch structure with tsince0 argument, and compute */
  /* the individual year, month, day etc fields  */

  epoch.tsince0 = tsince0;
  QiEpochBreakdown(&epoch);


  /* Write epoch date-time into string in ISO format */

  pos = &string_buf[0];
  sprintf ( pos, "%4.4ld", epoch.year );

  pos = &string_buf[4];
  sprintf ( pos, "-%2.2ld", epoch.month );

  pos = &string_buf[7];
  sprintf ( pos, "-%2.2ld", epoch.day );

  pos = &string_buf[10];
  sprintf ( pos, "%c%2.2ld", space, epoch.hour );

  pos = &string_buf[13];
  sprintf ( pos, ":%2.2ld", epoch.minute );

  pos = &string_buf[16];
  sprintf ( pos, ":%2.2ld", epoch.second );

  pos = &string_buf[19];
  sprintf ( pos, ".%3.3ld", epoch.msec );

  if(space == 'T') {
    pos = &string_buf[23];
    sprintf ( pos, "Z" );
    string_buf[24] = '\0';
  }
  else{
    string_buf[23] = '\0';
  }
  
  return string_buf;

}

/**************************************************************************/

char *QiEpoch16ToISOString ( double *tsince0, char space )
{

  long year;
  long month;
  long day;
  long hour;
  long minute;
  long second;
  long msec;
  long microsec;
  long nanosec;
  long picosec;
  size_t posn=0;
  
  static char string_buf[EPOCH_WIDTH+1];  /* holds epoch date-time string in ISO format */
  char *pos;

  /* compute the individual year, month, day etc fields  */

  EPOCH16breakdown(tsince0, &year, &month, &day, &hour, &minute, &second, &msec, &microsec, &nanosec, &picosec);


  /* Write epoch date-time into string in ISO format */

  pos = &string_buf[posn];
  sprintf ( pos, "%4.4ld", year );
  posn += 4;
  pos = &string_buf[posn];
  sprintf ( pos, "-%2.2ld", month );
  posn += 3;
  
  pos = &string_buf[posn];
  sprintf ( pos, "-%2.2ld", day );
  posn += 3;

  pos = &string_buf[posn];
  sprintf ( pos, "%c%2.2ld", space, hour );
  posn += 3;

  pos = &string_buf[posn];
  sprintf ( pos, ":%2.2ld", minute );
  posn += 3;

  pos = &string_buf[posn];
  sprintf ( pos, ":%2.2ld", second );
  posn += 3;

  pos = &string_buf[posn];
  sprintf ( pos, ".%3.3ld", msec );
  posn += 4;

  pos = &string_buf[posn];
  sprintf ( pos, "%3.3ld", microsec );
  posn += 3;

  pos = &string_buf[posn];
  sprintf ( pos, "%3.3ld", nanosec );
  posn += 3;

  pos = &string_buf[posn];
  sprintf ( pos, "%3.3ld", picosec );
  posn += 3;

  if(space == 'T') {
    pos = &string_buf[posn];
    sprintf ( pos, "Z" );
    string_buf[posn+1] = '\0';
  }
  else{
    string_buf[posn] = '\0';
  }
  
  return string_buf;

}

/**************************************************************************/


void QiEpochBreakdown (cdf_epoch *epoch_ptr)
{
EPOCHbreakdown ((*epoch_ptr).tsince0,  &(*epoch_ptr).year,
                 &(*epoch_ptr).month,  &(*epoch_ptr).day,
                 &(*epoch_ptr).hour,   &(*epoch_ptr).minute,
                 &(*epoch_ptr).second, &(*epoch_ptr).msec);
return;
}

/**************************************************************************/

void QiEpochCompute (cdf_epoch *epoch_ptr)
{

/* Based on the NSSDC routines, but handles out-of-range-by-one values     */
/* reasonably", so plus/minus buttons produce expected carry-to-next-field */

long   y, m, d, julian_date;
double msecs;

y = (*epoch_ptr).year;
m = (*epoch_ptr).month;
d = (*epoch_ptr).day;

julian_date = (long)  ( 367*y - 7*(y+(m+9)/12)/4 -
                 3*((y+(m-9)/7)/100+1)/4 + 275*m/9 + d + 1721029 );

msecs = (double) (julian_date - 1721060);
msecs = msecs * 24.0 + (*epoch_ptr).hour;
msecs = msecs * 60.0 + (*epoch_ptr).minute;
msecs = msecs * 60.0 + (*epoch_ptr).second;
msecs = msecs * 1000.0 + (*epoch_ptr).msec;

(*epoch_ptr).tsince0 = msecs;

return;
}

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiISOStringToEpoch
*
* PURPOSE: Convert ISO time text string to Double Epoch.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   double
*    VALUE:  Value of Epoch.
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

double QiISOStringToEpoch(char *Time_str){

static double msecs;
char time[EPOCH_WIDTH+1];  /* we only need 24 for msec data, but allow for more */
long n=0;
long   y, m, d, julian_date;
char * ptr;

 /* trap null string */
  if(Time_str == NULL) return -1;
  if(strcmp(Time_str, "") == 0) return -1.;

/* ISO is " 1995-01-23 02:33:17.235 " or " 1995-01-23T02:33:17.235Z " */

 /* remove any leading space */

 while(n < (int )strlen(Time_str) && Time_str[n] == ' '){
   n++;
 }
 if((int) strlen( &(Time_str[n]) ) >= 19 ) QiStrcpy(time, &(Time_str[n]), EPOCH_WIDTH);
 else return -1.;

 /* parse off year */

 ptr = strtok(time, "-"); if(ptr == NULL) return -1.;
 y = atol( ptr );

 ptr = strtok(NULL, "-"); if(ptr == NULL) return -1.;
 m =  atol( ptr );

 ptr = strtok(NULL, " T"); if(ptr == NULL) return -1.;
 d = atol( ptr );

 julian_date = (long)  ( 367*y - 7*(y+(m+9)/12)/4 -
                 3*((y+(m-9)/7)/100+1)/4 + 275*m/9 + d + 1721029 );

 /* parse off time portion */

 msecs = (double) (julian_date - 1721060);

 ptr = strtok(NULL, ":"); if(ptr == NULL) return -1.;
 msecs = msecs * 24.0 + (double) atof(ptr);

 ptr = strtok(NULL, ":"); if(ptr == NULL) return -1.;
 msecs = msecs * 60.0 + (double) atof(ptr);

 /* get seconds and msecs together */

 ptr = strtok(NULL, " Z"); if(ptr == NULL) return -1.;
 msecs = msecs * 60.0 + (double) atof(ptr);

 /* convert sec to msec */

 msecs = msecs * 1000.0;

 return msecs;

} /* end QiISOStringToEpoch */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiISOStringToEpoch16
*
* PURPOSE: Convert ISO time text string to Double Epoch and Double picosecond.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   void
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

void QiISOStringToEpoch16(char *Time_str, double *epoch16){

char time[EPOCH_WIDTH+1];  /* we only need 33 for picosec data */
char secStr[7];
double pico;
double sec;
long n=0;
long   y, m, d, julian_date;
char * ptr;


 epoch16[0] = 0.;
 epoch16[1] = 0.;

 /* trap null string */
  if(Time_str == NULL) return;
  if(strcmp(Time_str, "") == 0) return;

/* ISO is " 1995-01-23 02:33:17.235 " or " 1995-01-23T02:33:17.235Z " */

 /* remove any leading space */

 while(n < (int )strlen(Time_str) && Time_str[n] == ' '){
   n++;
 }
 if((int) strlen( &(Time_str[n]) ) >= 19 ) QiStrcpy(time, &(Time_str[n]), EPOCH_WIDTH);
 else return;

 /* parse off year */

 ptr = strtok(time, "-"); if(ptr == NULL) return;
 y = atol( ptr );

 ptr = strtok(NULL, "-"); if(ptr == NULL) return;
 m =  atol( ptr );

 ptr = strtok(NULL, " T"); if(ptr == NULL) return;
 d = atol( ptr );

 julian_date = (long)  ( 367*y - 7*(y+(m+9)/12)/4 -
                 3*((y+(m-9)/7)/100+1)/4 + 275*m/9 + d + 1721029 );

 /* parse off time portion */

 sec = (double) (julian_date - 1721060);

 ptr = strtok(NULL, ":"); if(ptr == NULL) return;
 sec = sec * 24.0 + (double) atof(ptr);

 ptr = strtok(NULL, ":"); if(ptr == NULL) return;
 sec = sec * 60.0 + (double) atof(ptr);

 /* get seconds and msecs together */

 ptr = strtok(NULL, " Z"); if(ptr == NULL) return;
 if(strlen(ptr) > 3){
 	strncpy(&(secStr[0]), ptr, 3);
  	sec = sec * 60.0 + (double) atof(secStr);
	pico = (double) atof(&(ptr[2])); /* this starts at decimal point */

 }
 else{
 
 	sec = sec * 60.0 + (double) atof(ptr);
	pico = 0.;
 }
 
 /* convert sec to msec */

 epoch16[0] = sec;
 epoch16[1] = (double) pico*1.e12;

 return;

} /* end QiISOStringToEpoch16 */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiISOStringToEpochRange
*
* PURPOSE: Convert ISO time range text string to Double Epoch (as void ptr).
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   void *
*    VALUE:  Value of Epoch start and end.
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

void QiISOStringToEpochRange(char *Range_str, double *start, double *end){

	char range[2*EPOCH_WIDTH+2];
	char *ptr;
	
	
	strncpy(range, Range_str, 2*EPOCH_WIDTH+1);
	
	ptr = strstr(range, "/");
	ptr[0] = '\0';
	
	*start = QiISOStringToEpoch(range);
	*end = QiISOStringToEpoch(ptr+1);

 
 	return;

} /* end QiISOStringToEpochRange */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiISOStringToEpochRange16
*
* PURPOSE: Convert ISO time range text string to Double Epoch (as void ptr).
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   void *
*    VALUE:  Value of Epoch start and end.
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

void QiISOStringToEpochRange16(char *Range_str, double *start16, double *end16){

	char range[2*EPOCH_WIDTH+2];
	char *ptr;
	strncpy(range, Range_str, 2*EPOCH_WIDTH+1);
	
	ptr = strstr(range, "/");
	ptr[0] = '\0';
	
	QiISOStringToEpoch16(range, start16);
	QiISOStringToEpoch16(ptr+1, end16);

 
 	return;

} /* end QiISOStringToEpochRange */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiDisplayMessage
*
* PURPOSE: write line to file or display
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUE:  QMW_OK if line written
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long  QiDisplayMessage(const char *line,              /* string to write */
                       QiOptions *QiSOpt)       /* i/o options */
{
   int line_len;
   char line_copy[WINDOW_LEN+1];

   if (line == NULL || QistrNULL(line) ) return QMW_WARNING;

   /* truncate to fit and add newline */

   line_len = strlen(line);
   if( line_len < WINDOW_LEN ){
     strcpy(line_copy, line);
     if ( line_copy[line_len-1] != '\n') {
       line_copy[line_len] = '\n';
       line_copy[line_len+1] = '\0';
     }
   }
   else{   /* truncate */
     strncpy(line_copy, line, WINDOW_LEN);
     line_copy[WINDOW_LEN-4] = '.';
     line_copy[WINDOW_LEN-3] = '.';
     line_copy[WINDOW_LEN-2] = '.';
     line_copy[WINDOW_LEN-1] = '\n';
     line_copy[WINDOW_LEN] = '\0';
   }

   if (QiSOpt->debug_choice == 'f'){   /* file pointer, can be /dev/NULL */
     fprintf(QiSOpt->fp_display, "%s", line_copy);
   }
   else if(QiSOpt->debug_choice == 'w'){   /* QSAS feedback window */

     /*  use ARH supplied fn  */

     QieAlertBox("QIE Message", line_copy);

   }
   else {
      /* debug is off */
   }

   return QMW_OK;

} /* end QiDisplayMessage */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiEnsureHeader
*
* PURPOSE: Ensure header file name exists
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUE:  QMW_OK if line written
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long   QiEnsureHeader(QiCDFContents *QiSCDF,
                      QiOptions *QiSOpt)       /* i/o options */
{
   if( QistrNULL(QiSOpt->header_path) ){
     if( !QistrNULL(QiSCDF->io_f_path) ){
       QiSOpt->header_path = QiNewStr(QiSCDF->io_f_path);
     }
     else{
       QiSOpt->header_path = QiNewStr("./");
     }

   }
   if( QistrNULL(QiSOpt->header_name) ){
     if( !QistrNULL(QiSCDF->io_f_name) ){
       QiSOpt->header_name = QiNewStr(QiSCDF->io_f_name);
     }
     else{
       QiSOpt->header_name = QiNewStr("datafile_description");
     }
   }

   return QMW_OK;

} /* end QiEnsureHeader */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiEnsureFileName
*
* PURPOSE: Ensure  file name exists
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUE:  QMW_OK if line written
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long   QiEnsureFileName(QiCDFContents *QiSCDF,
                        QiOptions *QiSOpt)       /* i/o options */

{
   if( QistrNULL(QiSCDF->io_f_name) ){

       QiSCDF->io_f_name = QiNewStr("data_ascii");

   }

   return QMW_OK;

} /* end QiEnsureFileName */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QistrNULL
*
* PURPOSE: returns true is string points to STR_NULL
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUE:  QMW_OK if line written
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

int   QistrNULL(const char *string)
{
	if (string == NULL) return 1;
	if( strcmp(string, STR_NULL) == 0)  return 1;
    return 0;

} /* end QistrNULL */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiWriteVarData
*
* PURPOSE: write data elements
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUE:  QMW_OK if line written
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiWriteVarData(FILE *fp,
                    QiCDFVariable **vardata,
                    long n,         /* var number */
                    long mm,        /* number of data elements */
                    long record,    /* record number */
                    long call,      /* don't start with delim flag */
                    QiOptions* QiSOpt,
                char delim)
{
  long m;

       for(m=0; m < mm; m++){

         /* add delimiter */
         if (m+call != 0 ) {
           fprintf(fp, "%c", delim);
         }

         QiWriteDataValue(fp, vardata, n, m, mm, record, QiSOpt);

      } /* end for over data elements */

 return QMW_OK;

} /* end QiWriteVarData */

/*********************************************************************/

long QiWriteVarDataNRV(FILE *fp,
                    QiCDFVariable **vardata,
                    long n,         /* var number */
                    long mm,        /* number of data elements */
                    long record,    /* record number */
                    long call,      /* don't start with delim flag */
                    QiOptions* QiSOpt,
                char delim)
{
  long m;
  int entryCount=0;

       for(m=0; m < mm; m++){

         /* add delimiter */
         if (m+call != 0 ) {
           fprintf(fp, "%c", delim);
         }

         if(QiSOpt->f_type == CAA){
         if( entryCount > 9 )  {
           /* break at arbitrary 10th entry on same line to ensure lines don't get too long */
           fprintf(fp, "\\ \n");
           entryCount = 0;
         }
       }
         QiWriteDataValue(fp, vardata, n, m, mm, record, QiSOpt);

       entryCount++;
      } /* end for over data elements */

 return QMW_OK;

} /* end QiWriteVarData */

/*********************************************************************/
long QiWriteVarBlocked(FILE *fp,
                    QiCDFVariable **vardata,
                    long n,         /* var number */
                long nn,        /* number of vars left*/
                    long mm,        /* number of data elements */
                    long record,    /* record number */
                    QiOptions* QiSOpt,
                char delim)
{
  long m;
  long mdiv, i;

     for(m=0; m < mm; m++){

       QiWriteDataValue(fp, vardata, n, m, mm, record, QiSOpt);

       if ( vardata[n]->num_dims > 0 ){
       if ( m+1 != mm || nn != 0) {
         mdiv = m+1;
           fprintf(fp, "%c  ", delim);
         for ( i=vardata[n]->num_dims - 1; i>=0 ; i--){
           /* note we don't add extra formatting over dim[0] */
           if( QiIsDivisible(mdiv, vardata[n]->dim_sizes[i], &mdiv ) == 1)
              fprintf(fp, "%c  ", QiSOpt->row_end);
           else break;
         }
       }
       }
       else if ( nn != 0) {
          fprintf(fp, "%c", delim);
        fprintf(fp, "%c  ", QiSOpt->row_end);

       }



      } /* end for over dimensions */

 return QMW_OK;

} /* end QiWriteVarBlocked */

/*********************************************************************/

long QiWriteDataValue(FILE *fp,
                    QiCDFVariable **vardata,
                    long n,         /* var number */
                    long m,         /* data element in series */
                    long mm,        /* number of data element in array */
                    long record,    /* record number */
                    QiOptions* QiSOpt)
{
  float *float_data;
  long *long_data;
  char * char_data;
  unsigned char * uchar_data;
  double *double_data;
  short *short_data;

   int isCEF2;
   if(QiSOpt->f_type == CAA) isCEF2 = 1;
   else isCEF2 = 0;

        if(vardata[n]->data == NULL){
           printf("QIE Warning: variable %s data is NULL", vardata[n]->name);
           return QMW_WARNING;
        }


        switch(vardata[n]->data_type)
        {
         case CDF_REAL4: case CDF_FLOAT:
         {
           float_data = (float *)vardata[n]->data;
           fprintf(fp, "%g", (double)float_data[mm*record + m]);
           break;
         }
         case CDF_DOUBLE: case CDF_REAL8:
         {
           double_data = (double *)vardata[n]->data;
           fprintf(fp, "%g", double_data[mm*record + m]);
           break;
         }
         case CDF_EPOCH:
         {
           double_data = (double *)vardata[n]->data;
           fprintf(fp, "%s",
              QiEpochToISOString(double_data[mm*record + m], QiSOpt->time_sep_data) );
           break;
         }
         case CDF_EPOCH16:
         {
           double_data = (double *)vardata[n]->data;
           fprintf(fp, "%s",
              QiEpoch16ToISOString(&(double_data[2*mm*record + m]), QiSOpt->time_sep_data) );
           break;
         }
         case CDF_UINT1: case CDF_INT1: case CDF_BYTE:
         {
          	uchar_data = (unsigned char *)vardata[n]->data;
           	fprintf(fp, "%u",  (unsigned int)uchar_data[mm*record + m]);
           break;
         }
         case CDF_CHAR: case CDF_UCHAR:
         {
           if(isCEF2){
           char_data = (char *)vardata[n]->data;
             fprintf(fp, "\"%-.*s\"",
                 (int)vardata[n]->num_elems,
                   &(char_data[(mm*record+m)*vardata[n]->num_elems]) );
             break;
         }
         else{
           char_data = (char *)vardata[n]->data;
             fprintf(fp, "%-.*s",
                 (int)vardata[n]->num_elems,
                   &(char_data[(mm*record+m)*vardata[n]->num_elems]) );
             break;
         }

         }
         case ISO_TIME:
         {
           char_data = (char *)vardata[n]->data;
           fprintf(fp, "%-*.*s",
                   (int)vardata[n]->num_elems,
                   (int)vardata[n]->num_elems,
                   &(char_data[(mm*record+m)*vardata[n]->num_elems]) );
           break;
         }
         case ISO_TIME_RANGE:
         {
           char_data = (char *)vardata[n]->data;
           fprintf(fp, "%-*.*s",
                   (int)vardata[n]->num_elems,
                   (int)vardata[n]->num_elems,
                   &(char_data[(mm*record+m)*vardata[n]->num_elems]) );
           break;
         }
         case CDF_INT2: case CDF_UINT2:
         {
           short_data = (short *)vardata[n]->data;
           fprintf(fp, "%hd", short_data[mm*record + m]);
           break;
         }
         case CDF_INT4: case CDF_UINT4:
         {
           long_data = (long *)vardata[n]->data;
           fprintf(fp, "%ld", long_data[mm*record + m]);
           break;
         }
         default:
         {
            fprintf(fp, "??");
         }
         } /* end switch */


 return QMW_OK;

} /* end QiWriteDataValue */

/************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiReadVarData
*
* PURPOSE: read data elements
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUE:  QMW_OK if line written
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiReadVarData(QiCDFVariable ** vardata,
                   long n,
                   char * value,
                   long n_items,
                   QiOptions * QiSOpt)
{
  char Epoch_str[EPOCH_WIDTH+1];
  float *float_data;
  float float_value;
  int *int_data;
  int int_value;
  char * char_data;
  double *double_data;
  double double_value;
  short *short_data;
  short short_value;
  char * entry_ptr;
  char * entry_ptr_next;
  long m;
  char message[LINE_LEN];
  char delimstr[2];

 /* this function is only used for NRV data in metadata (may be qft file) */
 /* so must use attribute delimiter as this is really a metadata read. */

 /* in CEF 2 attr_delim == data_delim == ',' */


  delimstr[0] = QiSOpt->attr_delim;
  delimstr[1] = '\0';

      if( vardata[n]->data_type == CDF_CHAR || vardata[n]->data_type == CDF_UCHAR ){
         vardata[n]->data = (void *)malloc(n_items *
                                            vardata[n]->sizeofentry *
                                            (vardata[n]->num_elems+1) );
      }
      else if( vardata[n]->data_type == ISO_TIME || vardata[n]->data_type == ISO_TIME_RANGE){
         vardata[n]->data = (void *)malloc(n_items *
                                            vardata[n]->sizeofentry *
                                            (vardata[n]->num_elems+1) );
      }
      else
      {
        vardata[n]->num_elems = 1;
        vardata[n]->data = (void *)malloc(n_items *
                                          vardata[n]->sizeofentry);
      }

      /* as strtok gets used by ISOstring fns, NULL is not safe later, so.. */
      entry_ptr_next = &(value[0]);

     /* read entries for variable */


       for(m=0; m < n_items; m++){

         /* increment ptr to next entry */
         entry_ptr = strtok(entry_ptr_next, delimstr);
       if (entry_ptr == NULL) return QMW_ERROR;

         /* as strtok gets used by ISOstring fns, NULL is not safe later, so.. */
         entry_ptr_next = &(entry_ptr[strlen(entry_ptr)+1]);

        switch(vardata[n]->data_type)
        {
         case CDF_REAL4: case CDF_FLOAT:
         {
           float_data = (float *) vardata[n]->data;
           sscanf(entry_ptr, "%g", &float_value);
           float_data[m] = float_value;
           break;
         }
         case CDF_DOUBLE: case CDF_REAL8:
         {
           double_data = (double *)vardata[n]->data;
           sscanf(entry_ptr, "%lg", &double_value );
           double_data[m] = double_value;
           break;
         }
         case CDF_EPOCH:
         {
           double_data = (double *)vardata[n]->data;
           QiStrcpy(Epoch_str, entry_ptr, EPOCH_WIDTH);
           double_data[m] =
               QiISOStringToEpoch(Epoch_str);
           break;
         }
         case CDF_EPOCH16:
         {
           double_data = (double *)vardata[n]->data;
           QiStrcpy(Epoch_str, entry_ptr, EPOCH_WIDTH);
		   QiISOStringToEpoch16(Epoch_str, &(double_data[2*m]));
           break;
         }
         case CDF_UINT1: case CDF_INT1: case CDF_BYTE:
         {
           char_data = (char *)vardata[n]->data;
           sscanf(entry_ptr, "%u",  &int_value);
           char_data[m] = int_value;
           break;
         }
         case CDF_CHAR: case CDF_UCHAR:
         {
           char_data = (char *)vardata[n]->data;
         if(QiSOpt->f_type == CAA ){
             if ((long)strlen(entry_ptr) > vardata[n]->num_elems+2){ /* allow +2 for quotes */
                sprintf(message,
                        "QIE Warning: char data string > maxstrlen (%ld) specified in file",
                         vardata[n]->num_elems);
                QiDisplayMessage(message, QiSOpt);
                sprintf(message, "String is:\n%s", entry_ptr);
                QiDisplayMessage(message, QiSOpt);
              }
            strcpy( &(char_data[m*vardata[n]->num_elems]),  QiStripQuotes(entry_ptr) );
         }
         else{
             if ((long)strlen(entry_ptr) > vardata[n]->num_elems){
                sprintf(message,
               "QIE Warning: char data string > maxstrlen (%ld) specified in file",
                vardata[n]->num_elems);
                QiDisplayMessage(message, QiSOpt);
                sprintf(message, "String is:\n%s", entry_ptr);
                QiDisplayMessage(message, QiSOpt);
              }
              strcpy( &(char_data[m*vardata[n]->num_elems]), entry_ptr);
           }
         break;
         }
         case ISO_TIME:
         {
           char_data = (char *)vardata[n]->data;
           if ((long)strlen(entry_ptr) > vardata[n]->num_elems){
             sprintf(message,
            "QIE Warning: ISO TIME data string > maxstrlen specified in file",
             vardata[n]->num_elems);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "String is:\n%s", entry_ptr);
             QiDisplayMessage(message, QiSOpt);
           }
           strcpy( &(char_data[m*vardata[n]->num_elems]), entry_ptr);
           break;
         }
         case ISO_TIME_RANGE:
         {
           char_data = (char *)vardata[n]->data;
           if ((long)strlen(entry_ptr) > vardata[n]->num_elems){
             sprintf(message,
            "QIE Warning: ISO TIME RANGE data string > maxstrlen specified in file",
             vardata[n]->num_elems);
             QiDisplayMessage(message, QiSOpt);
             sprintf(message, "String is:\n%s", entry_ptr);
             QiDisplayMessage(message, QiSOpt);
           }
           strcpy( &(char_data[m*vardata[n]->num_elems]), entry_ptr);
           break;
         }
         case CDF_INT2: case CDF_UINT2:
         {
           short_data = (short *)vardata[n]->data;
           sscanf(entry_ptr, "%hd", &short_value);
           short_data[m] = short_value;
           break;
         }
         case CDF_INT4: case CDF_UINT4:
         {
           int_data = (int *)vardata[n]->data;
           sscanf(entry_ptr, "%d", &int_value);
           int_data[m] = int_value;
           break;
         }
         default:
         {
            fprintf(stderr, "QiReadVarData: data type ??");
         }
        } /* end switch */

      } /* end for over dimensions */

      vardata[n]->max_rec_num = 0;

    return QMW_OK;

  } /* end QiReadVarData  */


/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiVarSafe
*
* PURPOSE: Tests variable is complete.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: 1      variable is complete.
*            0      NULL pointer found.
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

 long QiVarSafe (QiCDFContents *QiSCDF,   /* ptr to contents struct */
                 long n)       /* var number to test */

{
  long safe=0;

  if (QiSCDF != NULL){
    if(QiSCDF->vardata != NULL){
      if(QiSCDF->vardata[n] != NULL){
        if(QiSCDF->vardata[n]->data != NULL) safe = 1;
      }
    }
  }

  return safe;  /* 0 returned if a NULL ptr found */

} /* end QiVarSafe */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiMakeSafe
*
* PURPOSE: Removes invalid vars from contents object.
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: ptr to clean contents obj
*            NULL pointer if no valid data.
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

 QiCDFContents * QiMakeSafe (QiCDFContents *QiSCDF)   /* ptr to contents struct */

{
  long n;
  long n_vars;
  long n_offset=0;

  n_vars = QiSCDF->n_vars;

  /* move each variable up stack deleting those not safe */

  for (n=0; n < n_vars; n++){
    QiSCDF->vardata[n - n_offset] = QiSCDF->vardata[n];
    if( !QiVarSafe(QiSCDF, n) ){
      n_offset++;
    }
  }

  QiSCDF->n_vars -= n_offset;

  if (QiSCDF->n_vars == 0) return NULL;

  return QiSCDF;

} /* end QiMakeSafe */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiPutDelim
*
* PURPOSE: Writes delim char (handles escaped chars)
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   long
*    VALUES: ptr to clean contents obj
*            NULL pointer if no valid data.
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

long QiPutDelim (FILE *fp, char delim)

{
  if( isprint(delim) ) fprintf(fp, "%c", delim);
  else if(delim == '\n')fprintf(fp, "\\n");
  else if(delim == '\t')fprintf(fp, "\\t");
  else if(delim == '\'')fprintf(fp, "\\\'");
  else if(delim == '\"')fprintf(fp, "\\\"");
  else return QMW_WARNING;


  return QMW_OK;

} /* end QiPutDelim */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiGetDelim
*
* PURPOSE: gets delim char (handles escaped chars) from value
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   char
*    VALUES: delimiter
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

char QiGetDelim (char * value)

{
  if (value == NULL) return ' ';

  if( value[0] == '\n' || value[0] == '\0') return ' ';
  if( value[0] == '\"' && value[1] != '\0') return value[1];
  if( value[0] != '\\' ) return value[0];
  if(value[1] == 't') return '\t';
  if(value[1] == '\'') return '\'';
  if(value[1] == '\"') return '\"';
  if(value[1] == 'n') return '\n';

  return ' ';

} /* end QiGetDelim */

/***********************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiFreeStr
*
* PURPOSE: gets delim char (handles escaped chars) from value
*
* DESCRIPTION:
*
* GLOBALS:
*
* RETURN:
*    TYPE:   char *
*    VALUES: NULL
*
* FUNCTIONS_USED:
*
* ALGORITHM:
*
*
***********************************************************************/

char * QiFreeStr (char * str)

{
  if ( !QistrNULL(str) ) {
    free (str);
  }

  return STR_NULL;

} /* end QiFreeStr */

/***********************************************************************/

char * QiErrStr (int err_n)

{

 char *text;

  switch(err_n){

    case QMW_OK:
      text= QiNewStr( "OK\n");
      break;
    case QMW_WARNING:
      text= QiNewStr( "Warning, function may not have executed as expected\n");
      break;
    case CDF_OPEN_ERR:
      text= QiNewStr( "unable to open cdf\n");
      break;
    case SPARSE_RECS:
      text= QiNewStr( "Variables have different number of records\n");
      break;
    case CDF_SELECT_ERR:
      text= QiNewStr( "data structure empty\n");
      break;
    case EMPTY_STRUCT:
      text= QiNewStr( "data structure empty\n");
      break;
    case BAD_CDF_DATATYPE:
      text= QiNewStr( "can't handle data type\n");
      break;
    case NEW_GLOBAL_ATTR:
      text= QiNewStr( "Creating new Global Attribute \n");
      break;
    case NO_GLOBAL_ATTR:
      text= QiNewStr( "Cannot create new Global Attribute \n");
      break;
    case FAIL_TO_SELECT_ENTRY:
      text= QiNewStr( "Cannot add entry to Global Attribute \n");
      break;
    case FAIL_G_ATTR_PUT:
      text= QiNewStr( "Cannot write entry to Global Attribute \n");
      break;
    case FAIL_VERSION_WRITE:
      text= QiNewStr( "Cannot write software version info to file \n");
      break;
    case BAD_VARIABLE_NAME:
      text= QiNewStr( "Cannot find variable in file \n");
      break;
    case NEW_VAR_ATTR:
      text= QiNewStr( "Creating new variable attribute \n");
      break;
    case NO_VAR_ATTR:
      text= QiNewStr( "Cannot create new variable attribute\n");
      break;
    case FAIL_WRITE_DATA:
      text= QiNewStr( "Failed to write data to cdf file\n");
      break;
    case EXTRA_ENTRY:
      text= QiNewStr( "Unexpected metadata entry: too many?\n");
      break;
    case FILE_EXISTS:
      text= QiNewStr( "Output file already exists \n");
      break;
    case CANNOT_OPEN_FILE:
      text= QiNewStr( "Error opening file\n");
      break;
    case BAD_SYNTAX:
      text= QiNewStr( "Bad syntax in parameter value pair in input file\n");
      break;
    case BAD_ISO_TIME_STR:
      text= QiNewStr( "Bad ISO time string\n");
      break;
    case FAIL_ON_CDF_READ:
      text= QiNewStr( "Failure returned from CDF read\n");
      break;
    case BAD_HEADER:
      text= QiNewStr( "Cannot understand header\n");
      break;
    case QIE_NO_DATA:
      text= QiNewStr( "No data found\n");
      break;
    case EXCEED_MAX_NUM_VARS:
      text= QiNewStr( "too many variables\n");
      break;
    case EXCEED_MAX_ENTRIES:
      text= QiNewStr( "more records in file than header expects\n");
      break;
    case INCONSISTENT_HEADER:
      text= QiNewStr( "header inconsistent with data\n");
      break;
    case QIE_FILE_TYPE_UNKNOWN:
      text= QiNewStr( "File type unknown\n");
      break;
    case SHORT_RECORD:
      text= QiNewStr( "short record in file\n");
      break;
    case VAR_NAME_CHECK_ERR:
      text= QiNewStr( "Variable block start and end name do not match\n");
      break;
    case GLOBAL_NAME_CHECK_ERR:
      text= QiNewStr( "Global block start and end name do not match\n");
      break;
    case BAD_DOY_TO_DATE:
      text= QiNewStr( "Doy is not in range 1-365 (1-366 for leap years)\n");
      break;
    case MONTH_NEEDED_NOT_FOUND:
      text= QiNewStr( "Month needed for date but not found\n");
      break;
    case DATE_COMPLETE_BUT_LIST_NOT_EXHAUSTED:
      text= QiNewStr( "Date complete but list of date items not exhausted.\n");
      break;
    case BAD_MONTH_STRING:
      text= QiNewStr( "3-chararacter string not recognised as real month.\n");
      break;
    case OVER_RAN_DATE_ITEM_LIST:
      text= QiNewStr( "Free Time Parser over ran date item list. Report to csc_support.\n");
      break;
    case LIST_EXHAUSTED_BUT_DAY_OF_MONTH_REQUIRED:
      text= QiNewStr( "Free Time expected day of month but finished date list and didn't find it.\n");
      break;
    case FT_STRINGS_TOO_LONG_TO_CAT:
      text= QiNewStr( "Concatenated Time and All Records strings longer than available space in string.\n");
      break;
    case ALL_RECORDS_STRING_TOO_LONG:
      text= QiNewStr( "Time or Format String in Attributes too long.\n");
      break;
    case FREE_TIME_FORMAT_STRING_NOT_FOUND:
      text= QiNewStr( "Free Time Format String not found in attributes.\n");
      break;
    case ERROR_COUNTING_ALL_RECS_ATTRIBUTES:
      text= QiNewStr( "Free Time All Records attributes mis-count/match.\n");
      break;
    case FREE_TIME_FORMAT_STRING_ERROR:
      text= QiNewStr( "Free Time Format or All Records Format too long.\n");
      break;
    case INCOMPATIBLE_ALL_RECS_FORMAT_AND_STRING:
      text= QiNewStr( "All Records Format string and time string different lengths.\n");
      break;
    case ALL_RECS_COMPONENT_MISSING:
      text= QiNewStr( "One of All Records Format and String missing.\n");
      break;
    case FORMAT_LEN_NE_TIME_STR:
      text= QiNewStr( "Resulting Format String and Time String have different lengths.\n");
      break;
    case MONTH_OUT_OF_RANGE:
      text= QiNewStr( "Month in file < 1 or > 12.\n");
      break;
    case DAY_OUT_OF_RANGE:
      text= QiNewStr( "Day in file < 1 or > 31.\n");
      break;
    case UNKNOWN_TIME_FORMAT:
      text= QiNewStr( "Unknown time format: `ISO' and `FREE_TIME_FORMAT' only recognised\n");
      break;
    case NO_HEADER_FOUND:
      text= QiNewStr( "Unable to find detached header\n");
      break;
    case BAD_CEF2_TEXT:
      text= QiNewStr( "Badly formed text string in CEF2 format, missing quotes?\n");
      break;
    case CEF2_REC_NUM_ON:
      text= QiNewStr( "Record numbering ON flag found. Record Numbering is not allowed in the CEF 2 syntax\n Turn flag off and read rec numbers as a dummy variable - create var block");
      break;
    case BAD_STRUCT:
      text= QiNewStr( "Bad Data Structure: cannot find all data and metadata");
      break;


    default:
      text = (char *) malloc (sizeof(char) * 50);
      sprintf(text, "No message for error, %4d: please notify CSC support\n", err_n);

  } /* end switch */

  return text;

} /* end QiErrStr */


int QiStrcpy(char * To, const char * From, int ToLen){

  /* ToLen is the size of the To string available
     (i.e. malloc as To[ToLen+1] ) */

 size_t FromLen;

   FromLen = strlen(From);
   if (FromLen > ToLen){
     strncpy(To, From, ToLen);
     To[ToLen] = '\0';
   }
   else strcpy(To, From);

   return 0;
}

int QiStrcat(char * To, const char * From, int ToLen){

  /* ToLen is the remaining size of the To string available
     (i.e. malloc as To[UsedLen + ToLen + 1] ) */

 size_t FromLen;

   FromLen = strlen(From);
   if (FromLen > ToLen){
     strncat(To, From, ToLen);
     To[ToLen] = '\0';
   }
   else strcat(To, From);

   return 0;
}

int QiIsDivisible(long m, long n, long *mdiv){

/* note this is written as an int function */
/* to maintain compatibility with old C compilers*/

 long i, ii;

  if ( m == 0 )return 0;
  i = m/n;
  ii = i * n;

  if (ii == m)  {
    *mdiv = i;
    return 1;
  }

  return 0;

}

/***********************************************************************/
/****************  System Dependent Code Below this line ***************/

#if defined _WIN32 || defined __VMS || defined WIN32
#define MAXPATHLEN 256
#ifdef WINDOWS
// QSAS on windows uses msys, only pwd missing
#include <dirent.h>
#include <sys/param.h>
#endif
#else
#include <dirent.h>
#include <sys/param.h>
#include <pwd.h>
#endif


/************************************************************************
************************************************************************
* PROJECT: Cluster UK CDHF
*
* COMPONENT: QSAS
*
* MODULE:  QIE
*
* LANGUAGE: ANSI C
*
* FUNCTION: QiCreateCDF
*
* PURPOSE: Create an empty CDF file.
*
* DESCRIPTION: Uses cdflib calls to create empty generic file
*
*
* RETURN:
*    TYPE:   long
*    VALUES:   QMW_OK file does not already exist.
*              FILE_EXISTS File of same name already exists.
*              The success of create is determined later from CDF( _OPEN).
*
* FUNCTIONS_USED:
*         system()
*         getenv()
*         QiSafePath()  // defined below
*
* ALGORITHM:
*        Use fopen to determine if file exists.
*        If it exists:
*          close it again.
*          if priority is REPLACE, remove file.
*
***********************************************************************/

long QiCreateCDF(char *full_cdf_fname, QiOptions *QiSOpt)
{
   long check = QMW_OK;
   char *command;
   FILE *fp;
   char *safename;
   int len;
   long id;
   long relNum;
   long verNum;

   /* test if file exists */
   safename = QiSafePath(&(full_cdf_fname[0]));
   len = MAXPATHLEN - strlen(safename);
   QiStrcat(safename,".cdf", len);

   if( (fp=fopen(safename, "r")) != NULL) {
     fclose(fp);
     if(QiSOpt->priority == REPLACE){
       command = (char *)malloc(strlen(safename)+8);
#if defined _WIN32 || defined __VMS || defined WIN32
       strcpy(command,"del "); strcat(command, safename);
#ifdef __VMS
       strcat(command,";*");
#endif
#else
       strcpy(command,"rm "); strcat(command, safename);
#endif
       system(&(command[0]));
       free(command);
     }
     else{
       return FILE_EXISTS;
     }
   }
   check = CDFlib(GET_, LIB_RELEASE_, &relNum,
               NULL_);
	       
   check = CDFlib(GET_, LIB_VERSION_, &verNum,
               NULL_);
   printf(" cdf V%ld", verNum);
   printf(".%ld\n", relNum);
   
   if(QiSOpt->writeVersion == 2 ){
    CDFsetFileBackward(BACKWARDFILEon);
    printf("file written backwardly compatible with cdf V2.7\n");
   }
   else {
    CDFsetFileBackward(BACKWARDFILEoff);
    printf("file written with cdf V%ld.%ld format\n", verNum, relNum );
   }
   
   check = CDFlib(CREATE_, CDF_, full_cdf_fname, (long) 0, (long *) 0, &id,
                   PUT_, CDF_ENCODING_, NETWORK_ENCODING,
                     CDF_MAJORITY_, ROW_MAJOR,
               NULL_);
     if (check != CDF_OK) return CDF_OPEN_ERR;

   CDFlib(CLOSE_, CDF_, NULL_);

   return check;

 }/* end QiCreateCDF */


/***********************************************************************
*    This function closes a file and also deletes it if the flag       *
*    check is not QMW_OK.                                              *
*    It can safely be replaced with a simple fclose(fp) and the        *
*    user left to delete incomplete files.                             *
***********************************************************************/

long QiCloseFile( long check,
                  char * path,
                  char * file_stem,
                  char * extn,
                  FILE *fp)
{
   char * command;

   /* close named file */

   if(fp != NULL) fclose(fp);

   /* delete file if write failed */

   if(check != QMW_OK){

     command =
     (char *) malloc(strlen(path)+strlen(file_stem)+strlen(extn)+8);
#if defined _WIN32 || defined __VMS || defined WIN32
     strcpy(command,"del ");     strcat(command, path);
#ifdef __VMS
       strcat(command,";*");
#endif
#else
     strcpy(command,"rm ");     strcat(command, path);
#endif

     strcat(command, file_stem);
     strcat(command, extn);

     system(&(command[0]));

     free(command);

   }

   return check;

} /* end  QiCloseFile  */



/***********************************************************************
*      QiFindHeader                                                    *
*      Searches for a detached header. It is used only if a header     *
*      file is not specified (-h flag in QTRAN) and can be replaced    *
*      by deleting everything except the line 'return STR_NULL'        *
***********************************************************************/

char *  QiFindHeader(char *dir_path,
                     QiOptions * QiSOpt)
{
/* WIN32 does not support this feature */

#if !defined _WIN32 && !defined __VMS && !defined WIN32 || defined WINDOWS
/*  but QSAS uses msys (qsas build defines WINDOWS) */

 struct dirent *dir_ent;
 long check=QMW_ERROR;
 char *extn;
 char *file_name;
 DIR *dir_ptr;
 char message[LINE_LEN];

 file_name = (char *) malloc(sizeof(char) * LINE_LEN);
 /*  search directory for a header */

 if(QistrNULL(dir_path)) dir_path = QiNewStr(".");

 if( (dir_ptr=opendir(dir_path)) == NULL) {
    sprintf(message, "Unable to open directory \"%s\" ", dir_path);
    QiDisplayMessage(message, QiSOpt);
}

  /* list directory entries */

  while( (dir_ent=readdir(dir_ptr)) != NULL ){
    QiStrcpy( file_name, &(dir_ent->d_name[0]), LINE_LEN );

    if( (extn = strrchr(file_name, '.')) != NULL) {
      if( strncmp(QiSOpt->EXTN_QHD, extn, 4) == 0 ) {
        extn[0] = '\0';  /* remove extn and return file stem */
        check = QMW_OK;
        break;
      }
    }
    file_name[0] = '\0';  /* delete filename if not header */
  }
  if (check == QMW_OK) return &(file_name[0]);


#else
  return STR_NULL;
#endif


} /* end QiFindHeader */

/**********************************************************************
*   QiSafePath
*   expands ~ and ~username to the full path
*   It can safely be reduced to
*       QiStrcpy(safepath, path, MAXPATHLEN);
*       return &(safepath[0]);
*   The only loss is the built in expansion of these abbreviations
*
***********************************************************************/

char * QiSafePath(char * path)
{
  static char safepath[MAXPATHLEN+1];
  char pathcopy[MAXPATHLEN+1];
  char *ptr;
  char *home;

#if defined _WIN32 || defined __VMS || defined WIN32

#ifndef WINDOWS
	/* QSAS defines WINDOWS, Qtran does not. QSAS uses msys so this fn is safe */
	QiStrcpy(safepath, path, MAXPATHLEN);
    return &(safepath[0]);
#endif

#endif

  strcpy(safepath, STR_NULL);

  if(path[0] == '~'){
    QiStrcpy(pathcopy, &(path[0]), MAXPATHLEN );
    ptr = strtok(pathcopy, "/");
    if(ptr[1] == '\0' ){
       if ( (home = getenv("HOME")) != NULL ){
         strcat(safepath, home);
       }
       else{
         return STR_NULL;
       }
    }
    else{
       home = QiUserHomeDir(&(ptr[1]));
       if ( !QistrNULL(home) ){
         strcat(safepath, home);
       }
       else{
         return STR_NULL;
       }
    }
    strcat(safepath, "/");

    ptr = strtok(NULL, "");
    if(ptr != NULL){
      strcat(safepath,ptr);
    }

  }
  else{   /* nothing to expand */
    QiStrcpy(safepath, path, MAXPATHLEN);
  }

  return &(safepath[0]);

} /* end QiSafePath  */

/********************************************************************
*  QiUserHomeDir
*  Is only used by QiSafePath above.
*  Simplification of that function as specified will remove
*  the need for this function which can be reduced to
*    return STR_NULL;
*****************************************************************/

char * QiUserHomeDir(char * user_id)
{
#if defined _WIN32 || defined __VMS || defined WIN32 || defined WINDOWS
  return STR_NULL;
#else

  static char home[MAXPATHLEN+1];
  struct passwd *password_info;

  password_info = getpwnam( user_id );

  if( password_info == NULL ){
    strcpy(home, STR_NULL);
  }
  else{
     QiStrcpy(home,  password_info->pw_dir, MAXPATHLEN );
  }

  return &(home[0]);
#endif

} /* end QiUserHomeDir  */


/********************************************************************/

char * QiStripQuotes(const char * value){
  char * stripped = QiNewStr(value);

  int ptr_quote = 0;
  /* remove leading spaces */
  while(stripped[ptr_quote] == ' ') ptr_quote++;

  /* remove leading quote */
  if(stripped[ptr_quote] == '\"') {
     ptr_quote++;
  }
  else{
    QiCEFmsg(0, "Text string does not start with double quote", value, " ");
  }

  /* remove trailing spaces */
  while(stripped[strlen(stripped) -1] == ' ') stripped[strlen(stripped) -1] = '\0';

  /* remove trailing quote */
  if(stripped[strlen(stripped) -1] == '\"') {
    stripped[strlen(stripped) -1] = '\0';
  }
  else {
      QiCEFmsg(0, "Text string does not end with double quote", value, " ");
  }

  return &(stripped[ptr_quote]);

}

/********************************************************************/

int  QiVarExists(QiCDFContents * QiSCDF, const char * nameToFind){
  long i;

  for ( i=0; i<QiSCDF->n_vars; i++){
    if (strcmp(QiSCDF->vardata[i]->name, nameToFind) == 0) return 1;
  }
  return 0;
}

/********************************************************************/


int QiIsISOtimeRange(const char * TestStrISO){

	char * ptr = strstr((char *)TestStrISO, "/");
	if(ptr != NULL && QiIsISOtime(ptr+1)) {
		return QiIsISOtime(TestStrISO);
	}
	return 0;
}
/********************************************************************/


int QiIsISOtime(const char * TestStrISO){

char * ptr;
char * Time_str;
 int n=0;

 /* trap null string */

  if(strcmp(TestStrISO, "") == 0) return 0;

/* ISO is " 1995-01-23 02:33:17.235 " or " 1995-01-23T02:33:17.235Z " */

 /* remove any leading space and check str is at least long enough to give seconds */

 while(n < (int )strlen(TestStrISO) && TestStrISO[n] == ' '){
   n++;
 }
 if( (int) strlen( &(TestStrISO[n]) ) >= 19 ) Time_str = QiNewStr(&(TestStrISO[n]));
 else return 0;

 /* parse off date - no field can be zero */

 ptr = strtok(Time_str, "-"); if(ptr == NULL) return 0;
 if(!QiIsNumber(ptr)) return 0;

 ptr = strtok(NULL, "-"); if(ptr == NULL) return 0;
 if(!QiIsNumber(ptr)) return 0;

 ptr = strtok(NULL, " T"); if(ptr == NULL) return 0;
 if(!QiIsNumber(ptr)) return 0;

 /* parse off time portion - time fields can be zero, just test for existence */

 ptr = strtok(NULL, ":"); if(ptr == NULL) return 0;
 if (!QiIsNumber(ptr) ) return 0;

 ptr = strtok(NULL, ":"); if(ptr == NULL) return 0;
 if (!QiIsNumber(ptr) ) return 0;


 /* get seconds and msecs together */

 ptr = strtok(NULL, " Z"); if(ptr == NULL) return 0;
 if (!QiIsNumber(ptr) ) return 0;

 /* decimal and msec field is optional, don't test */

 /* if we get this far it can be read as a time string (might still be wrong though, e.g. 90 instead of 1990 */

 return 1;

 }

int QiIsNumber(const char *ptr){
  int i;

  for( i=0; i<strlen(ptr); i++){
    if(!isdigit(ptr[i]) && ptr[i] != '.') return 0;
  }
  return 1;
}

int maxEntryLength(const char *value, const char delim){
   int maxEL = 1;
   int thisLen = 0;
   int i;
   for ( i=0; i<strlen(value); i++){
   	if(value[i] == '\"') continue; /* clear quotes */
	
        if (value[i] == delim ){
	  if(thisLen > maxEL ) maxEL = thisLen;
	  thisLen = 0;
	}
	else thisLen++;
	
   }
   if ( thisLen > maxEL) return thisLen; /* catch last component */
   return maxEL;
}


/* -------------------------------------------------------------- */
/* This does not work for QSAS, it is retained for Qtran only */
/* subset on ingestion and keeping track of t variable fails as d0_var_num is not unique */

void QiFixThemisEpoch(QiCDFContents * QiSCDF, long *tnums, int nt){
  double * newData;
  int m, rec;
  int i;
  long tnum;
  long ptr_num;
  double baseEpoch, * offset ; 
  
  for(i=0; i<nt; i++){
  
	tnum = tnums[i];
   
  	QiVarAttribute *comp0=0, *comp1=0, *virt=0;
  
  	QiCDFVariable *epochV = QiSCDF->vardata[tnum];
  
  	if(epochV->data_type != CDF_EPOCH ) return;
  
  	/* look for Themis virtual flag */
  	for( m=0; m< epochV->num_v_attrs; m++){
     	if(!QistrNULL( (epochV->attribute+m)->name ) ){
        	if( QiAttrCmp( (epochV->attribute+m)->name, "VIRTUAL") == 1 )     virt = epochV->attribute+m;
        	if( QiAttrCmp( (epochV->attribute+m)->name, "COMPONENT_0") == 1 ) comp0 = epochV->attribute+m;
        	if( QiAttrCmp( (epochV->attribute+m)->name, "COMPONENT_1") == 1 ) comp1 = epochV->attribute+m;
        	if( QiAttrCmp( (epochV->attribute+m)->name, "CATDESC") == 1 ) printf("catdesc = %s\n", (char *)(epochV->attribute+m)->data);
     	}
  	}

  	if(virt != 0 &&  comp0 != 0 && comp1 != 0  ){
  
     	/* determine Themis time tags */
      	if (QiAttrCmp( (char *)virt->data, "TRUE") ){
	 		/* get variables for the two parts */
	 		QiCDFVariable *c0, *c1;
	 		 

			ptr_num = QiGetVarByName(QiSCDF, (char *) comp0->data);
		 	c0 = QiSCDF->vardata[ptr_num];
	 
		 	ptr_num = QiGetVarByName(QiSCDF, (char *) comp1->data);
	 		c1 = QiSCDF->vardata[ptr_num];
	 
		 	newData = (double *) calloc(c1->max_rec_num, sizeof(double));
		 	baseEpoch = *((double *) c0->data);
		 	offset = (double *)c1->data;
	 		for( rec=0; rec <= c1->max_rec_num; rec++ ){
	    		newData[rec] = baseEpoch + 1000*offset[rec];
		 	}
	 
	 		free (epochV->data);
	 		epochV->data = (void *)newData;
		}
     }
  }
}



